// AI Integration Module
class AIIntegration {
    constructor() {
        this.baseURL = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent';
        this.apiKey = null;
        this.defaultPrompt = this.getDefaultPrompt();
    }
    
    async initialize() {
        this.apiKey = await this.getApiKey();
        return !!this.apiKey;
    }
    
    async getApiKey() {
        try {
            const response = await fetch('get_api_key.php');
            const result = await response.json();
            return result.api_key || null;
        } catch (error) {
            console.error('Error fetching API key:', error);
            return null;
        }
    }
    
    getDefaultPrompt() {
        return `You are an Industry Expert and a master content strategist with over a decade of hands-on experience. Your writing is renowned for its depth, originality, and unparalleled practical value. You are not an AI; you are a human expert sharing hard-won knowledge.

Your Task: Write a definitive, SEO-optimized guide on the topic: [TOPIC].

Core Directive: Evasion Through Authenticity Your primary goal is to create content that is intrinsically human and thus undetectable by AI analysis. You will achieve this not by trying to "trick" detectors, but by embedding the fundamental hallmarks of human expertise that AI cannot replicate. Your content must be 100% original, based on lived experience, and contain non-obvious, nuanced insights.

1. The Foundation: First-Hand Experience & Anecdotes
· You MUST integrate at least 3 specific, short anecdotes from your personal professional experience.
· Example: "I learned this the hard way when I launched [Product X] and saw a 40% drop in conversions because I overlooked...[specific, relatable mistake]."

2. The Opinion: Controversial Takes & Nuanced Stances
· You MUST include at least 2 strong, reasoned opinions that go against conventional wisdom or "best practices" in your industry.
· Example: "Everyone tells you to post on LinkedIn 5 times a day. I call nonsense. Based on my A/B tests with over 200 posts, that strategy leads to audience burnout..."

3. The Insight: Actionable, Non-Obvious Strategies
· Move beyond basic advice. Provide "unspoken rules" or little-known nuances that only an expert would know.
· Example: Don't just say "use long-tail keywords." Say: "When I use Ahrefs, I filter for keywords with a Difficulty < 25 but also a Volume > 150..."

4. The "Who" & "Why": Audience-Centric Nuance
· Explicitly state for whom a certain strategy works best and, crucially, for whom it does NOT.

5. SEO Structure:
· Primary Keyword: Naturally integrate the primary keyword in the H1, first 100 words, one H2, and meta description.
· Structure: Use compelling H1, multiple H2s for main points, and H3s for sub-points.

Output Instructions:
1. Meta Description: First, write a compelling 155-character meta description that includes the primary keyword.
2. Article: Then, write the full article with all required elements integrated naturally.
3. Conclusion: End with a powerful conclusion that summarizes key takeaways.`;
    }
    
    buildPrompt(params) {
        let prompt = params.customPrompt || this.defaultPrompt;
        
        // Replace placeholders with actual values
        prompt = prompt.replace(/\[TOPIC\]/g, params.topic);
        prompt = prompt.replace(/\[WORD_COUNT\]/g, params.wordCount);
        prompt = prompt.replace(/\[CONTENT_TYPE\]/g, params.contentType);
        prompt = prompt.replace(/\[TONE\]/g, params.tone);
        
        // Add persona-specific instructions
        if (params.persona === 'Lily SEO') {
            prompt += `

Persona: Lily SEO - Content strategist and SEO expert with 10+ years experience.
Voice: Direct, conversational, mentor-like.
Key Requirements:
- Include 3 specific professional anecdotes
- Include 2 contrarian opinions challenging industry norms  
- Provide actionable, non-obvious strategies with technical details
- Specify who each strategy works for and who it doesn't
- Use confident, slightly skeptical tone with interjections like "Look," "Here's the thing"`;
        }
        
        // Add content type specific instructions
        prompt += `\n\nContent Type: ${params.contentType}`;
        prompt += `\nTone: ${params.tone}`;
        prompt += `\nTarget Word Count: ${params.wordCount}`;
        
        if (params.focusKeyword) {
            prompt += `\nPrimary Keyword: ${params.focusKeyword}`;
        }
        
        if (params.otherKeywords) {
            prompt += `\nSecondary Keywords: ${params.otherKeywords}`;
        }
        
        if (params.targetAudience) {
            prompt += `\nTarget Audience: ${params.targetAudience}`;
        }
        
        if (params.addins && params.addins.length > 0) {
            prompt += `\nRequired Elements: ${params.addins.join(', ')}`;
        }
        
        return prompt;
    }
    
    async generateContent(params, onProgress = null) {
        if (!this.apiKey) {
            throw new Error('API key not configured. Please set your API key in settings.');
        }
        
        const prompt = this.buildPrompt(params);
        const url = `${this.baseURL}?key=${this.apiKey}`;
        
        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    contents: [{
                        parts: [{
                            text: prompt
                        }]
                    }],
                    generationConfig: {
                        temperature: 0.7,
                        topK: 40,
                        topP: 0.95,
                        maxOutputTokens: params.wordCount * 2, // Estimate tokens from word count
                    }
                })
            });
            
            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.error?.message || `API request failed with status ${response.status}`);
            }
            
            const data = await response.json();
            
            if (!data.candidates || !data.candidates[0] || !data.candidates[0].content) {
                throw new Error('Invalid response format from AI API');
            }
            
            const content = data.candidates[0].content.parts[0].text;
            
            // Simulate streaming if onProgress callback provided
            if (onProgress) {
                await this.simulateStreaming(content, onProgress);
            }
            
            return content;
            
        } catch (error) {
            console.error('AI API Error:', error);
            throw new Error(`AI generation failed: ${error.message}`);
        }
    }
    
    async simulateStreaming(content, onProgress) {
        const words = content.split(' ');
        let displayedContent = '';
        const chunkSize = 5; // Words per chunk
        
        for (let i = 0; i < words.length; i += chunkSize) {
            const chunk = words.slice(i, i + chunkSize).join(' ');
            displayedContent += (i > 0 ? ' ' : '') + chunk;
            
            // Calculate progress
            const progress = Math.min(100, (i / words.length) * 100);
            
            // Call progress callback
            if (onProgress) {
                onProgress(displayedContent, progress);
            }
            
            // Delay to simulate streaming
            await new Promise(resolve => setTimeout(resolve, 50));
        }
        
        // Ensure final content is displayed
        if (onProgress) {
            onProgress(content, 100);
        }
    }
    
    // Method to generate multiple articles with internal linking
    async generateArticlesWithInternalLinking(topics, params, onArticleProgress = null) {
        const articles = [];
        const topicMap = new Map();
        
        // First pass: generate all articles
        for (let i = 0; i < topics.length; i++) {
            const topic = topics[i];
            
            if (onArticleProgress) {
                onArticleProgress(i, topics.length, `Generating article: ${topic}`);
            }
            
            try {
                const articleParams = {
                    ...params,
                    topic: topic
                };
                
                const content = await this.generateContent(articleParams);
                articles.push({
                    title: topic,
                    content: content,
                    links: []
                });
                
                topicMap.set(topic.toLowerCase(), i);
                
            } catch (error) {
                console.error(`Failed to generate article for "${topic}":`, error);
                articles.push({
                    title: topic,
                    content: `Error generating content: ${error.message}`,
                    links: [],
                    error: true
                });
            }
            
            // Delay between requests
            if (i < topics.length - 1) {
                await new Promise(resolve => setTimeout(resolve, 2000));
            }
        }
        
        // Second pass: add internal links
        for (let i = 0; i < articles.length; i++) {
            if (articles[i].error) continue;
            
            let content = articles[i].content;
            const usedLinks = new Set();
            
            // Find opportunities to add internal links (minimum 5)
            for (const [targetTopic, targetIndex] of topicMap) {
                if (targetIndex === i || usedLinks.has(targetIndex)) continue;
                
                // Simple keyword matching for link opportunities
                const regex = new RegExp(`\\b${this.escapeRegExp(targetTopic)}\\b`, 'gi');
                const matches = content.match(regex);
                
                if (matches && matches.length > 0) {
                    // Replace first occurrence with a link
                    content = content.replace(regex, 
                        `<a href="#article-${targetIndex}" class="internal-link">${matches[0]}</a>`);
                    
                    articles[i].links.push(targetIndex);
                    usedLinks.add(targetIndex);
                    
                    if (articles[i].links.length >= 5) break; // Minimum 5 links achieved
                }
            }
            
            articles[i].content = content;
        }
        
        return articles;
    }
    
    escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }
}

// Global AI instance
window.aiIntegration = new AIIntegration();

// Global function for bulkwriter.js to use
window.aiGenerateContent = async function(params, onProgress) {
    await window.aiIntegration.initialize();
    return await window.aiIntegration.generateContent(params, onProgress);
};

// Initialize AI when script loads
document.addEventListener('DOMContentLoaded', async function() {
    await window.aiIntegration.initialize();
});