// Bulk Writer Application Logic
class BulkWriter {
    constructor() {
        this.topics = [];
        this.generatedArticles = [];
        this.isGenerating = false;
        this.currentGenerations = 0;
        this.totalGenerations = 0;
        
        this.initializeEventListeners();
        this.loadDefaultPrompt();
    }
    
    initializeEventListeners() {
        // Add topic button
        document.getElementById('add-topic-btn').addEventListener('click', () => this.addTopic());
        
        // Generate all button
        document.getElementById('generate-all-btn').addEventListener('click', () => this.generateAllContent());
        
        // Export all button
        document.getElementById('export-all-btn').addEventListener('click', () => this.exportAllArticles());
        
        // Remove topic event delegation
        document.getElementById('topics-container').addEventListener('click', (e) => {
            if (e.target.closest('.remove-topic')) {
                this.removeTopic(e.target.closest('.topic-row'));
            }
        });
    }
    
    addTopic() {
        const container = document.getElementById('topics-container');
        const topicCount = container.children.length;
        
        if (topicCount >= 100) {
            alert('Maximum 100 topics allowed');
            return;
        }
        
        const newRow = document.createElement('div');
        newRow.className = 'topic-row';
        newRow.innerHTML = `
            <div class="form-group">
                <label>Topic/Title</label>
                <input type="text" class="topic-input" placeholder="Enter topic or title">
            </div>
            <div class="form-group">
                <label>Word Count</label>
                <input type="number" class="word-count-input" value="1000" min="100" max="5000">
            </div>
            <button type="button" class="btn btn-danger btn-icon remove-topic">
                <i class="mdi mdi-delete"></i>
            </button>
        `;
        
        container.appendChild(newRow);
        
        // Show remove buttons on all rows if there's more than one
        this.updateRemoveButtons();
    }
    
    removeTopic(row) {
        if (document.getElementById('topics-container').children.length > 1) {
            row.remove();
            this.updateRemoveButtons();
        }
    }
    
    updateRemoveButtons() {
        const rows = document.getElementById('topics-container').children;
        const showRemove = rows.length > 1;
        
        Array.from(rows).forEach(row => {
            const removeBtn = row.querySelector('.remove-topic');
            if (removeBtn) {
                removeBtn.style.display = showRemove ? 'flex' : 'none';
            }
        });
    }
    
    loadDefaultPrompt() {
        // This would be the hardcoded default prompt from the requirements
        // For now, we'll set a placeholder
        const customPrompt = document.getElementById('custom-prompt');
        if (customPrompt) {
            customPrompt.placeholder = "Leave empty to use the default high-quality article generation prompt...";
        }
    }
    
    async generateAllContent() {
        // Validate API key first
        const apiKey = await this.getApiKey();
        if (!apiKey) {
            showAlert('Please configure your API key in the settings first');
            document.getElementById('api-settings-btn').click();
            return;
        }
        
        // Collect topics
        this.topics = this.collectTopics();
        if (this.topics.length === 0) {
            showAlert('Please add at least one topic');
            return;
        }
        
        // Validate topics
        const invalidTopics = this.topics.filter(topic => !topic.title.trim());
        if (invalidTopics.length > 0) {
            showAlert('Please fill in all topic titles');
            return;
        }
        
        // Setup generation
        this.isGenerating = true;
        this.currentGenerations = 0;
        this.totalGenerations = this.topics.length;
        this.generatedArticles = [];
        
        // Show results section
        document.getElementById('results-section').style.display = 'block';
        document.getElementById('completed-count').textContent = '0';
        document.getElementById('total-count').textContent = this.totalGenerations.toString();
        
        // Disable generate button, enable export button when done
        document.getElementById('generate-all-btn').disabled = true;
        document.getElementById('export-all-btn').disabled = true;
        
        // Show loading overlay
        showLoadingOverlay('Starting content generation...');
        
        // Create article cards
        this.createArticleCards();
        
        // Generate content for each topic
        for (let i = 0; i < this.topics.length; i++) {
            if (!this.isGenerating) break; // Allow cancellation
            
            const topic = this.topics[i];
            await this.generateArticle(topic, i);
            this.currentGenerations++;
            
            // Update progress
            updateGlobalProgress(
                this.currentGenerations, 
                this.totalGenerations,
                `Generated ${this.currentGenerations} of ${this.totalGenerations} articles`
            );
            
            // Small delay to avoid rate limiting
            if (i < this.topics.length - 1) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
        }
        
        // Finalize
        this.isGenerating = false;
        hideLoadingOverlay();
        document.getElementById('generate-all-btn').disabled = false;
        document.getElementById('export-all-btn').disabled = false;
        
        showAlert(`Successfully generated ${this.currentGenerations} articles!`, 'success');
    }
    
    collectTopics() {
        const topics = [];
        const rows = document.getElementById('topics-container').children;
        
        Array.from(rows).forEach((row, index) => {
            const titleInput = row.querySelector('.topic-input');
            const wordCountInput = row.querySelector('.word-count-input');
            
            topics.push({
                id: index,
                title: titleInput.value,
                wordCount: parseInt(wordCountInput.value) || 1000,
                element: row
            });
        });
        
        return topics;
    }
    
    createArticleCards() {
        const container = document.getElementById('results-container');
        container.innerHTML = '';
        
        this.topics.forEach(topic => {
            const card = document.createElement('div');
            card.className = 'article-card';
            card.id = `article-${topic.id}`;
            card.innerHTML = `
                <div class="article-header">
                    <div class="article-title">${topic.title || 'Untitled'}</div>
                    <div class="article-meta">
                        <span class="word-count">Target: ${formatWordCount(topic.wordCount)} words</span>
                        <span class="status">Pending</span>
                    </div>
                </div>
                <div class="article-content">
                    <div class="article-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                    <div class="generated-content" id="content-${topic.id}">
                        <em>Content generation pending...</em>
                    </div>
                    <div class="article-actions">
                        <button class="btn btn-outline btn-sm copy-btn" data-article="${topic.id}">
                            <i class="mdi mdi-content-copy"></i> Copy
                        </button>
                        <button class="btn btn-outline btn-sm export-btn" data-article="${topic.id}">
                            <i class="mdi mdi-download"></i> Export as .doc
                        </button>
                    </div>
                </div>
            `;
            container.appendChild(card);
        });
        
        // Add event listeners for copy and export buttons
        this.initializeArticleActions();
    }
    
    initializeArticleActions() {
        // Copy buttons
        document.querySelectorAll('.copy-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.copy-btn').getAttribute('data-article');
                this.copyArticleContent(articleId);
            });
        });
        
        // Export buttons
        document.querySelectorAll('.export-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.export-btn').getAttribute('data-article');
                this.exportSingleArticle(articleId);
            });
        });
    }
    
    async generateArticle(topic, index) {
        const contentElement = document.getElementById(`content-${topic.id}`);
        const progressFill = contentElement.closest('.article-content').querySelector('.progress-fill');
        const statusElement = contentElement.closest('.article-card').querySelector('.status');
        
        try {
            statusElement.textContent = 'Generating...';
            statusElement.style.color = 'var(--warning-color)';
            
            // Get generation parameters
            const params = this.getGenerationParameters(topic);
            
            // Generate content using AI
            const content = await this.generateWithAI(params, (chunk, progress) => {
                // Update streaming content
                if (contentElement) {
                    contentElement.innerHTML = this.formatContent(chunk);
                }
                
                // Update progress
                if (progressFill && progress) {
                    progressFill.style.width = `${progress}%`;
                }
            });
            
            // Store generated article
            this.generatedArticles[index] = {
                id: topic.id,
                title: topic.title,
                content: content,
                wordCount: this.countWords(content)
            };
            
            // Update UI
            statusElement.textContent = 'Completed';
            statusElement.style.color = 'var(--success-color)';
            
            const wordCountElement = contentElement.closest('.article-card').querySelector('.word-count');
            wordCountElement.textContent = `Generated: ${formatWordCount(this.generatedArticles[index].wordCount)} words`;
            
            // Update completed count
            document.getElementById('completed-count').textContent = this.currentGenerations.toString();
            
        } catch (error) {
            console.error('Error generating article:', error);
            statusElement.textContent = 'Failed';
            statusElement.style.color = 'var(--error-color)';
            contentElement.innerHTML = `<p style="color: var(--error-color);">Error: ${error.message}</p>`;
        }
    }
    
    getGenerationParameters(topic) {
        const customPrompt = document.getElementById('custom-prompt').value.trim();
        const contentType = document.getElementById('content-type').value;
        const tone = document.getElementById('tone-voice').value;
        const persona = document.getElementById('ai-persona').value;
        const focusKeyword = document.getElementById('focus-keyword').value;
        const otherKeywords = document.getElementById('other-keywords').value;
        const targetAudience = document.getElementById('target-audience').value;
        
        // Get selected add-ins
        const addins = Array.from(document.querySelectorAll('input[name="addins"]:checked'))
            .map(checkbox => checkbox.value);
        
        return {
            topic: topic.title,
            wordCount: topic.wordCount,
            customPrompt: customPrompt,
            contentType: contentType,
            tone: tone,
            persona: persona,
            focusKeyword: focusKeyword,
            otherKeywords: otherKeywords,
            targetAudience: targetAudience,
            addins: addins
        };
    }
    
    async generateWithAI(params, onProgress) {
        // This will be implemented in ai.js
        // For now, we'll use a placeholder that simulates streaming
        return await window.aiGenerateContent(params, onProgress);
    }
    
    formatContent(text) {
        // Basic formatting for display
        return text
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/\n/g, '<br>')
            .replace(/# (.*?)(?=\n|$)/g, '<h1>$1</h1>')
            .replace(/## (.*?)(?=\n|$)/g, '<h2>$1</h2>')
            .replace(/### (.*?)(?=\n|$)/g, '<h3>$1</h3>')
            .replace(/- (.*?)(?=\n|$)/g, '<li>$1</li>')
            .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>');
    }
    
    countWords(text) {
        return text.trim().split(/\s+/).length;
    }
    
    copyArticleContent(articleId) {
        const article = this.generatedArticles.find(a => a.id == articleId);
        if (article && article.content) {
            navigator.clipboard.writeText(article.content)
                .then(() => showAlert('Content copied to clipboard!', 'success'))
                .catch(() => showAlert('Failed to copy content'));
        } else {
            showAlert('No content available to copy');
        }
    }
    
    exportSingleArticle(articleId) {
        const article = this.generatedArticles.find(a => a.id == articleId);
        if (article && article.content) {
            exportAsDoc(article.content, article.title);
        } else {
            showAlert('No content available to export');
        }
    }
    
    exportAllArticles() {
        if (this.generatedArticles.length === 0) {
            showAlert('No articles generated yet');
            return;
        }
        
        downloadAllArticles(this.generatedArticles);
        showAlert(`Exported ${this.generatedArticles.length} articles!`, 'success');
    }
    
    async getApiKey() {
        try {
            const response = await fetch('get_api_key.php');
            const result = await response.json();
            return result.api_key || null;
        } catch (error) {
            console.error('Error fetching API key:', error);
            return null;
        }
    }
}

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.bulkWriter = new BulkWriter();
});