// Global Variables
let currentTheme = localStorage.getItem('theme') || 'light';

// Initialize application
document.addEventListener('DOMContentLoaded', function() {
    initializeTheme();
    initializeAuthTabs();
    initializeModals();
    initializeUserMenu();
});

// Theme Management
function initializeTheme() {
    const themeSwitch = document.getElementById('theme-switch');
    if (themeSwitch) {
        themeSwitch.checked = currentTheme === 'dark';
        setTheme(currentTheme);
        
        themeSwitch.addEventListener('change', function() {
            const newTheme = this.checked ? 'dark' : 'light';
            setTheme(newTheme);
        });
    }
}

function setTheme(theme) {
    currentTheme = theme;
    document.documentElement.setAttribute('data-theme', theme);
    localStorage.setItem('theme', theme);
}

// Auth Tabs
function initializeAuthTabs() {
    const tabButtons = document.querySelectorAll('.tab-button');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            // Update active tab button
            tabButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Show target tab content
            tabContents.forEach(content => {
                content.classList.remove('active');
                if (content.id === `${targetTab}-tab`) {
                    content.classList.add('active');
                }
            });
        });
    });
}

// Modal Management
function initializeModals() {
    // API Settings Modal
    const apiSettingsBtn = document.getElementById('api-settings-btn');
    const apiSettingsModal = document.getElementById('api-settings-modal');
    const closeApiModal = document.getElementById('close-api-modal');
    const modalClose = document.querySelector('.modal-close');
    
    if (apiSettingsBtn && apiSettingsModal) {
        apiSettingsBtn.addEventListener('click', function(e) {
            e.preventDefault();
            apiSettingsModal.classList.add('show');
        });
        
        const closeModal = () => apiSettingsModal.classList.remove('show');
        
        if (closeApiModal) closeApiModal.addEventListener('click', closeModal);
        if (modalClose) modalClose.addEventListener('click', closeModal);
        
        apiSettingsModal.addEventListener('click', function(e) {
            if (e.target === apiSettingsModal) {
                closeModal();
            }
        });
    }
}

// User Menu
function initializeUserMenu() {
    const userBtn = document.querySelector('.user-btn');
    const userDropdown = document.querySelector('.user-dropdown');
    
    if (userBtn && userDropdown) {
        userBtn.addEventListener('click', function() {
            userDropdown.style.display = userDropdown.style.display === 'block' ? 'none' : 'block';
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (!userBtn.contains(e.target) && !userDropdown.contains(e.target)) {
                userDropdown.style.display = 'none';
            }
        });
    }
}

// Utility Functions
function showAlert(message, type = 'error') {
    const alert = document.createElement('div');
    alert.className = `alert alert-${type}`;
    alert.innerHTML = `
        <i class="mdi mdi-${type === 'error' ? 'alert-circle' : 'check-circle'}"></i>
        ${message}
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

function formatWordCount(count) {
    return count.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// File Export Functions
function exportAsDoc(content, filename) {
    const htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>${filename}</title>
            <style>
                body { 
                    font-family: Arial, sans-serif; 
                    line-height: 1.6; 
                    margin: 40px; 
                    color: #333;
                }
                h1, h2, h3 { 
                    color: #2c3e50; 
                    margin-top: 1.5em;
                }
                h1 { 
                    border-bottom: 2px solid #3498db; 
                    padding-bottom: 0.5em;
                }
                strong { font-weight: bold; }
                em { font-style: italic; }
                ul, ol { margin: 1em 0; padding-left: 2em; }
                li { margin-bottom: 0.5em; }
            </style>
        </head>
        <body>
            ${content}
        </body>
        </html>
    `;
    
    const blob = new Blob([htmlContent], { type: 'application/msword' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `${filename.replace(/[^a-z0-9]/gi, '_')}.doc`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

function downloadAllArticles(articles) {
    articles.forEach(article => {
        if (article.content && article.title) {
            exportAsDoc(article.content, article.title);
        }
    });
}

// Loading Management
function showLoadingOverlay(message = 'Processing...') {
    const overlay = document.getElementById('loading-overlay');
    const progressText = document.getElementById('progress-text');
    
    if (overlay) {
        if (progressText && message) {
            progressText.textContent = message;
        }
        overlay.classList.add('show');
    }
}

function hideLoadingOverlay() {
    const overlay = document.getElementById('loading-overlay');
    if (overlay) {
        overlay.classList.remove('show');
    }
}

function updateGlobalProgress(completed, total, message) {
    const progressFill = document.getElementById('global-progress');
    const progressText = document.getElementById('progress-text');
    
    if (progressFill) {
        const percentage = total > 0 ? (completed / total) * 100 : 0;
        progressFill.style.width = `${percentage}%`;
    }
    
    if (progressText && message) {
        progressText.textContent = message;
    }
}

// API Key Management
async function saveApiKey() {
    const apiKeyInput = document.getElementById('api-key');
    const partialKey = apiKeyInput.value.trim();
    
    if (!partialKey) {
        showAlert('Please enter your API key');
        return;
    }
    
    const fullKey = 'AIzaSy' + partialKey;
    
    try {
        // Save to users.json via PHP
        const response = await fetch('save_api_key.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                api_key: fullKey
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showAlert('API key saved successfully!', 'success');
            document.getElementById('api-settings-modal').classList.remove('show');
        } else {
            showAlert('Failed to save API key: ' + (result.message || 'Unknown error'));
        }
    } catch (error) {
        showAlert('Error saving API key: ' + error.message);
    }
}

// Initialize save API key button
document.addEventListener('DOMContentLoaded', function() {
    const saveApiKeyBtn = document.getElementById('save-api-key');
    if (saveApiKeyBtn) {
        saveApiKeyBtn.addEventListener('click', saveApiKey);
    }
});