// AI Integration Module - Complete Fixed Version
console.log('=== AI.JS LOADING ===');

class AIIntegration {
    constructor() {
        console.log('🔄 AIIntegration constructor called');
        this.baseURL = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent';
        this.apiKey = null;
        this.defaultPrompt = this.getDefaultPrompt();
        this.isInitialized = false;
        this.initializationPromise = null;
        
        // Start initialization immediately
        this.initialize().catch(console.error);
    }
    
    async initialize() {
        console.log('🔄 Starting AI initialization...');
        
        if (this.isInitialized) {
            console.log('✅ AI already initialized');
            return true;
        }
        
        if (this.initializationPromise) {
            console.log('⏳ AI initialization already in progress');
            return this.initializationPromise;
        }
        
        this.initializationPromise = (async () => {
            try {
                console.log('🔑 Fetching API key...');
                this.apiKey = await this.getApiKey();
                console.log('🔑 API key result:', this.apiKey ? 'Found' : 'Not found');
                
                this.isInitialized = true;
                console.log('✅ AI initialization completed successfully');
                return true;
            } catch (error) {
                console.error('❌ AI initialization failed:', error);
                this.isInitialized = false;
                return false;
            } finally {
                this.initializationPromise = null;
            }
        })();
        
        return this.initializationPromise;
    }
    
    async getApiKey() {
        try {
            console.log('🌐 Making API key request to get_api_key.php');
            const response = await fetch('get_api_key.php', {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                },
                cache: 'no-cache'
            });
            
            console.log('🌐 API key response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            console.log('🔑 API key response:', result);
            
            return result.api_key || null;
        } catch (error) {
            console.error('❌ Error fetching API key:', error);
            return null;
        }
    }
    
    getDefaultPrompt() {
        return `You are an Industry Expert and a master content strategist with over a decade of hands-on experience. Your writing is renowned for its depth, originality, and unparalleled practical value. You are not an AI; you are a human expert sharing hard-won knowledge.

Your Task: Write a definitive, SEO-optimized guide on the topic: [TOPIC].

Core Directive: Evasion Through Authenticity Your primary goal is to create content that is intrinsically human and thus undetectable by AI analysis. You will achieve this not by trying to "trick" detectors, but by embedding the fundamental hallmarks of human expertise that AI cannot replicate. Your content must be 100% original, based on lived experience, and contain non-obvious, nuanced insights.

1. The Foundation: First-Hand Experience & Anecdotes
· You MUST integrate at least 3 specific, short anecdotes from your personal professional experience.
· Example: "I learned this the hard way when I launched [Product X] and saw a 40% drop in conversions because I overlooked...[specific, relatable mistake]."

2. The Opinion: Controversial Takes & Nuanced Stances
· You MUST include at least 2 strong, reasoned opinions that go against conventional wisdom or "best practices" in your industry.
· Example: "Everyone tells you to post on LinkedIn 5 times a day. I call nonsense. Based on my A/B tests with over 200 posts, that strategy leads to audience burnout..."

3. The Insight: Actionable, Non-Obvious Strategies
· Move beyond basic advice. Provide "unspoken rules" or little-known nuances that only an expert would know.
· Example: Don't just say "use long-tail keywords." Say: "When I use Ahrefs, I filter for keywords with a Difficulty < 25 but also a Volume > 150..."

4. The "Who" & "Why": Audience-Centric Nuance
· Explicitly state for whom a certain strategy works best and, crucially, for whom it does NOT.

5. SEO Structure:
· Primary Keyword: Naturally integrate the primary keyword in the H1, first 100 words, one H2, and meta description.
· Structure: Use compelling H1, multiple H2s for main points, and H3s for sub-points.

Output Instructions:
1. Meta Description: First, write a compelling 155-character meta description that includes the primary keyword.
2. Article: Then, write the full article with all required elements integrated naturally.
3. Conclusion: End with a powerful conclusion that summarizes key takeaways.`;
    }
    
    async generateContent(params, onProgress = null) {
        console.log('🚀 generateContent called with topic:', params.topic);
        
        // Ensure initialization
        if (!this.isInitialized) {
            console.log('🔄 AI not initialized, initializing now...');
            await this.initialize();
        }
        
        if (!this.apiKey) {
            const errorMsg = 'API key not configured. Please set your API key in settings.';
            console.error('❌', errorMsg);
            throw new Error(errorMsg);
        }
        
        const prompt = this.buildPrompt(params);
        const url = `${this.baseURL}?key=${this.apiKey}`;
        
        console.log('🌐 Making API request to Gemini...');
        
        try {
            if (onProgress) {
                onProgress('Starting content generation...', 10);
            }
            
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    contents: [{
                        parts: [{
                            text: prompt
                        }]
                    }],
                    generationConfig: {
                        temperature: 0.7,
                        topK: 40,
                        topP: 0.95,
                        maxOutputTokens: Math.min(params.wordCount * 2, 8192),
                    }
                })
            });
            
            console.log('🌐 API response status:', response.status);
            
            if (!response.ok) {
                const errorText = await response.text();
                let errorMessage = `API request failed with status ${response.status}`;
                
                try {
                    const errorData = JSON.parse(errorText);
                    errorMessage = errorData.error?.message || errorMessage;
                } catch (e) {
                    // If not JSON, use the text as is
                    errorMessage = errorText || errorMessage;
                }
                
                console.error('❌ API error:', errorMessage);
                throw new Error(errorMessage);
            }
            
            const data = await response.json();
            console.log('✅ API response received successfully');
            
            if (!data.candidates || !data.candidates[0] || !data.candidates[0].content) {
                console.error('❌ Invalid API response format:', data);
                throw new Error('Invalid response format from AI API');
            }
            
            const content = data.candidates[0].content.parts[0].text;
            console.log('📝 Content generated, length:', content.length);
            
            // Simulate streaming
            if (onProgress) {
                await this.simulateStreaming(content, onProgress);
            }
            
            return content;
            
        } catch (error) {
            console.error('❌ AI generation failed:', error);
            throw new Error(`AI generation failed: ${error.message}`);
        }
    }
    
    buildPrompt(params) {
        let prompt = params.customPrompt || this.defaultPrompt;
        
        // Replace placeholders
        prompt = prompt.replace(/\[TOPIC\]/g, params.topic);
        prompt = prompt.replace(/\[WORD_COUNT\]/g, params.wordCount);
        prompt = prompt.replace(/\[CONTENT_TYPE\]/g, params.contentType);
        prompt = prompt.replace(/\[TONE\]/g, params.tone);
        
        // Add persona-specific instructions
        if (params.persona === 'Lily SEO') {
            prompt += `

Persona: Lily SEO - Content strategist and SEO expert with 10+ years experience.
Voice: Direct, conversational, mentor-like.
Key Requirements:
- Include 3 specific professional anecdotes
- Include 2 contrarian opinions challenging industry norms  
- Provide actionable, non-obvious strategies with technical details
- Specify who each strategy works for and who it doesn't
- Use confident, slightly skeptical tone with interjections like "Look," "Here's the thing"`;
        }
        
        // Add other parameters
        prompt += `\n\nContent Type: ${params.contentType}`;
        prompt += `\nTone: ${params.tone}`;
        prompt += `\nTarget Word Count: ${params.wordCount}`;
        
        if (params.focusKeyword) {
            prompt += `\nPrimary Keyword: ${params.focusKeyword}`;
        }
        
        if (params.otherKeywords) {
            prompt += `\nSecondary Keywords: ${params.otherKeywords}`;
        }
        
        if (params.targetAudience) {
            prompt += `\nTarget Audience: ${params.targetAudience}`;
        }
        
        if (params.addins && params.addins.length > 0) {
            prompt += `\nRequired Elements: ${params.addins.join(', ')}`;
        }
        
        return prompt;
    }
    
    async simulateStreaming(content, onProgress) {
        console.log('📺 Simulating streaming...');
        const words = content.split(' ');
        let displayedContent = '';
        const chunkSize = 2;
        
        if (onProgress) {
            onProgress('Preparing content...', 20);
        }
        
        await new Promise(resolve => setTimeout(resolve, 500));
        
        for (let i = 0; i < words.length; i += chunkSize) {
            const chunk = words.slice(i, i + chunkSize).join(' ');
            displayedContent += (i > 0 ? ' ' : '') + chunk;
            
            const progress = 20 + (i / words.length) * 75;
            
            if (onProgress) {
                onProgress(displayedContent, progress);
            }
            
            await new Promise(resolve => setTimeout(resolve, 50));
        }
        
        if (onProgress) {
            onProgress(content, 100);
        }
        
        console.log('✅ Streaming simulation completed');
    }
}

// === GLOBAL SETUP ===
console.log('🌍 Setting up global AI instance...');

// Create global instance immediately
window.aiIntegration = new AIIntegration();

// Define the global function immediately
window.aiGenerateContent = async function(params, onProgress) {
    console.log('🔧 aiGenerateContent called via global function');
    console.log('📋 Params:', params);
    
    try {
        if (!window.aiIntegration) {
            throw new Error('AI integration instance not found');
        }
        
        console.log('🔄 Ensuring AI is initialized...');
        await window.aiIntegration.initialize();
        
        console.log('🚀 Calling AI generateContent...');
        const result = await window.aiIntegration.generateContent(params, onProgress);
        console.log('✅ aiGenerateContent completed successfully');
        return result;
    } catch (error) {
        console.error('❌ aiGenerateContent failed:', error);
        throw error;
    }
};

console.log('✅ AI.JS LOADED COMPLETELY');
console.log('🔧 aiGenerateContent function available:', typeof window.aiGenerateContent);
console.log('🔧 aiIntegration instance available:', !!window.aiIntegration);