// Bulk Writer Application Logic - Complete Fixed Version
console.log('=== BULKWRITER.JS LOADING ===');

class BulkWriter {
    constructor() {
        console.log('🔄 BulkWriter constructor called');
        
        this.topics = [];
        this.generatedArticles = [];
        this.isGenerating = false;
        this.currentGenerations = 0;
        this.totalGenerations = 0;
        
        // Wait for DOM to be fully ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            setTimeout(() => this.init(), 100);
        }
    }
    
    init() {
        console.log('🚀 BulkWriter initializing...');
        console.log('🔧 Checking AI function:', typeof window.aiGenerateContent);
        console.log('🔧 Checking AI instance:', !!window.aiIntegration);
        
        this.initializeEventListeners();
        this.loadDefaultPrompt();
        
        console.log('✅ BulkWriter initialized successfully');
    }
    
    initializeEventListeners() {
        console.log('🔗 Initializing event listeners...');
        
        // Add topic button
        const addTopicBtn = document.getElementById('add-topic-btn');
        if (addTopicBtn) {
            addTopicBtn.addEventListener('click', () => this.addTopic());
            console.log('✅ Add topic listener added');
        }
        
        // Generate all button
        const generateAllBtn = document.getElementById('generate-all-btn');
        if (generateAllBtn) {
            generateAllBtn.addEventListener('click', () => this.generateAllContent());
            console.log('✅ Generate all listener added');
        }
        
        // Export all button
        const exportAllBtn = document.getElementById('export-all-btn');
        if (exportAllBtn) {
            exportAllBtn.addEventListener('click', () => this.exportAllArticles());
            console.log('✅ Export all listener added');
        }
        
        // Remove topic event delegation
        const topicsContainer = document.getElementById('topics-container');
        if (topicsContainer) {
            topicsContainer.addEventListener('click', (e) => {
                if (e.target.closest('.remove-topic')) {
                    this.removeTopic(e.target.closest('.topic-row'));
                }
            });
            console.log('✅ Topics container listener added');
        }
        
        console.log('✅ All event listeners initialized');
    }
    
    addTopic() {
        console.log('➕ Adding new topic...');
        const container = document.getElementById('topics-container');
        if (!container) return;
        
        const topicCount = container.children.length;
        if (topicCount >= 100) {
            alert('Maximum 100 topics allowed');
            return;
        }
        
        const newRow = document.createElement('div');
        newRow.className = 'topic-row';
        newRow.innerHTML = `
            <div class="form-group">
                <label>Topic/Title</label>
                <input type="text" class="topic-input" placeholder="Enter topic or title">
            </div>
            <div class="form-group">
                <label>Word Count</label>
                <input type="number" class="word-count-input" value="1000" min="100" max="5000">
            </div>
            <button type="button" class="btn btn-danger btn-icon remove-topic">
                <i class="mdi mdi-delete"></i>
            </button>
        `;
        
        container.appendChild(newRow);
        this.updateRemoveButtons();
        console.log('✅ New topic added');
    }
    
    removeTopic(row) {
        const container = document.getElementById('topics-container');
        if (!container || !row) return;
        
        if (container.children.length > 1) {
            row.remove();
            this.updateRemoveButtons();
        }
    }
    
    updateRemoveButtons() {
        const container = document.getElementById('topics-container');
        if (!container) return;
        
        const rows = container.children;
        const showRemove = rows.length > 1;
        
        Array.from(rows).forEach(row => {
            const removeBtn = row.querySelector('.remove-topic');
            if (removeBtn) {
                removeBtn.style.display = showRemove ? 'flex' : 'none';
            }
        });
    }
    
    loadDefaultPrompt() {
        const customPrompt = document.getElementById('custom-prompt');
        if (customPrompt) {
            customPrompt.placeholder = "Leave empty to use the default high-quality article generation prompt...";
        }
    }
    
    async generateAllContent() {
        console.log('🎯 generateAllContent started');
        
        // Check AI function availability
        if (typeof window.aiGenerateContent !== 'function') {
            const errorMsg = 'AI system is not properly initialized. Please refresh the page and try again.';
            console.error('❌', errorMsg);
            showAlert(errorMsg);
            return;
        }
        
        console.log('✅ AI function is available');
        
        // Check API key
        console.log('🔑 Checking API key...');
        const apiKey = await this.getApiKey();
        if (!apiKey) {
            const errorMsg = 'Please configure your API key in the settings first. Click your username → API Settings.';
            console.error('❌', errorMsg);
            showAlert(errorMsg);
            const apiSettingsBtn = document.getElementById('api-settings-btn');
            if (apiSettingsBtn) apiSettingsBtn.click();
            return;
        }
        
        console.log('✅ API key found');
        
        // Collect and validate topics
        this.topics = this.collectTopics();
        console.log('📝 Topics collected:', this.topics.length);
        
        if (this.topics.length === 0) {
            showAlert('Please add at least one topic');
            return;
        }
        
        const invalidTopics = this.topics.filter(topic => !topic.title.trim());
        if (invalidTopics.length > 0) {
            showAlert('Please fill in all topic titles');
            return;
        }
        
        console.log('✅ All topics validated');
        
        // Setup generation
        this.isGenerating = true;
        this.currentGenerations = 0;
        this.totalGenerations = this.topics.length;
        this.generatedArticles = [];
        
        // Show UI elements
        const resultsSection = document.getElementById('results-section');
        if (resultsSection) resultsSection.style.display = 'block';
        
        const completedCount = document.getElementById('completed-count');
        const totalCount = document.getElementById('total-count');
        if (completedCount && totalCount) {
            completedCount.textContent = '0';
            totalCount.textContent = this.totalGenerations.toString();
        }
        
        // Update buttons
        const generateBtn = document.getElementById('generate-all-btn');
        const exportBtn = document.getElementById('export-all-btn');
        if (generateBtn) generateBtn.disabled = true;
        if (exportBtn) exportBtn.disabled = true;
        
        // Show loading
        showLoadingOverlay('Starting content generation...');
        
        // Create article cards
        this.createArticleCards();
        
        console.log('🚀 Starting content generation loop...');
        
        // Generate each article
        for (let i = 0; i < this.topics.length; i++) {
            if (!this.isGenerating) break;
            
            const topic = this.topics[i];
            console.log(`📄 Generating article ${i + 1}/${this.topics.length}: "${topic.title}"`);
            
            await this.generateArticle(topic, i);
            this.currentGenerations++;
            
            updateGlobalProgress(
                this.currentGenerations, 
                this.totalGenerations,
                `Generated ${this.currentGenerations} of ${this.totalGenerations} articles`
            );
            
            // Delay between requests
            if (i < this.topics.length - 1) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
        }
        
        // Cleanup
        this.isGenerating = false;
        hideLoadingOverlay();
        if (generateBtn) generateBtn.disabled = false;
        if (exportBtn) exportBtn.disabled = false;
        
        const successMsg = `Successfully generated ${this.currentGenerations} articles!`;
        console.log('✅', successMsg);
        showAlert(successMsg, 'success');
    }
    
    collectTopics() {
        const topics = [];
        const container = document.getElementById('topics-container');
        if (!container) return topics;
        
        const rows = container.children;
        Array.from(rows).forEach((row, index) => {
            const titleInput = row.querySelector('.topic-input');
            const wordCountInput = row.querySelector('.word-count-input');
            
            if (titleInput && wordCountInput) {
                topics.push({
                    id: index,
                    title: titleInput.value,
                    wordCount: parseInt(wordCountInput.value) || 1000,
                    element: row
                });
            }
        });
        
        return topics;
    }
    
    createArticleCards() {
        const container = document.getElementById('results-container');
        if (!container) return;
        
        container.innerHTML = '';
        
        this.topics.forEach(topic => {
            const card = document.createElement('div');
            card.className = 'article-card';
            card.id = `article-${topic.id}`;
            card.innerHTML = `
                <div class="article-header">
                    <div class="article-title">${this.escapeHtml(topic.title) || 'Untitled'}</div>
                    <div class="article-meta">
                        <span class="word-count">Target: ${formatWordCount(topic.wordCount)} words</span>
                        <span class="status">Pending</span>
                    </div>
                </div>
                <div class="article-content">
                    <div class="article-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                    <div class="generated-content" id="content-${topic.id}">
                        <em>Content generation pending...</em>
                    </div>
                    <div class="article-actions">
                        <button class="btn btn-outline btn-sm copy-btn" data-article="${topic.id}">
                            <i class="mdi mdi-content-copy"></i> Copy
                        </button>
                        <button class="btn btn-outline btn-sm export-btn" data-article="${topic.id}">
                            <i class="mdi mdi-download"></i> Export as .doc
                        </button>
                    </div>
                </div>
            `;
            container.appendChild(card);
        });
        
        this.initializeArticleActions();
    }
    
    initializeArticleActions() {
        document.querySelectorAll('.copy-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.copy-btn').getAttribute('data-article');
                this.copyArticleContent(articleId);
            });
        });
        
        document.querySelectorAll('.export-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.export-btn').getAttribute('data-article');
                this.exportSingleArticle(articleId);
            });
        });
    }
    
    async generateArticle(topic, index) {
        console.log(`🔄 Generating article for: "${topic.title}"`);
        
        const contentElement = document.getElementById(`content-${topic.id}`);
        if (!contentElement) {
            console.error(`❌ Content element not found for topic ${topic.id}`);
            return;
        }
        
        const articleCard = contentElement.closest('.article-content');
        const progressFill = articleCard ? articleCard.querySelector('.progress-fill') : null;
        const statusElement = articleCard ? articleCard.closest('.article-card').querySelector('.status') : null;
        
        try {
            if (statusElement) {
                statusElement.textContent = 'Generating...';
                statusElement.style.color = 'var(--warning-color)';
            }
            
            const params = this.getGenerationParameters(topic);
            console.log('📋 Generation params:', params);
            
            const content = await window.aiGenerateContent(params, (chunk, progress) => {
                if (contentElement) {
                    contentElement.innerHTML = this.formatContent(chunk);
                }
                if (progressFill && progress) {
                    progressFill.style.width = `${progress}%`;
                }
            });
            
            console.log(`✅ Article generated: "${topic.title}"`);
            
            this.generatedArticles[index] = {
                id: topic.id,
                title: topic.title,
                content: content,
                wordCount: this.countWords(content)
            };
            
            if (statusElement) {
                statusElement.textContent = 'Completed';
                statusElement.style.color = 'var(--success-color)';
            }
            
            const wordCountElement = contentElement.closest('.article-card')?.querySelector('.word-count');
            if (wordCountElement) {
                wordCountElement.textContent = `Generated: ${formatWordCount(this.generatedArticles[index].wordCount)} words`;
            }
            
            const completedCount = document.getElementById('completed-count');
            if (completedCount) {
                completedCount.textContent = this.currentGenerations.toString();
            }
            
        } catch (error) {
            console.error(`❌ Failed to generate article "${topic.title}":`, error);
            if (statusElement) {
                statusElement.textContent = 'Failed';
                statusElement.style.color = 'var(--error-color)';
            }
            if (contentElement) {
                contentElement.innerHTML = `<p style="color: var(--error-color);">Error: ${this.escapeHtml(error.message)}</p>`;
            }
        }
    }
    
    getGenerationParameters(topic) {
        const getValue = (id, defaultValue = '') => {
            const element = document.getElementById(id);
            return element ? element.value : defaultValue;
        };
        
        const customPrompt = getValue('custom-prompt', '');
        const contentType = getValue('content-type', 'Articles');
        const tone = getValue('tone-voice', 'Professional');
        const persona = getValue('ai-persona', 'Lily SEO');
        const focusKeyword = getValue('focus-keyword', '');
        const otherKeywords = getValue('other-keywords', '');
        const targetAudience = getValue('target-audience', '');
        
        const addins = Array.from(document.querySelectorAll('input[name="addins"]:checked'))
            .map(checkbox => checkbox.value);
        
        return {
            topic: topic.title,
            wordCount: topic.wordCount,
            customPrompt: customPrompt,
            contentType: contentType,
            tone: tone,
            persona: persona,
            focusKeyword: focusKeyword,
            otherKeywords: otherKeywords,
            targetAudience: targetAudience,
            addins: addins
        };
    }
    
    formatContent(text) {
        if (!text) return '';
        return text
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/\n/g, '<br>')
            .replace(/# (.*?)(?=\n|$)/g, '<h1>$1</h1>')
            .replace(/## (.*?)(?=\n|$)/g, '<h2>$1</h2>')
            .replace(/### (.*?)(?=\n|$)/g, '<h3>$1</h3>')
            .replace(/- (.*?)(?=\n|$)/g, '<li>$1</li>')
            .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>');
    }
    
    countWords(text) {
        return text ? text.trim().split(/\s+/).length : 0;
    }
    
    escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
    
    copyArticleContent(articleId) {
        const article = this.generatedArticles.find(a => a && a.id == articleId);
        if (article && article.content) {
            navigator.clipboard.writeText(article.content)
                .then(() => showAlert('Content copied to clipboard!', 'success'))
                .catch(() => showAlert('Failed to copy content'));
        } else {
            showAlert('No content available to copy');
        }
    }
    
    exportSingleArticle(articleId) {
        const article = this.generatedArticles.find(a => a && a.id == articleId);
        if (article && article.content) {
            exportAsDoc(article.content, article.title);
        } else {
            showAlert('No content available to export');
        }
    }
    
    exportAllArticles() {
        const validArticles = this.generatedArticles.filter(a => a && a.content);
        if (validArticles.length === 0) {
            showAlert('No articles generated yet');
            return;
        }
        
        downloadAllArticles(validArticles);
        showAlert(`Exported ${validArticles.length} articles!`, 'success');
    }
    
    async getApiKey() {
        try {
            const response = await fetch('get_api_key.php');
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            const result = await response.json();
            return result.api_key || null;
        } catch (error) {
            console.error('Error fetching API key:', error);
            return null;
        }
    }
}

// Initialize when DOM is ready
console.log('🌍 Setting up BulkWriter...');

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        console.log('📄 DOM fully loaded, creating BulkWriter...');
        window.bulkWriter = new BulkWriter();
    });
} else {
    console.log('📄 DOM already ready, creating BulkWriter...');
    window.bulkWriter = new BulkWriter();
}

console.log('✅ BULKWRITER.JS LOADED COMPLETELY');