<?php
// js-proxy.php - Secure JavaScript Proxy

// Strict security headers
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: DENY");
header("X-XSS-Protection: 1; mode=block");
header("Referrer-Policy: strict-origin-when-cross-origin");
header("Feature-Policy: microphone 'none'; camera 'none'; geolocation 'none'");

// Cache control headers to prevent caching
header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
header("Cache-Control: post-check=0, pre-check=0", false);
header("Pragma: no-cache");

// Content Security Policy
header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline' https://fonts.googleapis.com; style-src 'self' 'unsafe-inline' https://fonts.googleapis.com; img-src 'self' data: https://randomuser.me; font-src 'self' https://fonts.gstatic.com; connect-src 'self' https://generativelanguage.googleapis.com;");

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    header('HTTP/1.1 405 Method Not Allowed');
    header('Allow: GET');
    exit('405 Method Not Allowed - Only GET requests are permitted');
}

// Validate the request origin
$allowed_origins = ['https://yourdomain.com', 'https://www.yourdomain.com'];
$origin = $_SERVER['HTTP_ORIGIN'] ?? '';
if (in_array($origin, $allowed_origins)) {
    header("Access-Control-Allow-Origin: $origin");
    header("Access-Control-Allow-Credentials: true");
}

// Path to your protected JavaScript file
$jsPath = '/home/fuzesale/cmp/app.js';

// Verify the file exists and is readable
if (!file_exists($jsPath) || !is_readable($jsPath)) {
    header('HTTP/1.1 500 Internal Server Error');
    error_log("JavaScript file not found or not readable: " . $jsPath);
    exit('console.error("Required JavaScript resources could not be loaded");');
}

// Verify the file is actually a JavaScript file
$mime_type = mime_content_type($jsPath);
if ($mime_type !== 'text/javascript' && $mime_type !== 'application/javascript') {
    header('HTTP/1.1 403 Forbidden');
    error_log("Attempt to access non-JavaScript file: " . $jsPath);
    exit('console.error("Invalid resource type");');
}

// Set the proper content type
header('Content-Type: application/javascript; charset=UTF-8');

// Add a Content-Length header
header('Content-Length: ' . filesize($jsPath));

// Security token verification (optional)
$expected_token = 'YOUR_SECURE_TOKEN_HASH'; // Set this in your environment
$provided_token = $_GET['token'] ?? '';
if (!hash_equals($expected_token, hash('sha256', $provided_token))) {
    header('HTTP/1.1 403 Forbidden');
    exit('console.error("Access denied");');
}

// File delivery with proper caching headers
$last_modified = filemtime($jsPath);
$etag = md5_file($jsPath);

header("Last-Modified: " . gmdate("D, d M Y H:i:s", $last_modified) . " GMT");
header("Etag: $etag");

if (isset($_SERVER['HTTP_IF_MODIFIED_SINCE']) || isset($_SERVER['HTTP_IF_NONE_MATCH'])) {
    if ($_SERVER['HTTP_IF_MODIFIED_SINCE'] == gmdate("D, d M Y H:i:s", $last_modified) . " GMT" || 
        trim($_SERVER['HTTP_IF_NONE_MATCH']) == $etag) {
        header('HTTP/1.1 304 Not Modified');
        exit();
    }
}

// Rate limiting (basic implementation)
$ip = $_SERVER['REMOTE_ADDR'];
$cache_dir = '/tmp/js_proxy_cache/';
if (!file_exists($cache_dir)) {
    mkdir($cache_dir, 0755, true);
}

$rate_limit_file = $cache_dir . md5($ip);
$requests = 1;

if (file_exists($rate_limit_file)) {
    $requests = (int)file_get_contents($rate_limit_file) + 1;
    if ($requests > 50) { // 50 requests per minute
        header('HTTP/1.1 429 Too Many Requests');
        header('Retry-After: 60');
        exit('console.error("Rate limit exceeded. Please try again later.");');
    }
}

file_put_contents($rate_limit_file, $requests);

// Clean old rate limit files
if (rand(1, 100) === 1) { // 1% chance to clean up
    foreach (glob($cache_dir . "*") as $file) {
        if (filemtime($file) < time() - 60) { // Older than 1 minute
            unlink($file);
        }
    }
}

// Deliver the JavaScript file with proper security headers
readfile($jsPath);

// Log successful delivery
error_log("JavaScript file delivered successfully to IP: " . $_SERVER['REMOTE_ADDR']);

exit();
?>