// Bulk Writer Application Logic - Enhanced Version
console.log('=== BULKWRITER.JS LOADING ===');

class BulkWriter {
    constructor() {
        console.log('🔄 BulkWriter constructor called');
        
        this.topics = [];
        this.generatedArticles = [];
        this.isGenerating = false;
        this.currentGenerations = 0;
        this.totalGenerations = 0;
        
        // Wait for DOM to be fully ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            setTimeout(() => this.init(), 100);
        }
    }
    
    init() {
        console.log('🚀 BulkWriter initializing...');
        console.log('🔧 Checking AI function:', typeof window.aiGenerateContent);
        console.log('🔧 Checking AI instance:', !!window.aiIntegration);
        
        this.initializeEventListeners();
        this.loadDefaultPrompt();
        
        console.log('✅ BulkWriter initialized successfully');
    }
    
    initializeEventListeners() {
        console.log('🔗 Initializing event listeners...');
        
        // Add topic button
        const addTopicBtn = document.getElementById('add-topic-btn');
        if (addTopicBtn) {
            addTopicBtn.addEventListener('click', () => this.addTopic());
            console.log('✅ Add topic listener added');
        }
        
        // Generate all button
        const generateAllBtn = document.getElementById('generate-all-btn');
        if (generateAllBtn) {
            generateAllBtn.addEventListener('click', () => this.generateAllContent());
            console.log('✅ Generate all listener added');
        }
        
        // Export all button
        const exportAllBtn = document.getElementById('export-all-btn');
        if (exportAllBtn) {
            exportAllBtn.addEventListener('click', () => this.exportAllArticles());
            console.log('✅ Export all listener added');
        }
        
        // Remove topic event delegation
        const topicsContainer = document.getElementById('topics-container');
        if (topicsContainer) {
            topicsContainer.addEventListener('click', (e) => {
                if (e.target.closest('.remove-topic')) {
                    this.removeTopic(e.target.closest('.topic-row'));
                }
            });
            console.log('✅ Topics container listener added');
        }
        
        console.log('✅ All event listeners initialized');
    }
    
    addTopic() {
        console.log('➕ Adding new topic...');
        const container = document.getElementById('topics-container');
        if (!container) return;
        
        const topicCount = container.children.length;
        if (topicCount >= 100) {
            alert('Maximum 100 topics allowed');
            return;
        }
        
        const newRow = document.createElement('div');
        newRow.className = 'topic-row';
        newRow.innerHTML = `
            <div class="form-group">
                <label>Topic/Title</label>
                <input type="text" class="topic-input" placeholder="Enter topic or title">
            </div>
            <div class="form-group">
                <label>Word Count</label>
                <input type="number" class="word-count-input" value="1000" min="100" max="5000">
            </div>
            <button type="button" class="btn btn-danger btn-icon remove-topic">
                <i class="mdi mdi-delete"></i>
            </button>
        `;
        
        container.appendChild(newRow);
        this.updateRemoveButtons();
        console.log('✅ New topic added');
    }
    
    removeTopic(row) {
        const container = document.getElementById('topics-container');
        if (!container || !row) return;
        
        if (container.children.length > 1) {
            row.remove();
            this.updateRemoveButtons();
        }
    }
    
    updateRemoveButtons() {
        const container = document.getElementById('topics-container');
        if (!container) return;
        
        const rows = container.children;
        const showRemove = rows.length > 1;
        
        Array.from(rows).forEach(row => {
            const removeBtn = row.querySelector('.remove-topic');
            if (removeBtn) {
                removeBtn.style.display = showRemove ? 'flex' : 'none';
            }
        });
    }
    
    loadDefaultPrompt() {
        const customPrompt = document.getElementById('custom-prompt');
        if (customPrompt) {
            customPrompt.placeholder = "Leave empty to use the default high-quality article generation prompt...";
        }
    }
    
    async generateAllContent() {
        console.log('🎯 generateAllContent started');
        
        // Check AI function availability
        if (typeof window.aiGenerateContent !== 'function') {
            const errorMsg = 'AI system is not properly initialized. Please refresh the page and try again.';
            console.error('❌', errorMsg);
            showAlert(errorMsg);
            return;
        }
        
        console.log('✅ AI function is available');
        
        // Check API key
        console.log('🔑 Checking API key...');
        const apiKey = await this.getApiKey();
        if (!apiKey) {
            const errorMsg = 'Please configure your AI Access key in the settings first. Click your username → API Settings.';
            console.error('❌', errorMsg);
            showAlert(errorMsg);
            const apiSettingsBtn = document.getElementById('api-settings-btn');
            if (apiSettingsBtn) apiSettingsBtn.click();
            return;
        }
        
        console.log('✅ API key found');
        
        // Collect and validate topics
        this.topics = this.collectTopics();
        console.log('📝 Topics collected:', this.topics.length);
        
        if (this.topics.length === 0) {
            showAlert('Please add at least one topic');
            return;
        }
        
        const invalidTopics = this.topics.filter(topic => !topic.title.trim());
        if (invalidTopics.length > 0) {
            showAlert('Please fill in all topic titles');
            return;
        }
        
        console.log('✅ All topics validated');
        
        // Setup generation
        this.isGenerating = true;
        this.currentGenerations = 0;
        this.totalGenerations = this.topics.length;
        this.generatedArticles = [];
        
        // Show UI elements
        const resultsSection = document.getElementById('results-section');
        if (resultsSection) resultsSection.style.display = 'block';
        
        const completedCount = document.getElementById('completed-count');
        const totalCount = document.getElementById('total-count');
        if (completedCount && totalCount) {
            completedCount.textContent = '0';
            totalCount.textContent = this.totalGenerations.toString();
        }
        
        // Update buttons
        const generateBtn = document.getElementById('generate-all-btn');
        const exportBtn = document.getElementById('export-all-btn');
        if (generateBtn) generateBtn.disabled = true;
        if (exportBtn) exportBtn.disabled = true;
        
        // Show loading
        showLoadingOverlay('Starting content generation...');
        
        // Create article cards
        this.createArticleCards();
        
        console.log('🚀 Starting content generation loop...');
        
        // Generate each article
        for (let i = 0; i < this.topics.length; i++) {
            if (!this.isGenerating) break;
            
            const topic = this.topics[i];
            console.log(`📄 Generating article ${i + 1}/${this.topics.length}: "${topic.title}"`);
            
            await this.generateArticle(topic, i);
            this.currentGenerations++;
            
            updateGlobalProgress(
                this.currentGenerations, 
                this.totalGenerations,
                `Generated ${this.currentGenerations} of ${this.totalGenerations} articles`
            );
            
            // Delay between requests
            if (i < this.topics.length - 1) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
        }
        
        // Cleanup
        this.isGenerating = false;
        hideLoadingOverlay();
        if (generateBtn) generateBtn.disabled = false;
        if (exportBtn) exportBtn.disabled = false;
        
        const successMsg = `Successfully generated ${this.currentGenerations} articles!`;
        console.log('✅', successMsg);
        showAlert(successMsg, 'success');
    }
    
    collectTopics() {
        const topics = [];
        const container = document.getElementById('topics-container');
        if (!container) return topics;
        
        const rows = container.children;
        Array.from(rows).forEach((row, index) => {
            const titleInput = row.querySelector('.topic-input');
            const wordCountInput = row.querySelector('.word-count-input');
            
            if (titleInput && wordCountInput) {
                topics.push({
                    id: index,
                    title: titleInput.value,
                    wordCount: parseInt(wordCountInput.value) || 1000,
                    element: row
                });
            }
        });
        
        return topics;
    }
    
    createArticleCards() {
        const container = document.getElementById('results-container');
        if (!container) return;
        
        container.innerHTML = '';
        
        this.topics.forEach(topic => {
            const card = document.createElement('div');
            card.className = 'article-card';
            card.id = `article-${topic.id}`;
            card.innerHTML = `
                <div class="article-header">
                    <div class="article-title">${this.escapeHtml(topic.title) || 'Untitled'}</div>
                    <div class="article-meta">
                        <span class="word-count">Target: ${formatWordCount(topic.wordCount)} words</span>
                        <span class="status status-pending">Pending</span>
                    </div>
                </div>
                <div class="article-content">
                    <div class="article-progress">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: 0%"></div>
                        </div>
                    </div>
                    <div class="generated-content" id="content-${topic.id}">
                        <em>Content generation pending...</em>
                    </div>
                    <div class="article-actions">
                        <button class="btn btn-outline btn-sm copy-btn" data-article="${topic.id}">
                            <i class="mdi mdi-content-copy"></i> Copy
                        </button>
                        <button class="btn btn-outline btn-sm export-btn" data-article="${topic.id}">
                            <i class="mdi mdi-download"></i> Export as .doc
                        </button>
                        <button class="btn btn-info btn-sm rewrite-btn" data-article="${topic.id}">
                            <i class="mdi mdi-refresh"></i> Re-Write
                        </button>
                        <button class="btn btn-success btn-sm retry-btn" data-article="${topic.id}" style="display: none;">
                            <i class="mdi mdi-reload"></i> Retry
                        </button>
                    </div>
                </div>
            `;
            container.appendChild(card);
        });
        
        this.initializeArticleActions();
    }
    
    initializeArticleActions() {
        // Copy buttons
        document.querySelectorAll('.copy-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.copy-btn').getAttribute('data-article');
                this.copyArticleContent(articleId);
            });
        });
        
        // Export buttons
        document.querySelectorAll('.export-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.export-btn').getAttribute('data-article');
                this.exportSingleArticle(articleId);
            });
        });
        
        // Rewrite buttons
        document.querySelectorAll('.rewrite-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.rewrite-btn').getAttribute('data-article');
                this.rewriteArticle(articleId);
            });
        });
        
        // Retry buttons
        document.querySelectorAll('.retry-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const articleId = e.target.closest('.retry-btn').getAttribute('data-article');
                this.retryArticle(articleId);
            });
        });
    }
    
    async generateArticle(topic, index) {
        console.log(`🔄 Generating article for: "${topic.title}"`);
        
        const contentElement = document.getElementById(`content-${topic.id}`);
        if (!contentElement) {
            console.error(`❌ Content element not found for topic ${topic.id}`);
            return;
        }
        
        const articleCard = contentElement.closest('.article-card');
        const progressFill = articleCard ? articleCard.querySelector('.progress-fill') : null;
        const statusElement = articleCard ? articleCard.querySelector('.status') : null;
        const retryBtn = articleCard ? articleCard.querySelector('.retry-btn') : null;
        const rewriteBtn = articleCard ? articleCard.querySelector('.rewrite-btn') : null;
        
        try {
            if (statusElement) {
                statusElement.textContent = 'Generating';
                statusElement.className = 'status status-generating';
            }
            if (retryBtn) retryBtn.style.display = 'none';
            if (rewriteBtn) rewriteBtn.style.display = 'none';
            
            const params = this.getGenerationParameters(topic);
            console.log('📋 Generation params:', params);
            
            const content = await window.aiGenerateContent(params, (chunk, progress) => {
                if (contentElement) {
                    contentElement.innerHTML = this.formatContentForDisplay(chunk);
                }
                if (progressFill && progress) {
                    progressFill.style.width = `${progress}%`;
                }
            });
            
            console.log(`✅ Article generated: "${topic.title}"`);
            
            // Process internal links
            const processedContent = this.processInternalLinks(content, index);
            
            this.generatedArticles[index] = {
                id: topic.id,
                title: topic.title,
                content: processedContent,
                displayContent: this.formatContentForDisplay(processedContent),
                wordCount: this.countWords(processedContent),
                originalContent: content
            };
            
            if (contentElement) {
                contentElement.innerHTML = this.generatedArticles[index].displayContent;
            }
            
            if (statusElement) {
                statusElement.textContent = 'Completed';
                statusElement.className = 'status status-completed';
            }
            if (rewriteBtn) rewriteBtn.style.display = 'flex';
            
            const wordCountElement = articleCard?.querySelector('.word-count');
            if (wordCountElement) {
                wordCountElement.textContent = `Generated: ${formatWordCount(this.generatedArticles[index].wordCount)} words`;
            }
            
            const completedCount = document.getElementById('completed-count');
            if (completedCount) {
                completedCount.textContent = this.currentGenerations.toString();
            }
            
        } catch (error) {
            console.error(`❌ Failed to generate article "${topic.title}":`, error);
            if (statusElement) {
                statusElement.textContent = 'Failed';
                statusElement.className = 'status status-failed';
            }
            if (contentElement) {
                contentElement.innerHTML = `
                    <div style="color: var(--error-color); padding: 20px; text-align: center;">
                        <i class="mdi mdi-alert-circle" style="font-size: 2em; margin-bottom: 10px;"></i>
                        <h3>Generation Failed</h3>
                        <p>${this.escapeHtml(error.message)}</p>
                    </div>
                `;
            }
            if (retryBtn) retryBtn.style.display = 'flex';
            if (rewriteBtn) rewriteBtn.style.display = 'none';
        }
    }
    
    async rewriteArticle(articleId) {
        console.log(`🔄 Rewriting article: ${articleId}`);
        
        const article = this.generatedArticles.find(a => a && a.id == articleId);
        if (!article) {
            showAlert('Article not found for rewriting');
            return;
        }
        
        const topic = this.topics.find(t => t.id == articleId);
        if (!topic) {
            showAlert('Topic not found for rewriting');
            return;
        }
        
        const contentElement = document.getElementById(`content-${articleId}`);
        const articleCard = contentElement?.closest('.article-card');
        const statusElement = articleCard?.querySelector('.status');
        const progressFill = articleCard?.querySelector('.progress-fill');
        const rewriteBtn = articleCard?.querySelector('.rewrite-btn');
        const retryBtn = articleCard?.querySelector('.retry-btn');
        
        if (statusElement) {
            statusElement.textContent = 'Rewriting';
            statusElement.className = 'status status-generating';
        }
        if (rewriteBtn) rewriteBtn.disabled = true;
        if (retryBtn) retryBtn.style.display = 'none';
        if (progressFill) progressFill.style.width = '0%';
        
        // Add regenerating animation
        if (contentElement) {
            contentElement.classList.add('regenerating');
        }
        
        try {
            const params = this.getGenerationParameters(topic);
            // Add rewrite instruction to prompt
            params.customPrompt = params.customPrompt || this.getDefaultRewritePrompt(article.title);
            
            const newContent = await window.aiGenerateContent(params, (chunk, progress) => {
                if (contentElement) {
                    contentElement.innerHTML = this.formatContentForDisplay(chunk);
                }
                if (progressFill && progress) {
                    progressFill.style.width = `${progress}%`;
                }
            });
            
            // Process internal links
            const processedContent = this.processInternalLinks(newContent, articleId);
            
            // Update article data
            article.content = processedContent;
            article.displayContent = this.formatContentForDisplay(processedContent);
            article.wordCount = this.countWords(processedContent);
            article.originalContent = newContent;
            
            if (contentElement) {
                contentElement.innerHTML = article.displayContent;
                contentElement.classList.remove('regenerating');
            }
            
            if (statusElement) {
                statusElement.textContent = 'Rewritten';
                statusElement.className = 'status status-completed';
            }
            if (rewriteBtn) rewriteBtn.disabled = false;
            
            showAlert('Article rewritten successfully!', 'success');
            
        } catch (error) {
            console.error(`❌ Failed to rewrite article:`, error);
            if (statusElement) {
                statusElement.textContent = 'Failed';
                statusElement.className = 'status status-failed';
            }
            if (contentElement) {
                contentElement.innerHTML = `
                    <div style="color: var(--error-color); padding: 20px; text-align: center;">
                        <i class="mdi mdi-alert-circle" style="font-size: 2em; margin-bottom: 10px;"></i>
                        <h3>Rewrite Failed</h3>
                        <p>${this.escapeHtml(error.message)}</p>
                    </div>
                `;
                contentElement.classList.remove('regenerating');
            }
            if (rewriteBtn) rewriteBtn.disabled = false;
            if (retryBtn) retryBtn.style.display = 'flex';
        }
    }
    
    async retryArticle(articleId) {
        console.log(`🔄 Retrying article: ${articleId}`);
        
        const topic = this.topics.find(t => t.id == articleId);
        if (!topic) {
            showAlert('Topic not found for retry');
            return;
        }
        
        const index = this.topics.findIndex(t => t.id == articleId);
        await this.generateArticle(topic, index);
    }
    
    getDefaultRewritePrompt(title) {
        return `Please rewrite the following article with a fresh perspective while maintaining the core information and SEO optimization. Add new insights and improve the flow.

Original Topic: ${title}

Requirements:
- Maintain all key information and SEO elements
- Add at least one new unique insight or example
- Improve readability and flow
- Keep the same professional tone
- Ensure all internal linking opportunities are utilized`;
    }
    
    processInternalLinks(content, currentIndex) {
        let processedContent = content;
        
        // Create internal links for other topics
        this.topics.forEach((topic, index) => {
            if (index !== currentIndex) {
                const slug = this.createSlug(topic.title);
                const regex = new RegExp(`\\b${this.escapeRegExp(topic.title)}\\b`, 'gi');
                processedContent = processedContent.replace(regex, 
                    `<a href="#article-${topic.id}" class="internal-link" data-slug="${slug}">${topic.title}</a>`);
            }
        });
        
        return processedContent;
    }
    
    createSlug(title) {
        return title
            .toLowerCase()
            .replace(/[^a-z0-9]+/g, '-')
            .replace(/(^-|-$)+/g, '');
    }
    
    escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }
    
    formatContentForDisplay(text) {
        if (!text) return '';
        
        return text
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/\n/g, '<br>')
            .replace(/# (.*?)(?=\n|$)/g, '<h1>$1</h1>')
            .replace(/## (.*?)(?=\n|$)/g, '<h2>$1</h2>')
            .replace(/### (.*?)(?=\n|$)/g, '<h3>$1</h3>')
            .replace(/- (.*?)(?=\n|$)/g, '<li>$1</li>')
            .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>')
            .replace(/`(.*?)`/g, '<code>$1</code>')
            .replace(/```([\s\S]*?)```/g, '<pre><code>$1</code></pre>')
            .replace(/^>(.*)$/gm, '<blockquote>$1</blockquote>');
    }
    
    formatContentForExport(text) {
        if (!text) return '';
        
        // Remove HTML tags for clean export but keep basic formatting
        let cleanText = text
            .replace(/<strong>(.*?)<\/strong>/g, '**$1**')
            .replace(/<em>(.*?)<\/em>/g, '*$1*')
            .replace(/<h1>(.*?)<\/h1>/g, '\n# $1\n')
            .replace(/<h2>(.*?)<\/h2>/g, '\n## $1\n')
            .replace(/<h3>(.*?)<\/h3>/g, '\n### $1\n')
            .replace(/<ul>\s*<li>(.*?)<\/li>\s*<\/ul>/gs, '\n• $1\n')
            .replace(/<li>(.*?)<\/li>/g, '\n• $1')
            .replace(/<code>(.*?)<\/code>/g, '`$1`')
            .replace(/<pre><code>([\s\S]*?)<\/code><\/pre>/g, '\n```\n$1\n```\n')
            .replace(/<blockquote>(.*?)<\/blockquote>/g, '\n> $1\n')
            .replace(/<br\s*\/?>/g, '\n')
            .replace(/<[^>]*>/g, '')
            .replace(/\n{3,}/g, '\n\n')
            .trim();
        
        return cleanText;
    }
    
    getGenerationParameters(topic) {
        const getValue = (id, defaultValue = '') => {
            const element = document.getElementById(id);
            return element ? element.value : defaultValue;
        };
        
        const customPrompt = getValue('custom-prompt', '');
        const contentType = getValue('content-type', 'Articles');
        const tone = getValue('tone-voice', 'Professional');
        const persona = getValue('ai-persona', 'Lily SEO');
        const focusKeyword = getValue('focus-keyword', '');
        const otherKeywords = getValue('other-keywords', '');
        const targetAudience = getValue('target-audience', '');
        
        const addins = Array.from(document.querySelectorAll('input[name="addins"]:checked'))
            .map(checkbox => checkbox.value);
        
        return {
            topic: topic.title,
            wordCount: topic.wordCount,
            customPrompt: customPrompt,
            contentType: contentType,
            tone: tone,
            persona: persona,
            focusKeyword: focusKeyword,
            otherKeywords: otherKeywords,
            targetAudience: targetAudience,
            addins: addins
        };
    }
    
    countWords(text) {
        return text ? text.trim().split(/\s+/).length : 0;
    }
    
    escapeHtml(unsafe) {
        if (!unsafe) return '';
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
    
    copyArticleContent(articleId) {
        const article = this.generatedArticles.find(a => a && a.id == articleId);
        if (article && article.content) {
            // Use the formatted content for copy
            const textToCopy = this.formatContentForExport(article.content);
            navigator.clipboard.writeText(textToCopy)
                .then(() => showAlert('Content copied to clipboard with perfect formatting!', 'success'))
                .catch(() => showAlert('Failed to copy content'));
        } else {
            showAlert('No content available to copy');
        }
    }
    
    exportSingleArticle(articleId) {
        const article = this.generatedArticles.find(a => a && a.id == articleId);
        if (article && article.content) {
            this.exportAsDocWithFormatting(article.content, article.title);
        } else {
            showAlert('No content available to export');
        }
    }
    
    exportAllArticles() {
        const validArticles = this.generatedArticles.filter(a => a && a.content);
        if (validArticles.length === 0) {
            showAlert('No articles generated yet');
            return;
        }
        
        validArticles.forEach(article => {
            this.exportAsDocWithFormatting(article.content, article.title);
        });
        
        showAlert(`Exported ${validArticles.length} articles with perfect formatting!`, 'success');
    }
    
    exportAsDocWithFormatting(content, filename) {
        const formattedContent = this.formatContentForExport(content);
        
        const htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>${filename}</title>
            <style>
                body { 
                    font-family: 'Calibri', 'Inter', sans-serif; 
                    line-height: 1.6; 
                    margin: 40px; 
                    color: #1e293b;
                    max-width: 800px;
                    margin-left: auto;
                    margin-right: auto;
                }
                h1 { 
                    font-size: 2.5rem;
                    font-weight: 700;
                    color: #1e293b;
                    margin: 1.5em 0 1em 0;
                    padding-bottom: 0.5em;
                    border-bottom: 3px solid #6366f1;
                    line-height: 1.3;
                }
                h2 { 
                    font-size: 2rem;
                    font-weight: 600;
                    color: #6366f1;
                    margin: 1.5em 0 1em 0;
                    padding-left: 0.5em;
                    border-left: 4px solid #6366f1;
                    line-height: 1.4;
                }
                h3 { 
                    font-size: 1.5rem;
                    font-weight: 600;
                    color: #1e293b;
                    margin: 1.5em 0 1em 0;
                    line-height: 1.4;
                }
                p { 
                    margin-bottom: 1.2em;
                    text-align: justify;
                    font-size: 16px;
                }
                strong { 
                    font-weight: 700;
                    color: #1e293b;
                }
                em { 
                    font-style: italic;
                    color: #64748b;
                }
                ul, ol { 
                    margin: 1.2em 0;
                    padding-left: 2em;
                }
                li { 
                    margin-bottom: 0.6em;
                }
                ul li::before {
                    content: '•';
                    color: #6366f1;
                    font-weight: bold;
                    position: absolute;
                    left: -1em;
                }
                ol {
                    counter-reset: list-counter;
                }
                ol li {
                    counter-increment: list-counter;
                    position: relative;
                }
                ol li::before {
                    content: counter(list-counter) '.';
                    color: #6366f1;
                    font-weight: bold;
                    position: absolute;
                    left: -1em;
                }
                blockquote {
                    border-left: 4px solid #6366f1;
                    background: #f8fafc;
                    padding: 1.5em;
                    margin: 1.5em 0;
                    border-radius: 0 8px 8px 0;
                    font-style: italic;
                }
                code {
                    background: #e2e8f0;
                    padding: 0.2em 0.4em;
                    border-radius: 4px;
                    font-family: 'Courier New', monospace;
                    font-size: 0.9em;
                }
                pre {
                    background: #e2e8f0;
                    padding: 1.5em;
                    border-radius: 8px;
                    overflow-x: auto;
                    margin: 1.5em 0;
                    border: 1px solid #cbd5e1;
                }
                pre code {
                    background: none;
                    padding: 0;
                }
                .internal-link {
                    color: #059669;
                    font-weight: 600;
                    text-decoration: none;
                    background: #ecfdf5;
                    padding: 0.1em 0.3em;
                    border-radius: 4px;
                    border: 1px solid #059669;
                }
            </style>
        </head>
        <body>
            <div class="export-formatting">
                ${this.formatContentForDisplay(content)}
            </div>
        </body>
        </html>
        `;
        
        const blob = new Blob([htmlContent], { type: 'application/msword' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `${filename.replace(/[^a-z0-9]/gi, '_')}.doc`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);
    }
    
    async getApiKey() {
        try {
            const response = await fetch('get_api_key.php');
            if (!response.ok) throw new Error(`HTTP error! status: ${response.status}`);
            const result = await response.json();
            return result.api_key || null;
        } catch (error) {
            console.error('Error fetching API key:', error);
            return null;
        }
    }
}

// Initialize when DOM is ready
console.log('🌍 Setting up BulkWriter...');

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        console.log('📄 DOM fully loaded, creating BulkWriter...');
        window.bulkWriter = new BulkWriter();
    });
} else {
    console.log('📄 DOM already ready, creating BulkWriter...');
    window.bulkWriter = new BulkWriter();
}

console.log('✅ BULKWRITER.JS LOADED COMPLETELY');