// Global Variables
let currentTheme = localStorage.getItem('theme') || 'light';

// Initialize application
document.addEventListener('DOMContentLoaded', function() {
    initializeTheme();
    initializeAuthTabs();
    initializeModals();
    initializeUserMenu();
    initializeAPISettings();
});

// Theme Management
function initializeTheme() {
    const themeSwitch = document.getElementById('theme-switch');
    if (themeSwitch) {
        themeSwitch.checked = currentTheme === 'dark';
        setTheme(currentTheme);
        
        themeSwitch.addEventListener('change', function() {
            const newTheme = this.checked ? 'dark' : 'light';
            setTheme(newTheme);
        });
    }
}

function setTheme(theme) {
    currentTheme = theme;
    document.documentElement.setAttribute('data-theme', theme);
    localStorage.setItem('theme', theme);
    
    // Update theme switch if it exists
    const themeSwitch = document.getElementById('theme-switch');
    if (themeSwitch) {
        themeSwitch.checked = theme === 'dark';
    }
}

// Auth Tabs
function initializeAuthTabs() {
    const tabButtons = document.querySelectorAll('.tab-button');
    const tabContents = document.querySelectorAll('.tab-content');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            // Update active tab button
            tabButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Show target tab content
            tabContents.forEach(content => {
                content.classList.remove('active');
                if (content.id === `${targetTab}-tab`) {
                    content.classList.add('active');
                }
            });
        });
    });
}

// Modal Management
function initializeModals() {
    // API Settings Modal
    const apiSettingsBtn = document.getElementById('api-settings-btn');
    const apiSettingsModal = document.getElementById('api-settings-modal');
    const closeApiModal = document.getElementById('close-api-modal');
    const modalClose = document.querySelector('.modal-close');
    
    if (apiSettingsBtn && apiSettingsModal) {
        apiSettingsBtn.addEventListener('click', function(e) {
            e.preventDefault();
            apiSettingsModal.classList.add('show');
        });
        
        const closeModal = () => apiSettingsModal.classList.remove('show');
        
        if (closeApiModal) closeApiModal.addEventListener('click', closeModal);
        if (modalClose) modalClose.addEventListener('click', closeModal);
        
        apiSettingsModal.addEventListener('click', function(e) {
            if (e.target === apiSettingsModal) {
                closeModal();
            }
        });
    }
}

// User Menu
function initializeUserMenu() {
    const userBtn = document.querySelector('.user-btn');
    const userDropdown = document.querySelector('.user-dropdown');
    
    if (userBtn && userDropdown) {
        userBtn.addEventListener('click', function() {
            const isVisible = userDropdown.style.display === 'block';
            userDropdown.style.display = isVisible ? 'none' : 'block';
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', function(e) {
            if (!userBtn.contains(e.target) && !userDropdown.contains(e.target)) {
                userDropdown.style.display = 'none';
            }
        });
    }
}

// API Settings Management
function initializeAPISettings() {
    const saveApiKeyBtn = document.getElementById('save-api-key');
    if (saveApiKeyBtn) {
        saveApiKeyBtn.addEventListener('click', saveApiKey);
    }
}

async function saveApiKey() {
    const apiKeyInput = document.getElementById('api-key');
    const partialKey = apiKeyInput.value.trim();
    
    if (!partialKey) {
        showAlert('Please enter your API key');
        return;
    }
    
    // Validate API key format (basic validation)
    if (partialKey.length < 10) {
        showAlert('Please enter a valid API key');
        return;
    }
    
    const fullKey = 'AIzaSy' + partialKey;
    
    try {
        showLoadingOverlay('Saving API key...');
        
        const response = await fetch('save_api_key.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                api_key: fullKey
            })
        });
        
        const result = await response.json();
        
        hideLoadingOverlay();
        
        if (result.success) {
            showAlert('AI Access key saved successfully!', 'success');
            document.getElementById('api-settings-modal').classList.remove('show');
            
            // Clear the input for security
            apiKeyInput.value = '';
        } else {
            showAlert('Failed to save API key: ' + (result.message || 'Unknown error'));
        }
    } catch (error) {
        hideLoadingOverlay();
        showAlert('Error saving API key: ' + error.message);
    }
}

// Utility Functions
function showAlert(message, type = 'error') {
    // Remove any existing alerts first
    const existingAlerts = document.querySelectorAll('.alert');
    existingAlerts.forEach(alert => alert.remove());
    
    const alert = document.createElement('div');
    alert.className = `alert alert-${type}`;
    alert.innerHTML = `
        <i class="mdi mdi-${type === 'error' ? 'alert-circle' : type === 'success' ? 'check-circle' : 'information'}"></i>
        ${message}
    `;
    
    // Add styles if not already present
    if (!document.querySelector('#alert-styles')) {
        const style = document.createElement('style');
        style.id = 'alert-styles';
        style.textContent = `
            .alert {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 10000;
                padding: 16px 20px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                display: flex;
                align-items: center;
                gap: 12px;
                max-width: 400px;
                animation: slideInRight 0.3s ease-out;
            }
            
            .alert-error {
                background: var(--error-color);
                color: white;
                border: 1px solid rgba(239, 68, 68, 0.2);
            }
            
            .alert-success {
                background: var(--success-color);
                color: white;
                border: 1px solid rgba(16, 185, 129, 0.2);
            }
            
            .alert-info {
                background: var(--info-color);
                color: white;
                border: 1px solid rgba(59, 130, 246, 0.2);
            }
            
            @keyframes slideInRight {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(style);
    }
    
    document.body.appendChild(alert);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (alert.parentNode) {
            alert.style.animation = 'slideOutRight 0.3s ease-in';
            setTimeout(() => alert.remove(), 300);
        }
    }, 5000);
}

function formatWordCount(count) {
    return count.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Enhanced File Export Functions
function exportAsDoc(content, filename) {
    const htmlContent = createExportHTML(content, filename);
    
    const blob = new Blob([htmlContent], { type: 'application/msword' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `${filename.replace(/[^a-z0-9]/gi, '_')}.doc`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

function createExportHTML(content, title) {
    return `<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>${title}</title>
    <style>
        body { 
            font-family: 'Calibri', 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            line-height: 1.7; 
            margin: 40px; 
            color: #1e293b;
            max-width: 800px;
            margin-left: auto;
            margin-right: auto;
            background: #ffffff;
        }
        
        .export-content {
            background: white;
            padding: 40px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        
        h1 { 
            font-size: 2.2rem;
            font-weight: 700;
            color: #1e293b;
            margin: 1.5em 0 1em 0;
            padding-bottom: 0.5em;
            border-bottom: 3px solid #6366f1;
            line-height: 1.3;
            text-align: center;
        }
        
        h2 { 
            font-size: 1.8rem;
            font-weight: 600;
            color: #6366f1;
            margin: 1.5em 0 1em 0;
            padding-left: 0.5em;
            border-left: 4px solid #6366f1;
            line-height: 1.4;
        }
        
        h3 { 
            font-size: 1.4rem;
            font-weight: 600;
            color: #1e293b;
            margin: 1.5em 0 0.8em 0;
            line-height: 1.4;
        }
        
        p { 
            margin-bottom: 1.2em;
            text-align: justify;
            font-size: 16px;
            color: #374151;
        }
        
        strong { 
            font-weight: 700;
            color: #1e293b;
            background: linear-gradient(120deg, #6366f1 0%, #6366f1 100%);
            background-repeat: no-repeat;
            background-size: 100% 0.2em;
            background-position: 0 88%;
        }
        
        em { 
            font-style: italic;
            color: #64748b;
        }
        
        ul, ol { 
            margin: 1.2em 0;
            padding-left: 2em;
        }
        
        li { 
            margin-bottom: 0.6em;
            position: relative;
            color: #374151;
        }
        
        ul li::before {
            content: '•';
            color: #6366f1;
            font-weight: bold;
            position: absolute;
            left: -1em;
        }
        
        ol {
            counter-reset: list-counter;
        }
        
        ol li {
            counter-increment: list-counter;
        }
        
        ol li::before {
            content: counter(list-counter) '.';
            color: #6366f1;
            font-weight: bold;
            position: absolute;
            left: -1em;
        }
        
        blockquote {
            border-left: 4px solid #6366f1;
            background: #f8fafc;
            padding: 1.5em;
            margin: 1.5em 0;
            border-radius: 0 8px 8px 0;
            font-style: italic;
            color: #64748b;
        }
        
        blockquote p {
            margin-bottom: 0;
            color: #64748b;
        }
        
        code {
            background: #e2e8f0;
            padding: 0.2em 0.4em;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 0.9em;
            color: #1e293b;
        }
        
        pre {
            background: #e2e8f0;
            padding: 1.5em;
            border-radius: 8px;
            overflow-x: auto;
            margin: 1.5em 0;
            border: 1px solid #cbd5e1;
        }
        
        pre code {
            background: none;
            padding: 0;
        }
        
        .internal-link {
            color: #059669;
            font-weight: 600;
            text-decoration: none;
            background: #ecfdf5;
            padding: 0.1em 0.3em;
            border-radius: 4px;
            border: 1px solid #059669;
        }
        
        .article-meta {
            text-align: center;
            color: #64748b;
            font-size: 0.9em;
            margin-bottom: 2em;
            padding-bottom: 1em;
            border-bottom: 1px solid #e2e8f0;
        }
        
        @media print {
            body {
                margin: 0;
                padding: 20px;
                background: white;
            }
            
            .export-content {
                box-shadow: none;
                padding: 0;
            }
            
            h1 { color: #000; }
            h2 { color: #333; }
            .internal-link { color: #059669; text-decoration: underline; }
        }
    </style>
</head>
<body>
    <div class="export-content">
        <h1>${title}</h1>
        <div class="article-meta">
            Generated with Bulk AI Writer | ${new Date().toLocaleDateString()}
        </div>
        <div class="generated-content">
            ${content}
        </div>
    </div>
</body>
</html>`;
}

function exportAsDocWithPerfectFormatting(content, filename) {
    exportAsDoc(content, filename);
}

function downloadAllArticles(articles) {
    if (!articles || articles.length === 0) {
        showAlert('No articles available to export');
        return;
    }
    
    showLoadingOverlay(`Exporting ${articles.length} articles...`);
    
    // Export each article with a small delay to avoid browser issues
    let exportedCount = 0;
    
    const exportNext = () => {
        if (exportedCount < articles.length) {
            const article = articles[exportedCount];
            if (article.content && article.title) {
                exportAsDoc(article.content, article.title);
            }
            exportedCount++;
            
            if (exportedCount < articles.length) {
                setTimeout(exportNext, 500);
            } else {
                setTimeout(() => {
                    hideLoadingOverlay();
                    showAlert(`Successfully exported ${articles.length} articles!`, 'success');
                }, 1000);
            }
        }
    };
    
    exportNext();
}

// Enhanced Copy Function
function copyWithPerfectFormatting(content) {
    if (!content) {
        showAlert('No content available to copy');
        return;
    }
    
    // Create a temporary div to parse HTML and extract clean text
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = content;
    
    // Process internal links to remove HTML but keep the text
    const internalLinks = tempDiv.querySelectorAll('.internal-link');
    internalLinks.forEach(link => {
        // Replace with just the text content
        const textNode = document.createTextNode(link.textContent);
        link.parentNode.replaceChild(textNode, link);
    });
    
    // Get clean text with proper formatting
    let cleanText = tempDiv.innerHTML
        .replace(/<h1[^>]*>(.*?)<\/h1>/gi, '\n\n# $1\n\n')
        .replace(/<h2[^>]*>(.*?)<\/h2>/gi, '\n\n## $1\n\n')
        .replace(/<h3[^>]*>(.*?)<\/h3>/gi, '\n\n### $1\n\n')
        .replace(/<p[^>]*>(.*?)<\/p>/gi, '$1\n\n')
        .replace(/<br\s*\/?>/gi, '\n')
        .replace(/<strong>(.*?)<\/strong>/gi, '**$1**')
        .replace(/<em>(.*?)<\/em>/gi, '*$1*')
        .replace(/<code>(.*?)<\/code>/gi, '`$1`')
        .replace(/<pre>(.*?)<\/pre>/gis, '\n```\n$1\n```\n')
        .replace(/<blockquote>(.*?)<\/blockquote>/gi, '\n> $1\n\n')
        .replace(/<ul>(.*?)<\/ul>/gis, (match, p1) => {
            const items = p1.replace(/<li>(.*?)<\/li>/gi, '• $1\n');
            return '\n' + items + '\n';
        })
        .replace(/<ol>(.*?)<\/ol>/gis, (match, p1) => {
            let counter = 1;
            const items = p1.replace(/<li>(.*?)<\/li>/gi, () => `${counter++}. $1\n`);
            return '\n' + items + '\n';
        })
        .replace(/<[^>]*>/g, '') // Remove any remaining HTML tags
        .replace(/\n{3,}/g, '\n\n') // Replace multiple newlines with double newlines
        .replace(/^\s+|\s+$/g, '') // Trim whitespace
        .trim();
    
    // Add header with title and date
    const titleMatch = content.match(/<h1[^>]*>(.*?)<\/h1>/i);
    const title = titleMatch ? titleMatch[1] : 'Generated Content';
    const header = `# ${title}\n\nGenerated on ${new Date().toLocaleDateString()}\n\n`;
    
    cleanText = header + cleanText;
    
    // Copy to clipboard
    navigator.clipboard.writeText(cleanText)
        .then(() => showAlert('Content copied to clipboard with perfect formatting!', 'success'))
        .catch((err) => {
            console.error('Failed to copy content:', err);
            showAlert('Failed to copy content. Please try again.');
        });
}

// Loading Management
function showLoadingOverlay(message = 'Processing...') {
    let overlay = document.getElementById('loading-overlay');
    
    if (!overlay) {
        overlay = document.createElement('div');
        overlay.id = 'loading-overlay';
        overlay.className = 'loading-overlay';
        overlay.innerHTML = `
            <div class="loading-content">
                <div class="loading-spinner"></div>
                <h3>${message}</h3>
                <p>Please wait while we process your request...</p>
                <div class="progress-bar">
                    <div class="progress-fill" id="global-progress"></div>
                </div>
                <div class="progress-text" id="progress-text">Starting...</div>
            </div>
        `;
        document.body.appendChild(overlay);
    } else {
        const loadingContent = overlay.querySelector('.loading-content h3');
        if (loadingContent) {
            loadingContent.textContent = message;
        }
    }
    
    overlay.classList.add('show');
}

function hideLoadingOverlay() {
    const overlay = document.getElementById('loading-overlay');
    if (overlay) {
        overlay.classList.remove('show');
        
        // Remove after animation completes
        setTimeout(() => {
            if (overlay.parentNode && !overlay.classList.contains('show')) {
                overlay.remove();
            }
        }, 300);
    }
}

function updateGlobalProgress(completed, total, message) {
    const progressFill = document.getElementById('global-progress');
    const progressText = document.getElementById('progress-text');
    
    if (progressFill) {
        const percentage = total > 0 ? (completed / total) * 100 : 0;
        progressFill.style.width = `${percentage}%`;
    }
    
    if (progressText && message) {
        progressText.textContent = message;
    }
}

// Formatting Utilities
function formatContentForDisplay(content) {
    if (!content) return '';
    
    return content
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\*(.*?)\*/g, '<em>$1</em>')
        .replace(/\n/g, '<br>')
        .replace(/# (.*?)(?=\n|$)/g, '<h1>$1</h1>')
        .replace(/## (.*?)(?=\n|$)/g, '<h2>$1</h2>')
        .replace(/### (.*?)(?=\n|$)/g, '<h3>$1</h3>')
        .replace(/- (.*?)(?=\n|$)/g, '<li>$1</li>')
        .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>')
        .replace(/`(.*?)`/g, '<code>$1</code>')
        .replace(/```([\s\S]*?)```/g, '<pre><code>$1</code></pre>')
        .replace(/^>(.*)$/gm, '<blockquote>$1</blockquote>');
}

function escapeHtml(unsafe) {
    if (!unsafe) return '';
    return unsafe
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#039;");
}

// API Key Management
async function getApiKey() {
    try {
        const response = await fetch('get_api_key.php');
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const result = await response.json();
        return result.api_key || null;
    } catch (error) {
        console.error('Error fetching API key:', error);
        return null;
    }
}

// Content Processing
function processInternalLinks(content, topics, currentIndex) {
    if (!content || !topics) return content;
    
    let processedContent = content;
    
    topics.forEach((topic, index) => {
        if (index !== currentIndex && topic.title) {
            const slug = createSlug(topic.title);
            const regex = new RegExp(`\\b${escapeRegExp(topic.title)}\\b`, 'gi');
            processedContent = processedContent.replace(regex, 
                `<a href="#article-${topic.id}" class="internal-link" data-slug="${slug}">${topic.title}</a>`);
        }
    });
    
    return processedContent;
}

function createSlug(title) {
    return title
        .toLowerCase()
        .replace(/[^a-z0-9]+/g, '-')
        .replace(/(^-|-$)+/g, '');
}

function escapeRegExp(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

// Error Handling
function handleApiError(error) {
    console.error('API Error:', error);
    
    let userMessage = 'An unexpected error occurred. Please try again.';
    
    if (error.message.includes('API key')) {
        userMessage = 'Invalid or missing API key. Please check your API settings.';
    } else if (error.message.includes('network') || error.message.includes('fetch')) {
        userMessage = 'Network error. Please check your internet connection.';
    } else if (error.message.includes('quota')) {
        userMessage = 'API quota exceeded. Please try again later or check your API limits.';
    } else if (error.message.includes('rate limit')) {
        userMessage = 'Rate limit exceeded. Please wait a moment and try again.';
    }
    
    showAlert(userMessage);
    return userMessage;
}

// Local Storage Utilities
function saveToLocalStorage(key, data) {
    try {
        localStorage.setItem(key, JSON.stringify(data));
        return true;
    } catch (error) {
        console.error('Error saving to localStorage:', error);
        return false;
    }
}

function loadFromLocalStorage(key) {
    try {
        const data = localStorage.getItem(key);
        return data ? JSON.parse(data) : null;
    } catch (error) {
        console.error('Error loading from localStorage:', error);
        return null;
    }
}

// Validation Functions
function validateEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function validatePassword(password) {
    return password && password.length >= 6;
}

function validateApiKey(apiKey) {
    return apiKey && apiKey.length >= 10;
}

// DOM Utilities
function createElement(tag, classes = '', content = '') {
    const element = document.createElement(tag);
    if (classes) element.className = classes;
    if (content) element.innerHTML = content;
    return element;
}

function removeAllChildren(element) {
    while (element.firstChild) {
        element.removeChild(element.firstChild);
    }
}

// Event Delegation Helper
function addDelegateEvent(container, selector, event, handler) {
    if (!container) return;
    
    container.addEventListener(event, function(e) {
        const target = e.target.closest(selector);
        if (target && container.contains(target)) {
            handler.call(target, e);
        }
    });
}

// Performance Monitoring
function measurePerformance(name, fn) {
    const start = performance.now();
    const result = fn();
    const end = performance.now();
    console.log(`${name} took ${(end - start).toFixed(2)}ms`);
    return result;
}

// Export all functions for global access
window.scriptUtils = {
    showAlert,
    hideLoadingOverlay,
    showLoadingOverlay,
    updateGlobalProgress,
    exportAsDoc,
    exportAsDocWithPerfectFormatting,
    copyWithPerfectFormatting,
    downloadAllArticles,
    formatWordCount,
    debounce,
    escapeHtml,
    formatContentForDisplay,
    processInternalLinks,
    createSlug,
    handleApiError,
    validateEmail,
    validatePassword,
    validateApiKey,
    createElement,
    removeAllChildren,
    addDelegateEvent,
    measurePerformance
};

console.log('✅ script.js loaded completely with all utilities');