// Settings Page JavaScript
class SettingsManager {
    constructor() {
        this.init();
    }
    
    init() {
        this.initializeEventListeners();
        this.initializeModals();
        console.log('SettingsManager initialized');
    }
    
    initializeEventListeners() {
        // Delete account button
        const deleteAccountBtn = document.getElementById('delete-account-btn');
        if (deleteAccountBtn) {
            deleteAccountBtn.addEventListener('click', () => this.showDeleteAccountModal());
        }
        
        // Delete account confirmation
        const confirmDeleteBtn = document.getElementById('confirm-delete-btn');
        const confirmDeleteInput = document.getElementById('confirm-delete');
        const cancelDeleteBtn = document.getElementById('cancel-delete');
        
        if (confirmDeleteInput) {
            confirmDeleteInput.addEventListener('input', (e) => {
                const confirmDeleteBtn = document.getElementById('confirm-delete-btn');
                if (confirmDeleteBtn) {
                    confirmDeleteBtn.disabled = e.target.value !== 'DELETE';
                }
            });
        }
        
        if (confirmDeleteBtn) {
            confirmDeleteBtn.addEventListener('click', () => this.deleteAccount());
        }
        
        if (cancelDeleteBtn) {
            cancelDeleteBtn.addEventListener('click', () => this.hideDeleteAccountModal());
        }
        
        // Form validation
        this.initializeFormValidation();
    }
    
    initializeModals() {
        // Delete account modal
        const deleteAccountModal = document.getElementById('delete-account-modal');
        const modalClose = deleteAccountModal?.querySelector('.modal-close');
        
        if (deleteAccountModal) {
            deleteAccountModal.addEventListener('click', (e) => {
                if (e.target === deleteAccountModal) {
                    this.hideDeleteAccountModal();
                }
            });
            
            if (modalClose) {
                modalClose.addEventListener('click', () => this.hideDeleteAccountModal());
            }
        }
    }
    
    initializeFormValidation() {
        // Password confirmation validation
        const newPassword = document.getElementById('new_password');
        const confirmPassword = document.getElementById('confirm_password');
        
        if (newPassword && confirmPassword) {
            const validatePasswords = () => {
                if (newPassword.value && confirmPassword.value) {
                    if (newPassword.value !== confirmPassword.value) {
                        confirmPassword.setCustomValidity('Passwords do not match');
                    } else {
                        confirmPassword.setCustomValidity('');
                    }
                }
            };
            
            newPassword.addEventListener('input', validatePasswords);
            confirmPassword.addEventListener('input', validatePasswords);
        }
        
        // Word count validation
        const wordCountInput = document.getElementById('default_word_count');
        if (wordCountInput) {
            wordCountInput.addEventListener('input', (e) => {
                const value = parseInt(e.target.value);
                if (value < 100 || value > 5000) {
                    e.target.setCustomValidity('Word count must be between 100 and 5000');
                } else {
                    e.target.setCustomValidity('');
                }
            });
        }
    }
    
    showDeleteAccountModal() {
        const modal = document.getElementById('delete-account-modal');
        if (modal) {
            modal.classList.add('show');
            // Clear confirmation input
            const confirmInput = document.getElementById('confirm-delete');
            if (confirmInput) {
                confirmInput.value = '';
            }
            // Disable confirm button
            const confirmBtn = document.getElementById('confirm-delete-btn');
            if (confirmBtn) {
                confirmBtn.disabled = true;
            }
        }
    }
    
    hideDeleteAccountModal() {
        const modal = document.getElementById('delete-account-modal');
        if (modal) {
            modal.classList.remove('show');
        }
    }
    
    async deleteAccount() {
        const confirmInput = document.getElementById('confirm-delete');
        if (!confirmInput || confirmInput.value !== 'DELETE') {
            showAlert('Please type "DELETE" to confirm account deletion');
            return;
        }
        
        if (!confirm('Are you absolutely sure you want to delete your account? This action cannot be undone.')) {
            return;
        }
        
        try {
            showLoadingOverlay('Deleting your account...');
            
            const response = await fetch('delete_account.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                }
            });
            
            const result = await response.json();
            
            hideLoadingOverlay();
            
            if (result.success) {
                showAlert('Account deleted successfully. Redirecting to home page...', 'success');
                setTimeout(() => {
                    window.location.href = 'index.php';
                }, 2000);
            } else {
                showAlert('Failed to delete account: ' + (result.message || 'Unknown error'));
            }
        } catch (error) {
            hideLoadingOverlay();
            showAlert('Error deleting account: ' + error.message);
        }
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    window.settingsManager = new SettingsManager();
});