<?php
require_once 'config.php';

if (!is_logged_in()) {
    redirect('index.php');
}

// Get user data
$users = json_decode(file_get_contents(USERS_FILE), true);
$current_user = null;
foreach ($users as $user) {
    if ($user['id'] === $_SESSION['user_id']) {
        $current_user = $user;
        break;
    }
}

if (!$current_user) {
    session_destroy();
    redirect('index.php');
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_profile'])) {
        $username = sanitize_input($_POST['username']);
        $email = sanitize_input($_POST['email']);
        
        // Validation
        if (empty($username) || empty($email)) {
            $profile_error = "All fields are required";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $profile_error = "Please enter a valid email address";
        } else {
            // Check if email is already taken by another user
            $email_taken = false;
            foreach ($users as $user) {
                if ($user['id'] !== $current_user['id'] && $user['email'] === $email) {
                    $email_taken = true;
                    break;
                }
            }
            
            if ($email_taken) {
                $profile_error = "Email already registered by another user";
            } else {
                // Update user data
                foreach ($users as &$user) {
                    if ($user['id'] === $current_user['id']) {
                        $user['username'] = $username;
                        $user['email'] = $email;
                        $current_user = $user;
                        $_SESSION['username'] = $username;
                        break;
                    }
                }
                
                file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT));
                $profile_success = "Profile updated successfully!";
            }
        }
    }
    
    if (isset($_POST['change_password'])) {
        $current_password = $_POST['current_password'];
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];
        
        if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
            $password_error = "All password fields are required";
        } elseif (!password_verify($current_password, $current_user['password'])) {
            $password_error = "Current password is incorrect";
        } elseif ($new_password !== $confirm_password) {
            $password_error = "New passwords do not match";
        } elseif (strlen($new_password) < 6) {
            $password_error = "New password must be at least 6 characters";
        } else {
            // Update password
            foreach ($users as &$user) {
                if ($user['id'] === $current_user['id']) {
                    $user['password'] = password_hash($new_password, PASSWORD_DEFAULT);
                    break;
                }
            }
            
            file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT));
            $password_success = "Password changed successfully!";
        }
    }
    
    if (isset($_POST['update_preferences'])) {
        $default_word_count = intval($_POST['default_word_count']);
        $default_content_type = sanitize_input($_POST['default_content_type']);
        $default_tone = sanitize_input($_POST['default_tone']);
        
        // Validate word count
        if ($default_word_count < 100 || $default_word_count > 5000) {
            $preferences_error = "Word count must be between 100 and 5000";
        } else {
            // Update user preferences (you might want to add a preferences field to your user data structure)
            foreach ($users as &$user) {
                if ($user['id'] === $current_user['id']) {
                    if (!isset($user['preferences'])) {
                        $user['preferences'] = [];
                    }
                    $user['preferences']['default_word_count'] = $default_word_count;
                    $user['preferences']['default_content_type'] = $default_content_type;
                    $user['preferences']['default_tone'] = $default_tone;
                    break;
                }
            }
            
            file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT));
            $preferences_success = "Preferences updated successfully!";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@6.9.96/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation Header -->
    <nav class="navbar">
        <div class="nav-brand">
            <i class="mdi mdi-pencil-box-multiple"></i>
            <span><?php echo APP_NAME; ?></span>
        </div>
        
        <div class="nav-controls">
            <div class="theme-toggle">
                <input type="checkbox" id="theme-switch" class="theme-switch">
                <label for="theme-switch" class="theme-label">
                    <i class="mdi mdi-weather-sunny"></i>
                    <i class="mdi mdi-weather-night"></i>
                </label>
            </div>
            
            <div class="user-menu">
                <button class="user-btn">
                    <i class="mdi mdi-account-circle"></i>
                    <span><?php echo htmlspecialchars($current_user['username']); ?></span>
                    <i class="mdi mdi-chevron-down"></i>
                </button>
                <div class="user-dropdown">
                    <a href="bulkwriter.php" class="dropdown-item">
                        <i class="mdi mdi-robot"></i> Bulk Writer
                    </a>
                    <a href="settings.php" class="dropdown-item">
                        <i class="mdi mdi-cog"></i> Settings
                    </a>
                    <a href="#" class="dropdown-item" id="api-settings-btn">
                        <i class="mdi mdi-key"></i> API Settings
                    </a>
                    <div class="dropdown-divider"></div>
                    <a href="logout.php" class="dropdown-item">
                        <i class="mdi mdi-logout"></i> Logout
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Main Content -->
    <main class="main-content">
        <div class="container-fluid">
            <!-- Page Header -->
            <div class="page-header">
                <h1><i class="mdi mdi-cog"></i> Settings</h1>
                <p>Manage your account settings and preferences</p>
            </div>

            <!-- API Settings Modal -->
            <div id="api-settings-modal" class="modal">
                <div class="modal-content">
                    <div class="modal-header">
                        <h2>API Configuration</h2>
                        <button class="modal-close">&times;</button>
                    </div>
                    <div class="modal-body">
                        <div class="form-group">
                            <label for="api-key">AI API Key</label>
                            <input type="password" id="api-key" placeholder="Enter your API key (without AIzaSy prefix)" 
                                   value="<?php echo substr($current_user['api_key'] ?? '', 6); ?>">
                            <small>We'll automatically add the "AIzaSy" prefix for security</small>
                        </div>
                        <div class="form-actions">
                            <button type="button" class="btn btn-secondary" id="close-api-modal">Cancel</button>
                            <button type="button" class="btn btn-primary" id="save-api-key">Save API Key</button>
                        </div>
                    </div>
                </div>
            </div>

            <div class="settings-container">
                <!-- Profile Settings -->
                <div class="settings-section">
                    <div class="section-header">
                        <h2><i class="mdi mdi-account"></i> Profile Settings</h2>
                    </div>
                    <div class="settings-card">
                        <form method="POST" class="settings-form">
                            <input type="hidden" name="update_profile" value="1">
                            
                            <?php if (isset($profile_error)): ?>
                                <div class="alert alert-error">
                                    <i class="mdi mdi-alert-circle"></i>
                                    <?php echo $profile_error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($profile_success)): ?>
                                <div class="alert alert-success">
                                    <i class="mdi mdi-check-circle"></i>
                                    <?php echo $profile_success; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="form-group">
                                <label for="username">Username</label>
                                <input type="text" id="username" name="username" 
                                       value="<?php echo htmlspecialchars($current_user['username']); ?>" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="email">Email Address</label>
                                <input type="email" id="email" name="email" 
                                       value="<?php echo htmlspecialchars($current_user['email']); ?>" required>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary">
                                    <i class="mdi mdi-content-save"></i> Update Profile
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Password Settings -->
                <div class="settings-section">
                    <div class="section-header">
                        <h2><i class="mdi mdi-lock"></i> Change Password</h2>
                    </div>
                    <div class="settings-card">
                        <form method="POST" class="settings-form">
                            <input type="hidden" name="change_password" value="1">
                            
                            <?php if (isset($password_error)): ?>
                                <div class="alert alert-error">
                                    <i class="mdi mdi-alert-circle"></i>
                                    <?php echo $password_error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($password_success)): ?>
                                <div class="alert alert-success">
                                    <i class="mdi mdi-check-circle"></i>
                                    <?php echo $password_success; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="form-group">
                                <label for="current_password">Current Password</label>
                                <input type="password" id="current_password" name="current_password" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="new_password">New Password</label>
                                <input type="password" id="new_password" name="new_password" required>
                                <small>Must be at least 6 characters long</small>
                            </div>
                            
                            <div class="form-group">
                                <label for="confirm_password">Confirm New Password</label>
                                <input type="password" id="confirm_password" name="confirm_password" required>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary">
                                    <i class="mdi mdi-key-change"></i> Change Password
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Application Preferences -->
                <div class="settings-section">
                    <div class="section-header">
                        <h2><i class="mdi mdi-tune"></i> Application Preferences</h2>
                    </div>
                    <div class="settings-card">
                        <form method="POST" class="settings-form">
                            <input type="hidden" name="update_preferences" value="1">
                            
                            <?php if (isset($preferences_error)): ?>
                                <div class="alert alert-error">
                                    <i class="mdi mdi-alert-circle"></i>
                                    <?php echo $preferences_error; ?>
                                </div>
                            <?php endif; ?>
                            
                            <?php if (isset($preferences_success)): ?>
                                <div class="alert alert-success">
                                    <i class="mdi mdi-check-circle"></i>
                                    <?php echo $preferences_success; ?>
                                </div>
                            <?php endif; ?>
                            
                            <div class="form-group">
                                <label for="default_word_count">Default Word Count</label>
                                <input type="number" id="default_word_count" name="default_word_count" 
                                       value="<?php echo $current_user['preferences']['default_word_count'] ?? 1000; ?>" 
                                       min="100" max="5000" required>
                            </div>
                            
                            <div class="form-group">
                                <label for="default_content_type">Default Content Type</label>
                                <select id="default_content_type" name="default_content_type">
                                    <option value="Articles" <?php echo ($current_user['preferences']['default_content_type'] ?? 'Articles') === 'Articles' ? 'selected' : ''; ?>>Articles</option>
                                    <option value="Blog Posts" <?php echo ($current_user['preferences']['default_content_type'] ?? 'Articles') === 'Blog Posts' ? 'selected' : ''; ?>>Blog Posts</option>
                                    <option value="Listicles" <?php echo ($current_user['preferences']['default_content_type'] ?? 'Articles') === 'Listicles' ? 'selected' : ''; ?>>Listicles</option>
                                    <option value="Product Descriptions" <?php echo ($current_user['preferences']['default_content_type'] ?? 'Articles') === 'Product Descriptions' ? 'selected' : ''; ?>>Product Descriptions</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="default_tone">Default Tone</label>
                                <select id="default_tone" name="default_tone">
                                    <option value="Professional" <?php echo ($current_user['preferences']['default_tone'] ?? 'Professional') === 'Professional' ? 'selected' : ''; ?>>Professional</option>
                                    <option value="Casual" <?php echo ($current_user['preferences']['default_tone'] ?? 'Professional') === 'Casual' ? 'selected' : ''; ?>>Casual</option>
                                    <option value="Friendly" <?php echo ($current_user['preferences']['default_tone'] ?? 'Professional') === 'Friendly' ? 'selected' : ''; ?>>Friendly</option>
                                    <option value="Authoritative" <?php echo ($current_user['preferences']['default_tone'] ?? 'Professional') === 'Authoritative' ? 'selected' : ''; ?>>Authoritative</option>
                                </select>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" class="btn btn-primary">
                                    <i class="mdi mdi-content-save"></i> Save Preferences
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Danger Zone -->
                <div class="settings-section">
                    <div class="section-header">
                        <h2><i class="mdi mdi-alert"></i> Danger Zone</h2>
                    </div>
                    <div class="settings-card danger-zone">
                        <div class="danger-content">
                            <div class="danger-icon">
                                <i class="mdi mdi-alert-circle"></i>
                            </div>
                            <div class="danger-text">
                                <h3>Delete Account</h3>
                                <p>Once you delete your account, there is no going back. Please be certain.</p>
                            </div>
                            <button type="button" class="btn btn-danger" id="delete-account-btn">
                                <i class="mdi mdi-delete"></i> Delete Account
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Delete Account Confirmation Modal -->
    <div id="delete-account-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Delete Account</h2>
                <button class="modal-close">&times;</button>
            </div>
            <div class="modal-body">
                <div class="delete-warning">
                    <i class="mdi mdi-alert-circle"></i>
                    <h3>Are you absolutely sure?</h3>
                    <p>This action cannot be undone. This will permanently delete your account and remove all your data from our servers.</p>
                </div>
                <div class="form-group">
                    <label for="confirm-delete">To confirm, type "DELETE" below:</label>
                    <input type="text" id="confirm-delete" placeholder="Type DELETE to confirm">
                </div>
                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" id="cancel-delete">Cancel</button>
                    <button type="button" class="btn btn-danger" id="confirm-delete-btn" disabled>
                        <i class="mdi mdi-delete"></i> Delete Account
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="script.js"></script>
    <script src="settings.js"></script>
</body>
</html>