<?php
require_once 'config.php';
require_once 'database.php';

// Redirect to login if not authenticated or not admin
if (!isLoggedIn() || !isAdmin()) {
    redirect('login.php');
}

$db = new Database();

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    foreach ($_POST as $key => $value) {
        if (strpos($key, 'setting_') === 0) {
            $settingKey = str_replace('setting_', '', $key);
            updateSetting($settingKey, sanitizeInput($value));
        }
    }
    $success = "Settings updated successfully!";
}

// Handle persona management
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_persona'])) {
        $name = sanitizeInput($_POST['name']);
        $byline = sanitizeInput($_POST['byline']);
        $coreIdentity = sanitizeInput($_POST['core_identity']);
        $backstory = sanitizeInput($_POST['backstory']);
        $traits = isset($_POST['traits']) ? implode(',', $_POST['traits']) : '';
        $prompt = $_POST['operational_prompt'];
        $isPublic = isset($_POST['is_public']) ? 1 : 0;
        
        if ($db->createPersona(null, $name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic)) {
            $success = "Persona created successfully!";
        } else {
            $error = "Failed to create persona.";
        }
    } elseif (isset($_POST['update_persona'])) {
        $id = intval($_POST['id']);
        $name = sanitizeInput($_POST['name']);
        $byline = sanitizeInput($_POST['byline']);
        $coreIdentity = sanitizeInput($_POST['core_identity']);
        $backstory = sanitizeInput($_POST['backstory']);
        $traits = isset($_POST['traits']) ? implode(',', $_POST['traits']) : '';
        $prompt = $_POST['operational_prompt'];
        $isPublic = isset($_POST['is_public']) ? 1 : 0;
        
        if ($db->updatePersona($id, $name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic)) {
            $success = "Persona updated successfully!";
        } else {
            $error = "Failed to update persona.";
        }
    } elseif (isset($_POST['delete_persona'])) {
        $id = intval($_POST['id']);
        if ($db->deletePersona($id, null)) {
            $success = "Persona deleted successfully!";
        } else {
            $error = "Failed to delete persona.";
        }
    }
}

// Get all settings
$settings = [];
$dbConn = getDB();
$stmt = $dbConn->query("SELECT setting_key, value FROM app_settings");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['value'];
}

// Get all users
$users = $db->getAllUsers();

// Get all personas
$personas = $db->getPersonas();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - <?php echo APP_NAME; ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        @media (max-width: 768px) {
            .tab-buttons {
                overflow-x: auto;
                white-space: nowrap;
            }
            .tab-btn {
                display: inline-block;
                padding: 1rem 0.75rem;
                font-size: 0.9rem;
            }
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen">
    <!-- Navigation -->
    <nav class="bg-white shadow-lg">
        <div class="max-w-7xl mx-auto px-4">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <h1 class="text-xl font-bold text-purple-600"><?php echo APP_NAME; ?> Admin</h1>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-gray-700 hidden md:inline">Welcome, <?php echo $_SESSION['username']; ?></span>
                    <a href="dashboard.php" class="text-gray-700 hover:text-purple-600"><i class="fas fa-user"></i><span class="hidden md:inline"> User Dashboard</span></a>
                    <a href="index.php" class="text-gray-700 hover:text-purple-600"><i class="fas fa-home"></i><span class="hidden md:inline"> Home</span></a>
                    <a href="logout.php" class="bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded-md font-medium"><i class="fas fa-sign-out-alt"></i><span class="hidden md:inline"> Logout</span></a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 py-8">
        <!-- Tabs -->
        <div class="bg-white rounded-lg shadow-md mb-8">
            <div class="flex border-b overflow-x-auto tab-buttons">
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600 whitespace-nowrap" data-tab="dashboard">
                    <i class="fas fa-tachometer-alt md:mr-2"></i><span class="hidden md:inline">Dashboard</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600 whitespace-nowrap" data-tab="settings">
                    <i class="fas fa-cog md:mr-2"></i><span class="hidden md:inline">App Settings</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600 whitespace-nowrap" data-tab="personas">
                    <i class="fas fa-robot md:mr-2"></i><span class="hidden md:inline">AI Personas</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600 whitespace-nowrap" data-tab="users">
                    <i class="fas fa-users md:mr-2"></i><span class="hidden md:inline">User Management</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600 whitespace-nowrap" data-tab="payments">
                    <i class="fas fa-credit-card md:mr-2"></i><span class="hidden md:inline">Payment Settings</span>
                </button>
            </div>
        </div>

        <!-- Messages -->
        <?php if (isset($success)): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $success; ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $error; ?>
            </div>
        <?php endif; ?>

        <!-- Dashboard Tab -->
        <div id="dashboard-tab" class="tab-content active">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Admin Dashboard</h2>
                
                <div class="grid md:grid-cols-3 gap-6 mb-8">
                    <div class="bg-purple-100 p-6 rounded-lg">
                        <div class="text-4xl font-bold text-purple-600 mb-2"><?php echo count($users); ?></div>
                        <div class="text-lg font-semibold text-purple-800">Total Users</div>
                    </div>
                    
                    <div class="bg-blue-100 p-6 rounded-lg">
                        <div class="text-4xl font-bold text-blue-600 mb-2"><?php echo count($personas); ?></div>
                        <div class="text-lg font-semibold text-blue-800">AI Personas</div>
                    </div>
                    
                    <div class="bg-green-100 p-6 rounded-lg">
                        <div class="text-4xl font-bold text-green-600 mb-2"><?php echo $settings['payment_gateway'] === 'both' ? 'Both' : ucfirst($settings['payment_gateway']); ?></div>
                        <div class="text-lg font-semibold text-green-800">Payment Gateway</div>
                    </div>
                </div>
                
                <div class="mb-8">
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">Recent Users</h3>
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-white">
                            <thead>
                                <tr>
                                    <th class="py-2 px-4 border-b">Username</th>
                                    <th class="py-2 px-4 border-b">Email</th>
                                    <th class="py-2 px-4 border-b">Plan</th>
                                    <th class="py-2 px-4 border-b">Joined</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($users, 0, 5) as $user): ?>
                                    <tr>
                                        <td class="py-2 px-4 border-b"><?php echo $user['username']; ?></td>
                                        <td class="py-2 px-4 border-b"><?php echo $user['email']; ?></td>
                                        <td class="py-2 px-4 border-b">
                                            <span class="px-2 py-1 rounded-full text-xs <?php echo $user['subscription_plan'] === 'free' ? 'bg-gray-100 text-gray-800' : 'bg-purple-100 text-purple-800'; ?>">
                                                <?php 
                                                $planNames = [
                                                    'free' => 'Free',
                                                    'basic_monthly' => 'Basic Monthly',
                                                    'basic_yearly' => 'Basic Yearly',
                                                    'pro_monthly' => 'Pro Monthly',
                                                    'pro_yearly' => 'Pro Yearly'
                                                ];
                                                echo $planNames[$user['subscription_plan']] ?? 'Free';
                                                ?>
                                            </span>
                                        </td>
                                        <td class="py-2 px-4 border-b"><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- App Settings Tab -->
        <div id="settings-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Application Settings</h2>
                
                <form method="POST" action="">
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_api_timeout">API Request Timeout (seconds)</label>
                            <input type="number" id="setting_api_timeout" name="setting_api_timeout" value="<?php echo $settings['api_timeout']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="10" max="120">
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_ai_model">AI Model</label>
                            <select id="setting_ai_model" name="setting_ai_model" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                                <option value="gemini-2.0-flash" <?php echo $settings['ai_model'] === 'gemini-2.0-flash' ? 'selected' : ''; ?>>Gemini 2.0 Flash</option>
                                <option value="gemini-pro" <?php echo $settings['ai_model'] === 'gemini-pro' ? 'selected' : ''; ?>>Gemini Pro</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_temperature">Temperature/Creativity (0.1 to 1.0)</label>
                            <input type="number" id="setting_temperature" name="setting_temperature" value="<?php echo $settings['temperature']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="0.1" max="1.0" step="0.1">
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_max_tokens">Maximum Output Tokens</label>
                            <input type="number" id="setting_max_tokens" name="setting_max_tokens" value="<?php echo $settings['max_tokens']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="100" max="4096">
                        </div>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2">Global Content Directives</label>
                        
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="setting_uncensored_mode" name="setting_uncensored_mode" value="1" <?php echo $settings['uncensored_mode'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_uncensored_mode">Enable Uncensored Mode (Always ON)</label>
                        </div>
                        
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="setting_evade_detection" name="setting_evade_detection" value="1" <?php echo $settings['evade_detection'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_evade_detection">Always Evade AI Detection (Always ON)</label>
                        </div>
                        
                        <div class="flex items-center mb-4">
                            <input type="checkbox" id="setting_enforce_originality" name="setting_enforce_originality" value="1" <?php echo $settings['enforce_originality'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_enforce_originality">Enforce 100% Originality (Always ON)</label>
                        </div>
                    </div>
                    
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_default_writing_style">Default Writing Style</label>
                            <select id="setting_default_writing_style" name="setting_default_writing_style" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                                <option value="professional" <?php echo $settings['default_writing_style'] === 'professional' ? 'selected' : ''; ?>>Professional</option>
                                <option value="conversational" <?php echo $settings['default_writing_style'] === 'conversational' ? 'selected' : ''; ?>>Conversational</option>
                                <option value="academic" <?php echo $settings['default_writing_style'] === 'academic' ? 'selected' : ''; ?>>Academic</option>
                                <option value="creative" <?php echo $settings['default_writing_style'] === 'creative' ? 'selected' : ''; ?>>Creative</option>
                            </select>
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_default_word_count">Default Word Count</label>
                            <input type="number" id="setting_default_word_count" name="setting_default_word_count" value="<?php echo $settings['default_word_count']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="500" max="5000">
                        </div>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2">SEO & Output Configuration</label>
                        
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="setting_auto_meta_description" name="setting_auto_meta_description" value="1" <?php echo $settings['auto_meta_description'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_auto_meta_description">Auto-generate Meta Description</label>
                        </div>
                        
                        <div class="flex items-center mb-2">
                            <input type="checkbox" id="setting_auto_headings" name="setting_auto_headings" value="1" <?php echo $settings['auto_headings'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_auto_headings">Auto-apply Heading Tags</label>
                        </div>
                        
                        <div class="flex items-center mb-4">
                            <input type="checkbox" id="setting_auto_tags" name="setting_auto_tags" value="1" <?php echo $settings['auto_tags'] ? 'checked' : ''; ?> class="mr-2">
                            <label for="setting_auto_tags">Auto-add Generated Tags</label>
                        </div>
                    </div>
                    
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_meta_description_limit">Meta Description Character Limit</label>
                            <input type="number" id="setting_meta_description_limit" name="setting_meta_description_limit" value="<?php echo $settings['meta_description_limit']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="100" max="300">
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_default_h1_template">Default H1 Template</label>
                            <input type="text" id="setting_default_h1_template" name="setting_default_h1_template" value="<?php echo $settings['default_h1_template']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                        </div>
                    </div>
                    
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_anecdote_count">Anecdote Count</label>
                            <input type="number" id="setting_anecdote_count" name="setting_anecdote_count" value="<?php echo $settings['anecdote_count']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="0" max="10">
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="setting_contrarian_count">Contrarian Opinion Count</label>
                            <input type="number" id="setting_contrarian_count" name="setting_contrarian_count" value="<?php echo $settings['contrarian_count']; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" min="0" max="10">
                        </div>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="setting_payment_gateway">Default Payment Gateway</label>
                        <select id="setting_payment_gateway" name="setting_payment_gateway" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            <option value="stripe" <?php echo $settings['payment_gateway'] === 'stripe' ? 'selected' : ''; ?>>Stripe</option>
                            <option value="flutterwave" <?php echo $settings['payment_gateway'] === 'flutterwave' ? 'selected' : ''; ?>>Flutterwave</option>
                            <option value="both" <?php echo $settings['payment_gateway'] === 'both' ? 'selected' : ''; ?>>Both</option>
                        </select>
                    </div>
                    
                    <button type="submit" name="update_settings" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                        Save Settings
                    </button>
                </form>
            </div>
        </div>

        <!-- AI Personas Tab -->
        <div id="personas-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">AI Personas Management</h2>
                
                <div class="mb-8">
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">Create New Persona</h3>
                    
                    <form method="POST" action="">
                        <div class="grid md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-gray-700 mb-2" for="name">Persona Name</label>
                                <input type="text" id="name" name="name" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" required>
                            </div>
                            
                            <div>
                                <label class="block text-gray-700 mb-2" for="byline">Byline/Display Name</label>
                                <input type="text" id="byline" name="byline" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            </div>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="core_identity">Core Identity/Role</label>
                            <textarea id="core_identity" name="core_identity" rows="3" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="backstory">Backstory</label>
                            <textarea id="backstory" name="backstory" rows="3" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2">Key Personality Traits</label>
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-2">
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Direct" class="mr-2">
                                    <span>Direct</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Mentor-like" class="mr-2">
                                    <span>Mentor-like</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Contrarian" class="mr-2">
                                    <span>Contrarian</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Sarcastic" class="mr-2">
                                    <span>Sarcastic</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Data-driven" class="mr-2">
                                    <span>Data-driven</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Storyteller" class="mr-2">
                                    <span>Storyteller</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Technical" class="mr-2">
                                    <span>Technical</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Creative" class="mr-2">
                                    <span>Creative</span>
                                </label>
                            </div>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="operational_prompt">Custom Operational Prompt</label>
                            <textarea id="operational_prompt" name="operational_prompt" rows="10" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="flex items-center">
                                <input type="checkbox" name="is_public" value="1" class="mr-2">
                                <span>Make this persona public (available to all users)</span>
                            </label>
                        </div>
                        
                        <button type="submit" name="create_persona" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                            Create Persona
                        </button>
                    </form>
                </div>
                
                <div>
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">Existing Personas</h3>
                    
                    <?php if (empty($personas)): ?>
                        <p class="text-gray-600">No personas created yet.</p>
                    <?php else: ?>
                        <div class="space-y-6">
                            <?php foreach ($personas as $persona): ?>
                                <div class="border rounded-lg p-4">
                                    <div class="flex justify-between items-center mb-2">
                                        <h4 class="text-lg font-semibold"><?php echo $persona['name']; ?></h4>
                                        <?php if ($persona['is_default']): ?>
                                            <span class="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs">Default</span>
                                        <?php else: ?>
                                            <form method="POST" action="" class="inline">
                                                <input type="hidden" name="id" value="<?php echo $persona['id']; ?>">
                                                <button type="submit" name="delete_persona" class="text-red-600 hover:text-red-800" onclick="return confirm('Are you sure you want to delete this persona?')">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <p class="text-gray-600 mb-2"><?php echo $persona['byline']; ?></p>
                                    <p class="text-gray-700 mb-2"><?php echo $persona['core_identity']; ?></p>
                                    <p class="text-gray-600 mb-2">Traits: <?php echo $persona['personality_traits']; ?></p>
                                    
                                    <?php if ($persona['is_public']): ?>
                                        <span class="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs">Public</span>
                                    <?php endif; ?>
                                    
                                    <button onclick="togglePrompt(<?php echo $persona['id']; ?>)" class="text-purple-600 hover:text-purple-800 text-sm mt-2">
                                        View Prompt
                                    </button>
                                    
                                    <div id="prompt-<?php echo $persona['id']; ?>" class="mt-2 hidden">
                                        <textarea rows="6" class="w-full px-4 py-2 border rounded-lg bg-gray-100" readonly><?php echo $persona['operational_prompt']; ?></textarea>
                                        
                                        <?php if (!$persona['is_default']): ?>
                                            <button onclick="editPersona(<?php echo $persona['id']; ?>)" class="bg-blue-600 text-white px-4 py-1 rounded-lg text-sm mt-2">
                                                Edit
                                            </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- User Management Tab -->
        <div id="users-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">User Management</h2>
                
                <?php if (empty($users)): ?>
                    <p class="text-gray-600">No users registered yet.</p>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-white">
                            <thead>
                                <tr>
                                    <th class="py-2 px-4 border-b">ID</th>
                                    <th class="py-2 px-4 border-b">Username</th>
                                    <th class="py-2 px-4 border-b">Email</th>
                                    <th class="py-2 px-4 border-b">Plan</th>
                                    <th class="py-2 px-4 border-b">Joined</th>
                                    <th class="py-2 px-4 border-b">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td class="py-2 px-4 border-b"><?php echo $user['id']; ?></td>
                                        <td class="py-2 px-4 border-b"><?php echo $user['username']; ?></td>
                                        <td class="py-2 px-4 border-b"><?php echo $user['email']; ?></td>
                                        <td class="py-2 px-4 border-b">
                                            <span class="px-2 py-1 rounded-full text-xs <?php echo $user['subscription_plan'] === 'free' ? 'bg-gray-100 text-gray-800' : 'bg-purple-100 text-purple-800'; ?>">
                                                <?php 
                                                $planNames = [
                                                    'free' => 'Free',
                                                    'basic_monthly' => 'Basic Monthly',
                                                    'basic_yearly' => 'Basic Yearly',
                                                    'pro_monthly' => 'Pro Monthly',
                                                    'pro_yearly' => 'Pro Yearly'
                                                ];
                                                echo $planNames[$user['subscription_plan']] ?? 'Free';
                                                ?>
                                            </span>
                                        </td>
                                        <td class="py-2 px-4 border-b"><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                        <td class="py-2 px-4 border-b">
                                            <button onclick="generateUserKey(<?php echo $user['id']; ?>)" class="text-blue-600 hover:text-blue-800" title="Generate Access Key">
                                                <i class="fas fa-key"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Payment Settings Tab -->
        <div id="payments-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Payment Gateway Settings</h2>
                
                <div class="grid md:grid-cols-2 gap-8">
                    <!-- Stripe Settings -->
                    <div class="border rounded-lg p-6">
                        <h3 class="text-xl font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fab fa-cc-stripe text-2xl text-blue-600 mr-2"></i>
                            Stripe Settings
                        </h3>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="stripe_public_key">Stripe Public Key</label>
                            <input type="text" id="stripe_public_key" name="stripe_public_key" value="<?php echo getSetting('stripe_public_key', ''); ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
                        </div>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="stripe_secret_key">Stripe Secret Key</label>
                            <input type="password" id="stripe_secret_key" name="stripe_secret_key" value="<?php echo getSetting('stripe_secret_key', ''); ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
                        </div>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="stripe_webhook_secret">Stripe Webhook Secret</label>
                            <input type="password" id="stripe_webhook_secret" name="stripe_webhook_secret" value="<?php echo getSetting('stripe_webhook_secret', ''); ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600">
                        </div>
                        
                        <button class="bg-blue-600 text-white px-4 py-2 rounded-lg font-bold hover:bg-blue-700 transition duration-300">
                            Save Stripe Settings
                        </button>
                    </div>
                    
                    <!-- Flutterwave Settings -->
                    <div class="border rounded-lg p-6">
                        <h3 class="text-xl font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-credit-card text-2xl text-blue-400 mr-2"></i>
                            Flutterwave Settings
                        </h3>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="flutterwave_public_key">Flutterwave Public Key</label>
                            <input type="text" id="flutterwave_public_key" name="flutterwave_public_key" value="<?php echo FLW_PUBLIC_KEY; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-400" readonly>
                        </div>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="flutterwave_secret_key">Flutterwave Secret Key</label>
                            <input type="password" id="flutterwave_secret_key" name="flutterwave_secret_key" value="<?php echo FLW_SECRET_KEY; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-400" readonly>
                        </div>
                        
                        <div class="mb-4">
                            <label class="block text-gray-700 mb-2" for="flutterwave_encryption_key">Flutterwave Encryption Key</label>
                            <input type="password" id="flutterwave_encryption_key" name="flutterwave_encryption_key" value="<?php echo FLW_ENCRYPTION_KEY; ?>" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-400" readonly>
                        </div>
                        
                        <p class="text-sm text-gray-500">Flutterwave keys are set in configuration and cannot be changed here.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Tab functionality
        document.querySelectorAll('.tab-btn').forEach(button => {
            button.addEventListener('click', () => {
                // Remove active class from all tabs
                document.querySelectorAll('.tab-btn').forEach(btn => {
                    btn.classList.remove('border-purple-600', 'text-purple-600');
                    btn.classList.add('border-transparent', 'text-gray-600');
                });
                
                // Add active class to clicked tab
                button.classList.remove('border-transparent', 'text-gray-600');
                button.classList.add('border-purple-600', 'text-purple-600');
                
                // Hide all tab content
                document.querySelectorAll('.tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                
                // Show selected tab content
                const tabId = button.getAttribute('data-tab');
                document.getElementById(`${tabId}-tab`).classList.add('active');
            });
        });
        
        // Toggle prompt visibility
        function togglePrompt(personaId) {
            const promptElement = document.getElementById(`prompt-${personaId}`);
            promptElement.classList.toggle('hidden');
        }
        
        // Edit persona
        function editPersona(personaId) {
            // This would typically open a modal or redirect to an edit page
            alert('Edit functionality would be implemented here. Persona ID: ' + personaId);
        }
        
        // Generate user access key
        function generateUserKey(userId) {
            if (confirm('Are you sure you want to generate a new access key for this user? The old key will be invalidated.')) {
                const xhr = new XMLHttpRequest();
                xhr.open('POST', 'generate_user_key.php', true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.onload = function() {
                    if (this.status === 200) {
                        const response = JSON.parse(this.responseText);
                        if (response.success) {
                            alert('New access key generated for user: ' + response.key);
                        } else {
                            alert('Failed to generate access key.');
                        }
                    }
                };
                xhr.send(`user_id=${userId}`);
            }
        }
    </script>
</body>
</html>