<?php
require_once 'config.php';
require_once 'database.php';

// Redirect to login if not authenticated
if (!isLoggedIn()) {
    redirect('login.php');
}

$db = new Database();
$userId = $_SESSION['user_id'];
$user = $db->getUser($_SESSION['username']);

// Handle API key update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_api_key'])) {
    $apiKey = sanitizeInput($_POST['api_key']);
    if ($db->updateUserAPIKey($userId, $apiKey)) {
        $_SESSION['success'] = "API key updated successfully!";
    } else {
        $_SESSION['error'] = "Failed to update API key.";
    }
    redirect('dashboard.php');
}

// Get user's articles
$articles = $db->getArticles($userId);

// Get AI personas (only public ones for basic users, all for pro users)
$personas = $db->getPersonas(canCreatePersonas() ? $userId : null, true);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo APP_NAME; ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/docx/7.8.1/docx.js"></script>
    <style>
        .tab-content {
            display: none;
        }
        .tab-content.active {
            display: block;
        }
        @media (max-width: 768px) {
            .tab-buttons {
                overflow-x: auto;
                white-space: nowrap;
            }
            .tab-btn {
                display: inline-block;
            }
        }
        .loading-spinner {
            display: none;
        }
    </style>
</head>
<body class="bg-gray-100 min-h-screen">
    <!-- Navigation -->
    <nav class="bg-white shadow-lg">
        <div class="max-w-7xl mx-auto px-4">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <h1 class="text-xl font-bold text-purple-600"><?php echo APP_NAME; ?></h1>
                    </div>
                </div>
                <div class="flex items-center space-x-4">
                    <span class="text-gray-700 hidden md:inline">Welcome, <?php echo $_SESSION['username']; ?></span>
                    <a href="index.php" class="text-gray-700 hover:text-purple-600"><i class="fas fa-home"></i><span class="hidden md:inline"> Home</span></a>
                    <a href="logout.php" class="bg-purple-600 hover:bg-purple-700 text-white px-3 py-2 rounded-md font-medium"><i class="fas fa-sign-out-alt"></i><span class="hidden md:inline"> Logout</span></a>
                </div>
            </div>
        </div>
    </nav>

    <div class="max-w-7xl mx-auto px-4 py-8">
        <!-- Tabs -->
        <div class="bg-white rounded-lg shadow-md mb-8">
            <div class="flex border-b tab-buttons">
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600" data-tab="generate">
                    <i class="fas fa-plus-circle md:mr-2"></i><span class="hidden md:inline">Generate</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600" data-tab="articles">
                    <i class="fas fa-file-alt md:mr-2"></i><span class="hidden md:inline">Articles</span>
                </button>
                <?php if (canCreatePersonas()): ?>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600" data-tab="personas">
                    <i class="fas fa-robot md:mr-2"></i><span class="hidden md:inline">Personas</span>
                </button>
                <?php endif; ?>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600" data-tab="settings">
                    <i class="fas fa-cog md:mr-2"></i><span class="hidden md:inline">Settings</span>
                </button>
                <button class="tab-btn py-4 px-4 md:px-6 text-lg font-medium text-gray-600 hover:text-purple-600 border-b-2 border-transparent hover:border-purple-600" data-tab="key">
                    <i class="fas fa-key md:mr-2"></i><span class="hidden md:inline">API Key</span>
                </button>
            </div>
        </div>

        <!-- Messages -->
        <?php if (isset($_SESSION['success'])): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
            </div>
        <?php endif; ?>
        
        <?php if (isset($_SESSION['error'])): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
            </div>
        <?php endif; ?>

        <!-- Generate Content Tab -->
        <div id="generate-tab" class="tab-content active">
            <div class="bg-white rounded-lg shadow-md p-6 mb-8">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Generate New Content</h2>
                
                <form id="generateForm">
                    <div class="grid md:grid-cols-2 gap-6 mb-6">
                        <div>
                            <label class="block text-gray-700 mb-2" for="topic">Article Topic</label>
                            <input type="text" id="topic" name="topic" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" required>
                        </div>
                        
                        <div>
                            <label class="block text-gray-700 mb-2" for="primary_keyword">Primary Keyword</label>
                            <input type="text" id="primary_keyword" name="primary_keyword" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" required>
                        </div>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="secondary_keywords">Secondary Keywords (comma separated)</label>
                        <input type="text" id="secondary_keywords" name="secondary_keywords" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="persona">AI Persona</label>
                        <select id="persona" name="persona" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            <?php foreach ($personas as $persona): ?>
                                <option value="<?php echo $persona['id']; ?>" <?php echo $persona['is_default'] ? 'selected' : ''; ?>>
                                    <?php echo $persona['name']; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <?php if (canUseAdvancedFeatures()): ?>
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2">Advanced Options</label>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block text-gray-700 mb-2" for="temperature">Temperature (0.1-1.0)</label>
                                <input type="number" id="temperature" name="temperature" value="<?php echo getSetting('temperature', 0.7); ?>" min="0.1" max="1.0" step="0.1" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            </div>
                            <div>
                                <label class="block text-gray-700 mb-2" for="max_tokens">Max Tokens</label>
                                <input type="number" id="max_tokens" name="max_tokens" value="<?php echo getSetting('max_tokens', 2048); ?>" min="100" max="4096" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                    <button type="submit" id="generateBtn" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                        Generate Content
                    </button>
                    
                    <div id="loadingSpinner" class="loading-spinner inline-block ml-4">
                        <div class="inline-block animate-spin rounded-full h-6 w-6 border-t-2 border-b-2 border-purple-600"></div>
                        <span class="ml-2 text-gray-600">Generating...</span>
                    </div>
                </form>
            </div>
            
            <div id="generatedContent" class="bg-white rounded-lg shadow-md p-6 hidden">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">Generated Content</h2>
                
                <form id="saveForm" method="POST" action="save_article.php">
                    <input type="hidden" id="generated_title" name="title">
                    <input type="hidden" id="generated_content" name="content">
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="article_title">Article Title</label>
                        <input type="text" id="article_title" name="article_title" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" required>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="excerpt">Excerpt/Meta Description</label>
                        <textarea id="excerpt" name="excerpt" rows="3" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="keywords">Keywords</label>
                        <input type="text" id="keywords" name="keywords" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="content_display">Content</label>
                        <textarea id="content_display" rows="20" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" readonly></textarea>
                    </div>
                    
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="status">Status</label>
                        <select id="status" name="status" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            <option value="draft">Save as Draft</option>
                            <option value="published">Publish Immediately</option>
                        </select>
                    </div>
                    
                    <div class="flex flex-wrap gap-4">
                        <button type="submit" name="save_article" class="bg-green-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-green-700 transition duration-300">
                            Save Article
                        </button>
                        <button type="button" id="copyContent" class="bg-blue-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-blue-700 transition duration-300">
                            Copy to Clipboard
                        </button>
                        <button type="button" id="exportDoc" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                            Export as DOC
                        </button>
                        <button type="button" id="regenerateContent" class="bg-yellow-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-yellow-700 transition duration-300">
                            Regenerate
                        </button>
                        <button type="button" id="discardContent" class="bg-gray-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-gray-700 transition duration-300">
                            Discard
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- My Articles Tab -->
        <div id="articles-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">My Articles</h2>
                
                <?php if (empty($articles)): ?>
                    <p class="text-gray-600">You haven't created any articles yet.</p>
                <?php else: ?>
                    <div class="overflow-x-auto">
                        <table class="min-w-full bg-white">
                            <thead>
                                <tr>
                                    <th class="py-2 px-4 border-b">Title</th>
                                    <th class="py-2 px-4 border-b hidden md:table-cell">Status</th>
                                    <th class="py-2 px-4 border-b hidden md:table-cell">Created</th>
                                    <th class="py-2 px-4 border-b">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($articles as $article): ?>
                                    <tr>
                                        <td class="py-2 px-4 border-b"><?php echo $article['title']; ?></td>
                                        <td class="py-2 px-4 border-b hidden md:table-cell">
                                            <span class="px-2 py-1 rounded-full text-xs <?php echo $article['status'] === 'published' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                                <?php echo ucfirst($article['status']); ?>
                                            </span>
                                        </td>
                                        <td class="py-2 px-4 border-b hidden md:table-cell"><?php echo date('M j, Y', strtotime($article['created_at'])); ?></td>
                                        <td class="py-2 px-4 border-b">
                                            <div class="flex space-x-2">
                                                <a href="edit_article.php?id=<?php echo $article['id']; ?>" class="text-blue-600 hover:text-blue-800" title="Edit"><i class="fas fa-edit"></i></a>
                                                <a href="export_article.php?id=<?php echo $article['id']; ?>" class="text-purple-600 hover:text-purple-800" title="Export"><i class="fas fa-download"></i></a>
                                                <a href="delete_article.php?id=<?php echo $article['id']; ?>" class="text-red-600 hover:text-red-800" title="Delete" onclick="return confirm('Are you sure you want to delete this article?')"><i class="fas fa-trash"></i></a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- AI Personas Tab (Only for Pro users) -->
        <?php if (canCreatePersonas()): ?>
        <div id="personas-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">AI Personas</h2>
                
                <div class="mb-8">
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">Create New Persona</h3>
                    
                    <form id="personaForm" method="POST" action="save_persona.php">
                        <div class="grid md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-gray-700 mb-2" for="persona_name">Persona Name</label>
                                <input type="text" id="persona_name" name="name" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600" required>
                            </div>
                            
                            <div>
                                <label class="block text-gray-700 mb-2" for="persona_byline">Byline/Display Name</label>
                                <input type="text" id="persona_byline" name="byline" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600">
                            </div>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="persona_core_identity">Core Identity/Role</label>
                            <textarea id="persona_core_identity" name="core_identity" rows="3" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="persona_backstory">Backstory</label>
                            <textarea id="persona_backstory" name="backstory" rows="3" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2">Key Personality Traits</label>
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-2">
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Direct" class="mr-2">
                                    <span>Direct</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Mentor-like" class="mr-2">
                                    <span>Mentor-like</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Contrarian" class="mr-2">
                                    <span>Contrarian</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Sarcastic" class="mr-2">
                                    <span>Sarcastic</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Data-driven" class="mr-2">
                                    <span>Data-driven</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Storyteller" class="mr-2">
                                    <span>Storyteller</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Technical" class="mr-2">
                                    <span>Technical</span>
                                </label>
                                <label class="flex items-center">
                                    <input type="checkbox" name="traits[]" value="Creative" class="mr-2">
                                    <span>Creative</span>
                                </label>
                            </div>
                        </div>
                        
                        <div class="mb-6">
                            <label class="block text-gray-700 mb-2" for="persona_prompt">Custom Operational Prompt</label>
                            <textarea id="persona_prompt" name="operational_prompt" rows="10" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-600"></textarea>
                        </div>
                        
                        <div class="mb-6">
                            <label class="flex items-center">
                                <input type="checkbox" name="is_public" value="1" class="mr-2">
                                <span>Make this persona public (available to all users)</span>
                            </label>
                        </div>
                        
                        <button type="submit" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                            Create Persona
                        </button>
                    </form>
                </div>
                
                <div>
                    <h3 class="text-xl font-semibold text-gray-800 mb-4">My Personas</h3>
                    
                    <?php 
                    $myPersonas = $db->getPersonas($userId, false);
                    if (empty($myPersonas)): ?>
                        <p class="text-gray-600">You haven't created any personas yet.</p>
                    <?php else: ?>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <?php foreach ($myPersonas as $persona): ?>
                                <div class="border rounded-lg p-4">
                                    <div class="flex justify-between items-center mb-2">
                                        <h4 class="text-lg font-semibold"><?php echo $persona['name']; ?></h4>
                                        <div>
                                            <a href="edit_persona.php?id=<?php echo $persona['id']; ?>" class="text-blue-600 hover:text-blue-800 mr-2"><i class="fas fa-edit"></i></a>
                                            <a href="delete_persona.php?id=<?php echo $persona['id']; ?>" class="text-red-600 hover:text-red-800" onclick="return confirm('Are you sure you want to delete this persona?')"><i class="fas fa-trash"></i></a>
                                        </div>
                                    </div>
                                    
                                    <p class="text-gray-600 mb-2"><?php echo $persona['byline']; ?></p>
                                    <p class="text-gray-700 mb-2 text-sm"><?php echo strlen($persona['core_identity']) > 100 ? substr($persona['core_identity'], 0, 100) . '...' : $persona['core_identity']; ?></p>
                                    <p class="text-gray-600 mb-2 text-sm">Traits: <?php echo $persona['personality_traits']; ?></p>
                                    
                                    <?php if ($persona['is_public']): ?>
                                        <span class="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs">Public</span>
                                    <?php endif; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Settings Tab -->
        <div id="settings-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">User Settings</h2>
                
                <div class="mb-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-2">Subscription Plan</h3>
                    <p class="text-gray-600">
                        Current Plan: 
                        <span class="font-bold <?php echo $user['subscription_plan'] === 'free' ? 'text-gray-800' : 'text-purple-600'; ?>">
                            <?php 
                            $planNames = [
                                'free' => 'Free',
                                'basic_monthly' => 'Basic Monthly',
                                'basic_yearly' => 'Basic Yearly',
                                'pro_monthly' => 'Pro Monthly',
                                'pro_yearly' => 'Pro Yearly'
                            ];
                            echo $planNames[$user['subscription_plan']] ?? 'Free';
                            ?>
                        </span>
                    </p>
                    <?php if ($user['subscription_plan'] !== 'free' && $user['subscription_end']): ?>
                        <p class="text-gray-600">Renews on: <?php echo date('M j, Y', strtotime($user['subscription_end'])); ?></p>
                    <?php endif; ?>
                    
                    <div class="mt-4">
                        <a href="index.php#pricing" class="bg-purple-600 text-white px-4 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                            Upgrade Plan
                        </a>
                    </div>
                </div>
                
                <div class="mb-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-2">Access Key</h3>
                    <div class="flex items-center">
                        <input type="text" value="<?php echo $user['access_key'] ? $user['access_key'] : 'Not generated'; ?>" class="bg-gray-100 px-4 py-2 rounded-lg flex-grow" readonly>
                        <button onclick="generateAccessKey()" class="bg-blue-600 text-white px-4 py-2 rounded-lg font-bold hover:bg-blue-700 transition duration-300 ml-2">
                            Generate Key
                        </button>
                    </div>
                    <p class="text-sm text-gray-500 mt-2">This key is used for API access to your account.</p>
                </div>
            </div>
        </div>

        <!-- API Key Tab -->
        <div id="key-tab" class="tab-content">
            <div class="bg-white rounded-lg shadow-md p-6">
                <h2 class="text-2xl font-bold text-gray-900 mb-6">API Key Settings</h2>
                
                <form method="POST" action="">
                    <div class="mb-6">
                        <label class="block text-gray-700 mb-2" for="api_key">Your API Key</label>
                        <div class="flex items-center">
                            <span class="bg-gray-200 px-4 py-2 rounded-l-lg">AIzaSy</span>
                            <input type="text" id="api_key" name="api_key" value="<?php echo str_replace('AIzaSy', '', $user['api_key']); ?>" class="flex-grow px-4 py-2 border-y rounded-r-lg focus:outline-none focus:ring-2 focus:ring-purple-600" placeholder="Enter your API key (without AIzaSy prefix)">
                        </div>
                        <p class="text-sm text-gray-500 mt-2">Enter your Gemini API key without the "AIzaSy" prefix (it's added automatically).</p>
                    </div>
                    
                    <button type="submit" name="update_api_key" class="bg-purple-600 text-white px-6 py-2 rounded-lg font-bold hover:bg-purple-700 transition duration-300">
                        Save API Key
                    </button>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Tab functionality
        document.querySelectorAll('.tab-btn').forEach(button => {
            button.addEventListener('click', () => {
                // Remove active class from all tabs
                document.querySelectorAll('.tab-btn').forEach(btn => {
                    btn.classList.remove('border-purple-600', 'text-purple-600');
                    btn.classList.add('border-transparent', 'text-gray-600');
                });
                
                // Add active class to clicked tab
                button.classList.remove('border-transparent', 'text-gray-600');
                button.classList.add('border-purple-600', 'text-purple-600');
                
                // Hide all tab content
                document.querySelectorAll('.tab-content').forEach(content => {
                    content.classList.remove('active');
                });
                
                // Show selected tab content
                const tabId = button.getAttribute('data-tab');
                document.getElementById(`${tabId}-tab`).classList.add('active');
            });
        });
        
        // Generate access key
        function generateAccessKey() {
            if (confirm('Are you sure you want to generate a new access key? The old key will be invalidated.')) {
                const xhr = new XMLHttpRequest();
                xhr.open('POST', 'generate_key.php', true);
                xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                xhr.onload = function() {
                    if (this.status === 200) {
                        const response = JSON.parse(this.responseText);
                        if (response.success) {
                            alert('New access key generated: ' + response.key);
                            window.location.reload();
                        } else {
                            alert('Failed to generate access key.');
                        }
                    }
                };
                xhr.send();
            }
        }
        
        // AJAX content generation
        document.getElementById('generateForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const generateBtn = document.getElementById('generateBtn');
            const loadingSpinner = document.getElementById('loadingSpinner');
            
            generateBtn.disabled = true;
            loadingSpinner.style.display = 'inline-block';
            
            const formData = new FormData(this);
            
            const xhr = new XMLHttpRequest();
            xhr.open('POST', 'generate_content_ajax.php', true);
            xhr.onload = function() {
                generateBtn.disabled = false;
                loadingSpinner.style.display = 'none';
                
                if (this.status === 200) {
                    const response = JSON.parse(this.responseText);
                    
                    if (response.success) {
                        document.getElementById('generatedContent').classList.remove('hidden');
                        document.getElementById('article_title').value = response.title;
                        document.getElementById('generated_title').value = response.title;
                        document.getElementById('excerpt').value = response.excerpt;
                        document.getElementById('content_display').value = response.content;
                        document.getElementById('generated_content').value = response.content;
                        document.getElementById('keywords').value = response.keywords;
                        
                        // Scroll to results
                        document.getElementById('generatedContent').scrollIntoView({ behavior: 'smooth' });
                    } else {
                        alert('Error: ' + response.error);
                    }
                } else {
                    alert('Request failed. Please try again.');
                }
            };
            
            xhr.send(formData);
        });
        
        // Copy content to clipboard
        document.getElementById('copyContent').addEventListener('click', function() {
            const content = document.getElementById('content_display');
            content.select();
            document.execCommand('copy');
            alert('Content copied to clipboard!');
        });
        
        // Export as DOC
        document.getElementById('exportDoc').addEventListener('click', function() {
            const title = document.getElementById('article_title').value || 'Generated Article';
            const content = document.getElementById('content_display').value;
            
            const htmlContent = `<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>${title}</title>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; margin: 40px; }
        h1 { color: #333; border-bottom: 2px solid #333; padding-bottom: 10px; }
    </style>
</head>
<body>
    <h1>${title}</h1>
    <div>${content.replace(/\n/g, '<br>')}</div>
</body>
</html>`;
            
            const blob = new Blob([htmlContent], { type: 'application/msword' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `${title.replace(/[^a-z0-9]/gi, '_').toLowerCase()}.doc`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        });
        
        // Regenerate content
        document.getElementById('regenerateContent').addEventListener('click', function() {
            document.getElementById('generateForm').dispatchEvent(new Event('submit'));
        });
        
        // Discard content
        document.getElementById('discardContent').addEventListener('click', function() {
            if (confirm('Are you sure you want to discard this content?')) {
                document.getElementById('generatedContent').classList.add('hidden');
                document.getElementById('generateForm').reset();
            }
        });
    </script>
</body>
</html>