<?php
require_once 'config.php';

class Database {
    private $pdo;
    
    public function __construct() {
        $this->pdo = getDB();
    }
    
    // User management
    public function createUser($username, $email, $password) {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $this->pdo->prepare("INSERT INTO users (username, email, password) VALUES (?, ?, ?)");
        return $stmt->execute([$username, $email, $hashedPassword]);
    }
    
    public function getUser($identifier) {
        $stmt = $this->pdo->prepare("SELECT * FROM users WHERE username = ? OR email = ?");
        $stmt->execute([$identifier, $identifier]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function verifyUser($identifier, $password) {
        $user = $this->getUser($identifier);
        if ($user && password_verify($password, $user['password'])) {
            return $user;
        }
        return false;
    }
    
    public function updateUserAPIKey($userId, $apiKey) {
        // Add the hardcoded prefix
        $fullApiKey = 'AIzaSy' . $apiKey;
        $stmt = $this->pdo->prepare("UPDATE users SET api_key = ? WHERE id = ?");
        return $stmt->execute([$fullApiKey, $userId]);
    }
    
    public function generateAccessKey($userId) {
        $accessKey = 'UNAI_' . bin2hex(random_bytes(10));
        $stmt = $this->pdo->prepare("UPDATE users SET access_key = ? WHERE id = ?");
        if ($stmt->execute([$accessKey, $userId])) {
            return $accessKey;
        }
        return false;
    }
    
    public function updateUserSubscription($userId, $plan, $duration) {
        $endDate = date('Y-m-d', strtotime("+$duration"));
        $stmt = $this->pdo->prepare("UPDATE users SET subscription_plan = ?, subscription_end = ? WHERE id = ?");
        return $stmt->execute([$plan, $endDate, $userId]);
    }
    
    // Article management
    public function createArticle($userId, $title, $content, $excerpt = '', $keywords = '', $status = 'draft') {
        $stmt = $this->pdo->prepare("INSERT INTO articles (user_id, title, content, excerpt, keywords, status) VALUES (?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$userId, $title, $content, $excerpt, $keywords, $status]);
    }
    
    public function getArticles($userId, $limit = 100, $offset = 0) {
        $stmt = $this->pdo->prepare("SELECT * FROM articles WHERE user_id = ? ORDER BY created_at DESC LIMIT ? OFFSET ?");
        $stmt->bindValue(1, $userId, PDO::PARAM_INT);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->bindValue(3, $offset, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getArticle($id, $userId = null) {
        if ($userId) {
            $stmt = $this->pdo->prepare("SELECT * FROM articles WHERE id = ? AND user_id = ?");
            $stmt->execute([$id, $userId]);
        } else {
            $stmt = $this->pdo->prepare("SELECT * FROM articles WHERE id = ?");
            $stmt->execute([$id]);
        }
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function updateArticle($id, $userId, $title, $content, $excerpt = '', $keywords = '', $status = 'draft') {
        $stmt = $this->pdo->prepare("UPDATE articles SET title = ?, content = ?, excerpt = ?, keywords = ?, status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ? AND user_id = ?");
        return $stmt->execute([$title, $content, $excerpt, $keywords, $status, $id, $userId]);
    }
    
    public function deleteArticle($id, $userId) {
        $stmt = $this->pdo->prepare("DELETE FROM articles WHERE id = ? AND user_id = ?");
        return $stmt->execute([$id, $userId]);
    }
    
    // AI Personas management
    public function getPersonas($userId = null, $includePublic = true) {
        if ($userId) {
            if ($includePublic) {
                $stmt = $this->pdo->prepare("SELECT * FROM ai_personas WHERE user_id = ? OR is_public = TRUE ORDER BY is_default DESC, name ASC");
                $stmt->execute([$userId]);
            } else {
                $stmt = $this->pdo->prepare("SELECT * FROM ai_personas WHERE user_id = ? ORDER BY name ASC");
                $stmt->execute([$userId]);
            }
        } else {
            $stmt = $this->pdo->prepare("SELECT * FROM ai_personas WHERE is_public = TRUE ORDER BY is_default DESC, name ASC");
            $stmt->execute();
        }
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getPersona($id) {
        $stmt = $this->pdo->prepare("SELECT * FROM ai_personas WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function createPersona($userId, $name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic = false) {
        $stmt = $this->pdo->prepare("INSERT INTO ai_personas (user_id, name, byline, core_identity, backstory, personality_traits, operational_prompt, is_public) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$userId, $name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic]);
    }
    
    public function updatePersona($id, $name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic = false) {
        $stmt = $this->pdo->prepare("UPDATE ai_personas SET name = ?, byline = ?, core_identity = ?, backstory = ?, personality_traits = ?, operational_prompt = ?, is_public = ? WHERE id = ?");
        return $stmt->execute([$name, $byline, $coreIdentity, $backstory, $traits, $prompt, $isPublic, $id]);
    }
    
    public function deletePersona($id, $userId) {
        $stmt = $this->pdo->prepare("DELETE FROM ai_personas WHERE id = ? AND user_id = ? AND is_default = FALSE");
        return $stmt->execute([$id, $userId]);
    }
    
    // Admin functions
    public function getAllUsers() {
        $stmt = $this->pdo->prepare("SELECT id, username, email, subscription_plan, subscription_end, created_at FROM users ORDER BY created_at DESC");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function updateUserRole($userId, $role) {
        $stmt = $this->pdo->prepare("UPDATE users SET role = ? WHERE id = ?");
        return $stmt->execute([$role, $userId]);
    }
    
    // Payment management
    public function createPayment($userId, $transactionId, $amount, $plan, $gateway, $status = 'pending') {
        $stmt = $this->pdo->prepare("INSERT INTO payments (user_id, transaction_id, amount, plan, gateway, status) VALUES (?, ?, ?, ?, ?, ?)");
        return $stmt->execute([$userId, $transactionId, $amount, $plan, $gateway, $status]);
    }
    
    public function updatePaymentStatus($transactionId, $status) {
        $stmt = $this->pdo->prepare("UPDATE payments SET status = ? WHERE transaction_id = ?");
        return $stmt->execute([$status, $transactionId]);
    }
    
    public function getUserPayments($userId) {
        $stmt = $this->pdo->prepare("SELECT * FROM payments WHERE user_id = ? ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getPaymentByTxRef($transactionId) {
        $stmt = $this->pdo->prepare("SELECT * FROM payments WHERE transaction_id = ?");
        $stmt->execute([$transactionId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // Export article as DOC
    public function exportArticleAsDoc($articleId, $userId) {
        $article = $this->getArticle($articleId, $userId);
        if (!$article) {
            return false;
        }
        
        $filename = "article_" . $articleId . ".doc";
        $content = "<html>
        <head>
            <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\">
            <title>{$article['title']}</title>
        </head>
        <body>
            <h1>{$article['title']}</h1>
            <div>{$article['content']}</div>
        </body>
        </html>";
        
        header("Content-type: application/vnd.ms-word");
        header("Content-Disposition: attachment; filename=$filename");
        
        echo $content;
        exit;
    }
}
?>