<?php
require_once 'config.php';
require_once 'database.php';

// Handle payment form request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'payment_form') {
        $method = sanitizeInput($_POST['method']);
        $plan = sanitizeInput($_POST['plan']);
        
        $amount = $plan === 'monthly' ? 29 : 249;
        $currency = 'USD';
        $planName = $plan === 'monthly' ? 'Monthly Pro' : 'Yearly Pro';
        
        if ($method === 'stripe') {
            // Stripe payment form
            echo '
            <div class="mb-4">
                <label class="block text-gray-700 mb-2" for="card_number">Card Number</label>
                <input type="text" id="card_number" name="card_number" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600" placeholder="1234 5678 9012 3456" required>
            </div>
            
            <div class="grid grid-cols-2 gap-4 mb-4">
                <div>
                    <label class="block text-gray-700 mb-2" for="expiry">Expiry Date</label>
                    <input type="text" id="expiry" name="expiry" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600" placeholder="MM/YY" required>
                </div>
                <div>
                    <label class="block text-gray-700 mb-2" for="cvc">CVC</label>
                    <input type="text" id="cvc" name="cvc" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600" placeholder="123" required>
                </div>
            </div>
            
            <div class="mb-4">
                <label class="block text-gray-700 mb-2" for="name">Cardholder Name</label>
                <input type="text" id="name" name="name" class="w-full px-4 py-2 border rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-600" required>
            </div>
            
            <button type="button" onclick="processStripePayment()" class="w-full bg-blue-600 text-white py-2 rounded-lg font-bold hover:bg-blue-700 transition duration-300">
                Pay $' . $amount . ' with Stripe
            </button>';
        } elseif ($method === 'flutterwave') {
            // Flutterwave payment form
            echo '
            <div class="mb-4">
                <p class="text-gray-600">You will be redirected to Flutterwave to complete your payment of $' . $amount . ' for ' . $planName . ' plan.</p>
            </div>
            
            <button type="button" onclick="processFlutterwavePayment()" class="w-full bg-blue-400 text-white py-2 rounded-lg font-bold hover:bg-blue-500 transition duration-300">
                Pay with Flutterwave
            </button>';
        }
    } elseif ($_POST['action'] === 'process_stripe') {
        // Process Stripe payment
        $token = sanitizeInput($_POST['stripeToken']);
        $plan = sanitizeInput($_POST['plan']);
        $amount = $plan === 'monthly' ? 2900 : 24900; // in cents
        
        // Here you would typically use the Stripe PHP library
        // For this example, we'll simulate a successful payment
        
        $db = new Database();
        $userId = $_SESSION['user_id'];
        $transactionId = 'stripe_' . bin2hex(random_bytes(8));
        
        if ($db->createPayment($userId, $transactionId, $amount / 100, $plan, 'stripe', 'completed')) {
            // Update user subscription
            $duration = $plan === 'monthly' ? '1 month' : '1 year';
            $db->updateUserSubscription($userId, $plan, $duration);
            
            echo json_encode(['success' => true, 'message' => 'Payment processed successfully!']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Payment failed.']);
        }
    } elseif ($_POST['action'] === 'process_flutterwave') {
        // Process Flutterwave payment
        $plan = sanitizeInput($_POST['plan']);
        $amount = $plan === 'monthly' ? 29 : 249;
        
        // Generate transaction reference
        $txRef = 'flw_' . bin2hex(random_bytes(8));
        
        // Flutterwave payment parameters
        $paymentData = [
            'public_key' => FLW_PUBLIC_KEY,
            'tx_ref' => $txRef,
            'amount' => $amount,
            'currency' => 'USD',
            'payment_options' => 'card,ussd,mobilemoney',
            'redirect_url' => 'http://' . $_SERVER['HTTP_HOST'] . '/payment_verify.php',
            'customer' => [
                'email' => $_SESSION['email'] ?? '',
                'name' => $_SESSION['username'] ?? ''
            ],
            'customizations' => [
                'title' => APP_NAME,
                'description' => $plan === 'monthly' ? 'Monthly Pro Subscription' : 'Yearly Pro Subscription'
            ]
        ];
        
        // Store transaction in database
        $db = new Database();
        $userId = $_SESSION['user_id'];
        $db->createPayment($userId, $txRef, $amount, $plan, 'flutterwave', 'pending');
        
        echo json_encode(['success' => true, 'tx_ref' => $txRef, 'payment_data' => $paymentData]);
    }
    exit;
}

// Handle payment verification
if (isset($_GET['tx_ref'])) {
    $txRef = sanitizeInput($_GET['tx_ref']);
    
    // Verify payment with Flutterwave
    $url = "https://api.flutterwave.com/v3/transactions/" . $txRef . "/verify";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . FLW_SECRET_KEY
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        $responseData = json_decode($response, true);
        if ($responseData['status'] === 'success' && $responseData['data']['status'] === 'successful') {
            // Update payment status
            $db = new Database();
            $db->updatePaymentStatus($txRef, 'completed');
            
            // Update user subscription
            $payment = $db->getPaymentByTxRef($txRef);
            if ($payment) {
                $duration = $payment['plan'] === 'monthly' ? '1 month' : '1 year';
                $db->updateUserSubscription($payment['user_id'], $payment['plan'], $duration);
                
                $_SESSION['success'] = 'Payment completed successfully!';
                redirect('dashboard.php');
            }
        }
    }
    
    $_SESSION['error'] = 'Payment verification failed.';
    redirect('dashboard.php');
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Processing - <?php echo APP_NAME; ?></title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen flex items-center justify-center">
    <div class="bg-white p-8 rounded-lg shadow-md w-full max-w-md">
        <div class="text-center mb-6">
            <h1 class="text-2xl font-bold text-purple-600">Payment Processing</h1>
            <p class="text-gray-600">Completing your subscription</p>
        </div>
        
        <div id="payment-result" class="hidden">
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4" id="success-message"></div>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4" id="error-message"></div>
        </div>
        
        <div class="text-center">
            <div class="inline-block animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-purple-600 mb-4"></div>
            <p class="text-gray-600">Processing your payment...</p>
        </div>
    </div>

    <script>
        // Process Stripe payment
        function processStripePayment() {
            // This would typically use Stripe.js to create a token
            // For this example, we'll simulate the process
            
            const xhr = new XMLHttpRequest();
            xhr.open('POST', 'payment.php', true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            xhr.onload = function() {
                if (this.status === 200) {
                    const response = JSON.parse(this.responseText);
                    if (response.success) {
                        document.getElementById('success-message').textContent = response.message;
                        document.getElementById('payment-result').classList.remove('hidden');
                        setTimeout(() => {
                            window.location.href = 'dashboard.php';
                        }, 2000);
                    } else {
                        document.getElementById('error-message').textContent = response.message;
                        document.getElementById('payment-result').classList.remove('hidden');
                    }
                }
            };
            xhr.send('action=process_stripe&plan=<?php echo $plan; ?>&stripeToken=simulated_token');
        }
        
        // Process Flutterwave payment
        function processFlutterwavePayment() {
            const xhr = new XMLHttpRequest();
            xhr.open('POST', 'payment.php', true);
            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            xhr.onload = function() {
                if (this.status === 200) {
                    const response = JSON.parse(this.responseText);
                    if (response.success) {
                        // Redirect to Flutterwave payment page
                        const form = document.createElement('form');
                        form.method = 'POST';
                        form.action = 'https://checkout.flutterwave.com/v3/hosted/pay';
                        
                        for (const key in response.payment_data) {
                            if (typeof response.payment_data[key] === 'object') {
                                for (const subKey in response.payment_data[key]) {
                                    const input = document.createElement('input');
                                    input.type = 'hidden';
                                    input.name = `${key}[${subKey}]`;
                                    input.value = response.payment_data[key][subKey];
                                    form.appendChild(input);
                                }
                            } else {
                                const input = document.createElement('input');
                                input.type = 'hidden';
                                input.name = key;
                                input.value = response.payment_data[key];
                                form.appendChild(input);
                            }
                        }
                        
                        document.body.appendChild(form);
                        form.submit();
                    } else {
                        document.getElementById('error-message').textContent = 'Payment initialization failed.';
                        document.getElementById('payment-result').classList.remove('hidden');
                    }
                }
            };
            xhr.send('action=process_flutterwave&plan=<?php echo $plan; ?>');
        }
        
        // Auto-process payment based on selected method
        window.onload = function() {
            const urlParams = new URLSearchParams(window.location.search);
            const method = urlParams.get('method');
            const plan = urlParams.get('plan');
            
            if (method && plan) {
                if (method === 'stripe') {
                    processStripePayment();
                } else if (method === 'flutterwave') {
                    processFlutterwavePayment();
                }
            }
        };
    </script>
</body>
</html>