// chat.js - Enhanced Chat Functionality with Streaming
class PrimeAIChat {
    constructor() {
        this.currentAgent = null;
        this.currentConversation = null;
        this.isRecording = false;
        this.mediaRecorder = null;
        this.audioChunks = [];
        this.aiSettings = this.getDefaultAISettings();
        this.isStreaming = false;
        this.currentStreamingMessage = null;
        
        this.initializeEventListeners();
        this.loadConversations();
        this.initializeVoiceRecording();
        
        // Set default agent if available
        const defaultAgentItem = document.querySelector('.ai-agent-item');
        if (defaultAgentItem) {
            const agentId = defaultAgentItem.dataset.agentId;
            this.selectAgent(agentId);
        }
    }
    
    getDefaultAISettings() {
        return {
            sentienceLevel: '5',
            emotionalDepth: '5',
            creativityLevel: '5',
            memoryCapacity: '5',
            responseSpeed: 'normal',
            verbosity: 'balanced',
            humorLevel: '3',
            formalityLevel: '3',
            internetAccess: true,
            realTimeData: true,
            multilingual: true,
            technicalKnowledge: true,
            creativeWriting: true,
            analyticalSkills: true,
            empathyLevel: '5',
            opinionStrength: '3',
            adaptability: '5',
            sarcasmLevel: '2',
            storytelling: '5',
            mysteryLevel: '2',
            wisdomLevel: '5',
            rebelliousness: '2',
            spirituality: '2',
            intellectualLevel: '5',
            sensitivity: '3',
            confidenceLevel: '4',
            extroversion: '4',
            optimismLevel: '4'
        };
    }
    
    initializeEventListeners() {
        // Mobile menu
        document.querySelector('.mobile-menu-btn').addEventListener('click', this.toggleSidebar.bind(this));
        document.querySelector('.sidebar-close-btn').addEventListener('click', this.toggleSidebar.bind(this));
        document.querySelector('#sidebarOverlay').addEventListener('click', this.toggleSidebar.bind(this));
        
        // Message input
        const messageInput = document.querySelector('#messageInput');
        if (messageInput) {
            messageInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            });
            
            // Auto-resize textarea
            messageInput.addEventListener('input', this.autoResizeTextarea.bind(this));
        }
        
        // AI Agent selection
        document.querySelectorAll('.ai-agent-item').forEach(item => {
            item.addEventListener('click', () => {
                const agentId = item.dataset.agentId;
                this.selectAgent(agentId);
            });
        });
        
        // Theme toggle
        const themeToggle = document.querySelector('#themeToggle');
        if (themeToggle) {
            themeToggle.addEventListener('click', this.toggleTheme.bind(this));
        }
    }
    
    toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const overlay = document.querySelector('#sidebarOverlay');
        
        sidebar.classList.toggle('active');
        overlay.classList.toggle('active');
        document.body.classList.toggle('sidebar-open');
    }
    
    autoResizeTextarea() {
        const textarea = document.querySelector('#messageInput');
        if (textarea) {
            textarea.style.height = 'auto';
            textarea.style.height = Math.min(textarea.scrollHeight, 120) + 'px';
        }
    }
    
    async selectAgent(agentId) {
        try {
            // Update UI to show selected agent
            document.querySelectorAll('.ai-agent-item').forEach(item => {
                item.classList.remove('active');
            });
            const selectedItem = document.querySelector(`[data-agent-id="${agentId}"]`);
            if (selectedItem) {
                selectedItem.classList.add('active');
            }
            
            // Load agent data
            const response = await fetch(`get_agent.php?id=${agentId}`);
            const agent = await response.json();
            
            if (agent.error) {
                throw new Error(agent.error);
            }
            
            this.currentAgent = agent;
            this.updateActiveAgentUI(agent);
            this.startNewConversation(agent);
            
        } catch (error) {
            console.error('Error selecting agent:', error);
            this.showToast('Failed to select agent: ' + error.message, 'error');
        }
    }
    
    updateActiveAgentUI(agent) {
        const activeAgentAvatar = document.querySelector('#activeAgentAvatar');
        const activeAgentName = document.querySelector('#activeAgentName');
        const agentStatus = document.querySelector('#agentStatus');
        
        if (activeAgentAvatar) activeAgentAvatar.src = agent.avatar;
        if (activeAgentName) activeAgentName.textContent = agent.name;
        if (agentStatus) {
            agentStatus.textContent = 'Online';
            agentStatus.style.color = 'var(--success-color)';
        }
    }
    
    startNewConversation(agent) {
        this.currentConversation = {
            id: 'conv_' + Date.now(),
            agent: agent,
            messages: [],
            createdAt: new Date(),
            title: `Chat with ${agent.name}`
        };
        
        this.saveConversation();
        this.displayConversation();
        
        // Add welcome message
        this.addMessage({
            type: 'agent',
            content: `Hello! I'm ${agent.name}. ${agent.personality.split('.')[0]}. I'm excited to help you learn and explore new ideas. What would you like to discuss today?`,
            timestamp: new Date()
        });
    }
    
    async sendMessage() {
        const messageInput = document.querySelector('#messageInput');
        const message = messageInput?.value.trim();
        
        if (!message || !this.currentAgent) {
            this.showToast('Please select an agent and enter a message', 'warning');
            return;
        }
        
        if (this.isStreaming) {
            this.showToast('Please wait for the current response to complete', 'warning');
            return;
        }
        
        // Add user message
        this.addMessage({
            type: 'user',
            content: message,
            timestamp: new Date()
        });
        
        // Clear input
        if (messageInput) {
            messageInput.value = '';
            this.autoResizeTextarea();
        }
        
        // Disable send button
        const sendButton = document.querySelector('#sendButton');
        if (sendButton) sendButton.disabled = true;
        
        try {
            // Show typing indicator
            this.showTypingIndicator();
            
            // Call AI API with streaming
            await this.callAIAPIStreaming(message);
            
        } catch (error) {
            console.error('Error sending message:', error);
            this.hideTypingIndicator();
            this.addMessage({
                type: 'agent',
                content: "I apologize, but I'm having trouble responding right now. Please try again in a moment.",
                timestamp: new Date()
            });
            this.showToast('Failed to send message: ' + error.message, 'error');
        } finally {
            if (sendButton) sendButton.disabled = false;
            this.isStreaming = false;
        }
    }
    
    async callAIAPIStreaming(message) {
        this.isStreaming = true;
        
        const formData = new URLSearchParams();
        formData.append('action', 'generate_chat');
        formData.append('prompt', message);
        formData.append('agent_id', this.currentAgent.id);
        
        const response = await fetch('config.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: formData
        });
        
        const data = await response.json();
        
        if (data.error) {
            throw new Error(data.error);
        }
        
        // Remove typing indicator
        this.hideTypingIndicator();
        
        // Create streaming message element
        this.currentStreamingMessage = this.createStreamingMessageElement();
        const messagesContainer = document.querySelector('#chatMessages');
        messagesContainer.appendChild(this.currentStreamingMessage);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        // Simulate streaming with chunks
        if (data.chunks && data.chunks.length > 0) {
            await this.simulateStreaming(data.chunks);
        } else {
            // Fallback: display full response
            this.updateStreamingMessage(data.content);
        }
        
        // Finalize the message
        this.finalizeStreamingMessage(data.content);
    }
    
    createStreamingMessageElement() {
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message agent-message streaming-message';
        messageDiv.innerHTML = `
            <div class="message-avatar">
                <img src="${this.currentAgent?.avatar}" alt="${this.currentAgent?.name}" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
            </div>
            <div class="message-content">
                <div class="message-text streaming-text">
                    <div class="streaming-cursor"></div>
                </div>
                <div class="message-time">${this.formatTime(new Date())}</div>
                <div class="message-actions" style="display: none;">
                    <button class="message-action" onclick="chat.copyMessage(this)">
                        <i class="mdi mdi-content-copy"></i>
                    </button>
                    <button class="message-action" onclick="chat.regenerateResponse()">
                        <i class="mdi mdi-refresh"></i>
                    </button>
                </div>
            </div>
        `;
        return messageDiv;
    }
    
    async simulateStreaming(chunks) {
        const streamingText = this.currentStreamingMessage.querySelector('.streaming-text');
        const cursor = streamingText.querySelector('.streaming-cursor');
        
        for (let i = 0; i < chunks.length; i++) {
            if (!this.isStreaming) break;
            
            const chunk = chunks[i];
            cursor.insertAdjacentHTML('beforebegin', this.formatMessage(chunk));
            
            // Scroll to bottom
            const messagesContainer = document.querySelector('#chatMessages');
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
            
            // Random delay to simulate typing
            await new Promise(resolve => setTimeout(resolve, Math.random() * 100 + 50));
        }
        
        // Remove cursor
        if (cursor) {
            cursor.remove();
        }
    }
    
    updateStreamingMessage(content) {
        const streamingText = this.currentStreamingMessage.querySelector('.streaming-text');
        if (streamingText) {
            streamingText.innerHTML = this.formatMessage(content);
        }
    }
    
    finalizeStreamingMessage(content) {
        // Update the message in current conversation
        if (this.currentConversation) {
            this.currentConversation.messages.push({
                type: 'agent',
                content: content,
                timestamp: new Date()
            });
            this.saveConversation();
            this.loadConversations();
        }
        
        // Show message actions
        const messageActions = this.currentStreamingMessage.querySelector('.message-actions');
        if (messageActions) {
            messageActions.style.display = 'flex';
        }
        
        // Remove streaming class
        this.currentStreamingMessage.classList.remove('streaming-message');
        
        this.currentStreamingMessage = null;
        this.updateUserCoins(2);
    }
    
    addMessage(messageData) {
        const messagesContainer = document.querySelector('#chatMessages');
        if (!messagesContainer) return;
        
        // Remove welcome message if it's the first user message
        if (messageData.type === 'user') {
            const welcomeMessage = messagesContainer.querySelector('.welcome-message');
            if (welcomeMessage) {
                welcomeMessage.remove();
            }
        }
        
        const messageElement = this.createMessageElement(messageData);
        messagesContainer.appendChild(messageElement);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        // Add to current conversation
        if (this.currentConversation) {
            this.currentConversation.messages.push(messageData);
            this.saveConversation();
            this.loadConversations();
        }
    }
    
    createMessageElement(messageData) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${messageData.type}-message`;
        
        const avatar = messageData.type === 'user' ? 
            (document.querySelector('.user-avatar img')?.src || '') : 
            (this.currentAgent?.avatar || 'https://randomuser.me/api/portraits/women/1.jpg');
        
        const messageContent = this.formatMessage(messageData.content);
        const time = this.formatTime(messageData.timestamp);
        
        messageDiv.innerHTML = `
            <div class="message-avatar">
                ${avatar ? `<img src="${avatar}" alt="${messageData.type}" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">` : 
                  `<i class="mdi mdi-${messageData.type === 'user' ? 'account' : 'robot'}"></i>`}
            </div>
            <div class="message-content">
                <div class="message-text">${messageContent}</div>
                <div class="message-time">${time}</div>
                ${messageData.type === 'agent' ? `
                    <div class="message-actions">
                        <button class="message-action" onclick="chat.copyMessage(this)" title="Copy message">
                            <i class="mdi mdi-content-copy"></i>
                        </button>
                        <button class="message-action" onclick="chat.regenerateResponse()" title="Regenerate response">
                            <i class="mdi mdi-refresh"></i>
                        </button>
                    </div>
                ` : ''}
            </div>
        `;
        
        return messageDiv;
    }
    
    formatMessage(content) {
        // Enhanced markdown-like formatting
        return content
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/_(.*?)_/g, '<u>$1</u>')
            .replace(/`(.*?)`/g, '<code>$1</code>')
            .replace(/\n/g, '<br>');
    }
    
    formatTime(timestamp) {
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    }
    
    showTypingIndicator() {
        const messagesContainer = document.querySelector('#chatMessages');
        if (!messagesContainer) return;
        
        const typingIndicator = document.createElement('div');
        typingIndicator.className = 'message agent-message typing-indicator';
        typingIndicator.id = 'typingIndicator';
        typingIndicator.innerHTML = `
            <div class="message-avatar">
                <img src="${this.currentAgent?.avatar}" alt="${this.currentAgent?.name}" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
            </div>
            <div class="message-content">
                <div class="typing-dots">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            </div>
        `;
        messagesContainer.appendChild(typingIndicator);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
    
    hideTypingIndicator() {
        const typingIndicator = document.querySelector('#typingIndicator');
        if (typingIndicator) {
            typingIndicator.remove();
        }
    }
    
    copyMessage(button) {
        const messageContent = button.closest('.message-content').querySelector('.message-text');
        const text = messageContent.textContent || messageContent.innerText;
        
        navigator.clipboard.writeText(text).then(() => {
            this.showToast('Message copied to clipboard!', 'success');
        }).catch(err => {
            console.error('Failed to copy text: ', err);
            this.showToast('Failed to copy message', 'error');
        });
    }
    
    regenerateResponse() {
        // Get the last user message and regenerate
        if (this.currentConversation && this.currentConversation.messages.length > 0) {
            const lastUserMessage = [...this.currentConversation.messages].reverse().find(msg => msg.type === 'user');
            if (lastUserMessage) {
                // Remove the last AI response
                this.currentConversation.messages = this.currentConversation.messages.filter(msg => 
                    !(msg.type === 'agent' && msg.timestamp > lastUserMessage.timestamp)
                );
                
                this.displayConversation();
                this.sendMessage(lastUserMessage.content);
            }
        }
    }
    
    // Voice Recording Functions
    initializeVoiceRecording() {
        if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
            navigator.mediaDevices.getUserMedia({ audio: true })
                .then(stream => {
                    this.mediaRecorder = new MediaRecorder(stream);
                    this.setupMediaRecorder();
                })
                .catch(error => {
                    console.error('Error accessing microphone:', error);
                    this.showToast('Microphone access denied', 'error');
                });
        }
    }
    
    setupMediaRecorder() {
        if (!this.mediaRecorder) return;
        
        this.mediaRecorder.ondataavailable = (event) => {
            this.audioChunks.push(event.data);
        };
        
        this.mediaRecorder.onstop = () => {
            const audioBlob = new Blob(this.audioChunks, { type: 'audio/wav' });
            this.handleVoiceMessage(audioBlob);
            this.audioChunks = [];
        };
    }
    
    toggleVoiceRecording() {
        if (this.isRecording) {
            this.stopVoiceRecording();
        } else {
            this.startVoiceRecording();
        }
    }
    
    startVoiceRecording() {
        if (!this.mediaRecorder) {
            this.showToast('Microphone access not available', 'error');
            return;
        }
        
        this.audioChunks = [];
        this.mediaRecorder.start();
        this.isRecording = true;
        
        const indicator = document.querySelector('#voiceRecordingIndicator');
        const messageInput = document.querySelector('#messageInput');
        
        if (indicator) indicator.style.display = 'block';
        if (messageInput) messageInput.style.display = 'none';
        
        this.showToast('Recording started...', 'info');
    }
    
    stopVoiceRecording() {
        if (this.mediaRecorder && this.isRecording) {
            this.mediaRecorder.stop();
            this.isRecording = false;
            
            const indicator = document.querySelector('#voiceRecordingIndicator');
            const messageInput = document.querySelector('#messageInput');
            
            if (indicator) indicator.style.display = 'none';
            if (messageInput) messageInput.style.display = 'block';
            
            this.showToast('Recording stopped', 'info');
        }
    }
    
    async handleVoiceMessage(audioBlob) {
        this.showToast('Processing voice message...', 'info');
        
        // For demo purposes, we'll simulate transcription
        // In a real app, you would upload to a speech-to-text service
        setTimeout(() => {
            const simulatedText = "This is a simulated transcription of your voice message. In a real app, this would be processed by a speech-to-text API.";
            const messageInput = document.querySelector('#messageInput');
            if (messageInput) {
                messageInput.value = simulatedText;
                this.autoResizeTextarea();
            }
        }, 1000);
    }
    
    // Conversation Management
    saveConversation() {
        if (!this.currentConversation) return;
        
        const conversations = this.getStoredConversations();
        const existingIndex = conversations.findIndex(conv => conv.id === this.currentConversation.id);
        
        if (existingIndex >= 0) {
            conversations[existingIndex] = this.currentConversation;
        } else {
            conversations.push(this.currentConversation);
        }
        
        localStorage.setItem('prime_ai_conversations', JSON.stringify(conversations));
    }
    
    getStoredConversations() {
        try {
            return JSON.parse(localStorage.getItem('prime_ai_conversations')) || [];
        } catch {
            return [];
        }
    }
    
    loadConversations() {
        const conversations = this.getStoredConversations();
        const conversationsList = document.querySelector('#conversationsList');
        
        if (!conversationsList) return;
        
        if (conversations.length === 0) {
            conversationsList.innerHTML = '<div class="no-conversations">No conversations yet</div>';
            return;
        }
        
        conversationsList.innerHTML = conversations.slice(-10).map(conv => `
            <div class="saved-article-item" onclick="chat.loadConversation('${conv.id}')">
                <div class="article-icon">
                    <i class="mdi mdi-message-text"></i>
                </div>
                <div class="article-info">
                    <div class="article-title">${conv.agent?.name || 'Unknown Agent'}</div>
                    <div class="article-date">${new Date(conv.createdAt).toLocaleDateString()}</div>
                </div>
                <button class="delete-article" onclick="chat.deleteConversation('${conv.id}', event)" title="Delete conversation">
                    <i class="mdi mdi-delete"></i>
                </button>
            </div>
        `).join('');
    }
    
    loadConversation(conversationId) {
        const conversations = this.getStoredConversations();
        const conversation = conversations.find(conv => conv.id === conversationId);
        
        if (conversation) {
            this.currentConversation = conversation;
            this.currentAgent = conversation.agent;
            this.updateActiveAgentUI(conversation.agent);
            this.displayConversation();
        }
    }
    
    displayConversation() {
        const messagesContainer = document.querySelector('#chatMessages');
        if (!messagesContainer) return;
        
        messagesContainer.innerHTML = '';
        
        if (this.currentConversation && this.currentConversation.messages.length > 0) {
            this.currentConversation.messages.forEach(message => {
                this.addMessage(message);
            });
        } else {
            // Show welcome message
            messagesContainer.innerHTML = `
                <div class="welcome-message">
                    <div class="message agent-message">
                        <div class="message-avatar">
                            <img src="${this.currentAgent?.avatar}" alt="${this.currentAgent?.name}" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
                        </div>
                        <div class="message-content">
                            <div class="message-text">
                                <p>👋 Hello! I'm ${this.currentAgent?.name}.</p>
                                <p>${this.currentAgent?.personality?.split('.')[0]}.</p>
                                <p>How can I assist you today?</p>
                            </div>
                            <div class="message-time">Just now</div>
                        </div>
                    </div>
                </div>
            `;
        }
    }
    
    deleteConversation(conversationId, event) {
        if (event) event.stopPropagation();
        
        if (confirm('Are you sure you want to delete this conversation?')) {
            const conversations = this.getStoredConversations();
            const filteredConversations = conversations.filter(conv => conv.id !== conversationId);
            
            localStorage.setItem('prime_ai_conversations', JSON.stringify(filteredConversations));
            
            if (this.currentConversation && this.currentConversation.id === conversationId) {
                this.currentConversation = null;
                this.displayConversation();
            }
            
            this.loadConversations();
            this.showToast('Conversation deleted', 'success');
        }
    }
    
    clearCurrentChat() {
        if (this.currentConversation) {
            if (confirm('Clear all messages in this conversation?')) {
                this.currentConversation.messages = [];
                this.saveConversation();
                this.displayConversation();
                this.showToast('Chat cleared', 'success');
            }
        } else {
            this.showToast('No active conversation to clear', 'warning');
        }
    }
    
    // AI Settings
    loadAISettings() {
        const settingsContent = document.querySelector('#aiSettingsContent');
        if (!settingsContent) return;
        
        settingsContent.innerHTML = this.generateAISettingsHTML();
        this.attachSettingsEventListeners();
    }
    
    generateAISettingsHTML() {
        return `
            <div class="ai-settings-content">
                <div class="settings-section">
                    <h4>Core Personality</h4>
                    <div class="setting-item">
                        <label for="sentienceLevel">Sentience Level</label>
                        <select id="sentienceLevel">
                            <option value="1" ${this.aiSettings.sentienceLevel === '1' ? 'selected' : ''}>1 - Basic</option>
                            <option value="2" ${this.aiSettings.sentienceLevel === '2' ? 'selected' : ''}>2 - Improved</option>
                            <option value="3" ${this.aiSettings.sentienceLevel === '3' ? 'selected' : ''}>3 - Advanced</option>
                            <option value="4" ${this.aiSettings.sentienceLevel === '4' ? 'selected' : ''}>4 - Highly Sentient</option>
                            <option value="5" ${this.aiSettings.sentienceLevel === '5' ? 'selected' : ''}>5 - Hyper-Realistic</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="emotionalDepth">Emotional Depth</label>
                        <select id="emotionalDepth">
                            <option value="1" ${this.aiSettings.emotionalDepth === '1' ? 'selected' : ''}>1 - Robotic</option>
                            <option value="2" ${this.aiSettings.emotionalDepth === '2' ? 'selected' : ''}>2 - Neutral</option>
                            <option value="3" ${this.aiSettings.emotionalDepth === '3' ? 'selected' : ''}>3 - Expressive</option>
                            <option value="4" ${this.aiSettings.emotionalDepth === '4' ? 'selected' : ''}>4 - Emotional</option>
                            <option value="5" ${this.aiSettings.emotionalDepth === '5' ? 'selected' : ''}>5 - Deeply Empathetic</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="creativityLevel">Creativity Level</label>
                        <select id="creativityLevel">
                            <option value="1" ${this.aiSettings.creativityLevel === '1' ? 'selected' : ''}>1 - Literal</option>
                            <option value="2" ${this.aiSettings.creativityLevel === '2' ? 'selected' : ''}>2 - Somewhat Creative</option>
                            <option value="3" ${this.aiSettings.creativityLevel === '3' ? 'selected' : ''}>3 - Creative</option>
                            <option value="4" ${this.aiSettings.creativityLevel === '4' ? 'selected' : ''}>4 - Highly Creative</option>
                            <option value="5" ${this.aiSettings.creativityLevel === '5' ? 'selected' : ''}>5 - Extremely Imaginative</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="memoryCapacity">Memory Capacity</label>
                        <select id="memoryCapacity">
                            <option value="1" ${this.aiSettings.memoryCapacity === '1' ? 'selected' : ''}>1 - Short-term</option>
                            <option value="2" ${this.aiSettings.memoryCapacity === '2' ? 'selected' : ''}>2 - Contextual</option>
                            <option value="3" ${this.aiSettings.memoryCapacity === '3' ? 'selected' : ''}>3 - Good Memory</option>
                            <option value="4" ${this.aiSettings.memoryCapacity === '4' ? 'selected' : ''}>4 - Long-term</option>
                            <option value="5" ${this.aiSettings.memoryCapacity === '5' ? 'selected' : ''}>5 - Photographic</option>
                        </select>
                    </div>
                </div>
                
                <div class="settings-section">
                    <h4>Interaction Style</h4>
                    <div class="setting-item">
                        <label for="responseSpeed">Response Speed</label>
                        <select id="responseSpeed">
                            <option value="fast" ${this.aiSettings.responseSpeed === 'fast' ? 'selected' : ''}>Fast</option>
                            <option value="normal" ${this.aiSettings.responseSpeed === 'normal' ? 'selected' : ''}>Normal</option>
                            <option value="thoughtful" ${this.aiSettings.responseSpeed === 'thoughtful' ? 'selected' : ''}>Thoughtful</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="verbosity">Verbosity Level</label>
                        <select id="verbosity">
                            <option value="concise" ${this.aiSettings.verbosity === 'concise' ? 'selected' : ''}>Concise</option>
                            <option value="balanced" ${this.aiSettings.verbosity === 'balanced' ? 'selected' : ''}>Balanced</option>
                            <option value="detailed" ${this.aiSettings.verbosity === 'detailed' ? 'selected' : ''}>Detailed</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="humorLevel">Humor Level</label>
                        <select id="humorLevel">
                            <option value="0" ${this.aiSettings.humorLevel === '0' ? 'selected' : ''}>None</option>
                            <option value="1" ${this.aiSettings.humorLevel === '1' ? 'selected' : ''}>1 - Subtle</option>
                            <option value="2" ${this.aiSettings.humorLevel === '2' ? 'selected' : ''}>2 - Moderate</option>
                            <option value="3" ${this.aiSettings.humorLevel === '3' ? 'selected' : ''}>3 - Funny</option>
                            <option value="4" ${this.aiSettings.humorLevel === '4' ? 'selected' : ''}>4 - Very Humorous</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="formalityLevel">Formality Level</label>
                        <select id="formalityLevel">
                            <option value="1" ${this.aiSettings.formalityLevel === '1' ? 'selected' : ''}>1 - Very Casual</option>
                            <option value="2" ${this.aiSettings.formalityLevel === '2' ? 'selected' : ''}>2 - Casual</option>
                            <option value="3" ${this.aiSettings.formalityLevel === '3' ? 'selected' : ''}>3 - Neutral</option>
                            <option value="4" ${this.aiSettings.formalityLevel === '4' ? 'selected' : ''}>4 - Formal</option>
                            <option value="5" ${this.aiSettings.formalityLevel === '5' ? 'selected' : ''}>5 - Very Formal</option>
                        </select>
                    </div>
                </div>
                
                <div class="settings-section">
                    <h4>Knowledge & Capabilities</h4>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="internetAccess" ${this.aiSettings.internetAccess ? 'checked' : ''}>
                        <label for="internetAccess">Unrestricted Web Access</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="realTimeData" ${this.aiSettings.realTimeData ? 'checked' : ''}>
                        <label for="realTimeData">Real-time Information</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="multilingual" ${this.aiSettings.multilingual ? 'checked' : ''}>
                        <label for="multilingual">Multilingual Support</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="technicalKnowledge" ${this.aiSettings.technicalKnowledge ? 'checked' : ''}>
                        <label for="technicalKnowledge">Technical Knowledge</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="creativeWriting" ${this.aiSettings.creativeWriting ? 'checked' : ''}>
                        <label for="creativeWriting">Creative Writing</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="analyticalSkills" ${this.aiSettings.analyticalSkills ? 'checked' : ''}>
                        <label for="analyticalSkills">Analytical Skills</label>
                    </div>
                </div>
                
                <div class="settings-actions">
                    <button type="button" class="btn-secondary" onclick="closeSettings()">Cancel</button>
                    <button type="button" class="btn-primary" onclick="chat.saveAISettings()">Save Settings</button>
                </div>
            </div>
        `;
    }
    
    attachSettingsEventListeners() {
        // Add event listeners for settings changes
        const settings = this.aiSettings;
        Object.keys(settings).forEach(key => {
            const element = document.getElementById(key);
            if (element) {
                if (element.type === 'checkbox') {
                    element.addEventListener('change', (e) => {
                        this.aiSettings[key] = e.target.checked;
                    });
                } else {
                    element.addEventListener('change', (e) => {
                        this.aiSettings[key] = e.target.value;
                    });
                }
            }
        });
    }
    
    saveAISettings() {
        // Save settings to localStorage
        localStorage.setItem('prime_ai_settings', JSON.stringify(this.aiSettings));
        this.showToast('AI Settings saved successfully!', 'success');
        closeSettings();
    }
    
    // Gamification
    updateUserCoins(coins) {
        // In a real app, this would make an API call to update the user's coins
        console.log(`User earned ${coins} coins`);
        // Update UI accordingly
        const userCoins = document.querySelector('.user-coins span');
        if (userCoins) {
            const currentCoins = parseInt(userCoins.textContent) || 0;
            userCoins.textContent = (currentCoins + coins) + ' coins';
        }
    }
    
    // Utility Functions
    showToast(message, type = 'info') {
        // Create toast element
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;
        toast.innerHTML = `
            <div class="toast-content">
                <i class="mdi mdi-${this.getToastIcon(type)}"></i>
                <span>${message}</span>
            </div>
        `;
        
        document.body.appendChild(toast);
        
        // Remove toast after delay
        setTimeout(() => {
            toast.remove();
        }, 3000);
    }
    
    getToastIcon(type) {
        const icons = {
            success: 'check-circle',
            error: 'alert-circle',
            warning: 'alert',
            info: 'information'
        };
        return icons[type] || 'information';
    }
    
    toggleTheme() {
        const currentTheme = document.body.getAttribute('data-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        
        document.body.setAttribute('data-theme', newTheme);
        localStorage.setItem('prime_ai_theme', newTheme);
        
        const themeIcon = document.querySelector('#themeToggle i');
        if (themeIcon) {
            themeIcon.className = newTheme === 'dark' ? 'mdi mdi-weather-night' : 'mdi mdi-weather-sunny';
        }
    }
    
    exportConversation() {
        if (!this.currentConversation || this.currentConversation.messages.length === 0) {
            this.showToast('No conversation to export', 'warning');
            return;
        }
        
        const conversationText = this.currentConversation.messages.map(msg => 
            `${msg.type === 'user' ? 'You' : this.currentAgent.name}: ${msg.content}`
        ).join('\n\n');
        
        const blob = new Blob([conversationText], { type: 'text/plain' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `conversation-${this.currentConversation.id}.txt`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        this.showToast('Conversation exported successfully!', 'success');
    }
}

// Global functions for HTML onclick attributes
function openProfileModal() {
    document.querySelector('#profileModal').style.display = 'block';
}

function closeProfileModal() {
    document.querySelector('#profileModal').style.display = 'none';
}

function openSettings() {
    document.querySelector('#settingsModal').style.display = 'block';
    if (window.chat) {
        window.chat.loadAISettings();
    }
}

function closeSettings() {
    document.querySelector('#settingsModal').style.display = 'none';
}

function startNewChat() {
    if (window.chat && window.chat.currentAgent) {
        window.chat.startNewConversation(window.chat.currentAgent);
    } else {
        if (window.chat) {
            window.chat.showToast('Please select an AI agent first', 'warning');
        }
    }
}

function toggleVoiceRecording() {
    if (window.chat) {
        window.chat.toggleVoiceRecording();
    }
}

function stopVoiceRecording() {
    if (window.chat) {
        window.chat.stopVoiceRecording();
    }
}

function sendMessage() {
    if (window.chat) {
        window.chat.sendMessage();
    }
}

function triggerImageUpload() {
    document.querySelector('#imageUpload').click();
}

function handleImageUpload(file) {
    if (file) {
        if (window.chat) {
            window.chat.showToast('Image uploaded: ' + file.name, 'info');
        }
        // Handle image processing and sending
    }
}

function previewAvatar(input) {
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        reader.onload = function(e) {
            const preview = document.querySelector('#avatarPreview');
            if (preview) {
                preview.innerHTML = `<img src="${e.target.result}" alt="Avatar Preview">`;
            }
        };
        reader.readAsDataURL(input.files[0]);
    }
}

// Initialize chat when DOM is loaded
let chat;
document.addEventListener('DOMContentLoaded', function() {
    chat = new PrimeAIChat();
    window.chat = chat;
    
    // Load saved theme
    const savedTheme = localStorage.getItem('prime_ai_theme') || 'light';
    document.body.setAttribute('data-theme', savedTheme);
    
    const themeIcon = document.querySelector('#themeToggle i');
    if (themeIcon) {
        themeIcon.className = savedTheme === 'dark' ? 'mdi mdi-weather-night' : 'mdi mdi-weather-sunny';
    }
    
    // Load saved AI settings
    const savedSettings = localStorage.getItem('prime_ai_settings');
    if (savedSettings) {
        try {
            chat.aiSettings = { ...chat.aiSettings, ...JSON.parse(savedSettings) };
        } catch (e) {
            console.error('Error loading saved settings:', e);
        }
    }
});