<?php
// chat.php - Main Chat Interface
require_once 'config.php';

// Check authentication
if (!isset($_SESSION['authenticated']) || $_SESSION['authenticated'] !== true) {
    header('Location: index.php');
    exit;
}

$userID = $_SESSION['user_id'];
$user = getUserByID($userID);

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_profile'])) {
    $username = $_POST['username'] ?? '';
    $age = $_POST['age'] ?? '';
    $bio = $_POST['bio'] ?? '';
    $avatar = null;
    
    if (isset($_FILES['avatar']) && $_FILES['avatar']['error'] === UPLOAD_ERR_OK) {
        $avatar = handleFileUpload($_FILES['avatar']);
    }
    
    if (updateUserProfile($userID, $username, $age, $bio, $avatar)) {
        $user = getUserByID($userID); // Refresh user data
        $success = "Profile updated successfully!";
    } else {
        $error = "Failed to update profile.";
    }
}

// Get AI Agents
$pdo = getDBConnection();
$stmt = $pdo->prepare("SELECT * FROM ai_agents WHERE is_active = TRUE ORDER BY is_default DESC, name ASC");
$stmt->execute();
$aiAgents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get default agent for initial load
$defaultAgent = $aiAgents[0] ?? null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PRIME-AI CHAT - Intelligent Tutoring Platform</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@6.9.96/css/materialdesignicons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div class="writer-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <i class="mdi mdi-robot-excited-outline"></i>
                <h2>PRIME-AI</h2>
                <button class="sidebar-close-btn">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            
            <!-- User Profile Section -->
            <div class="user-profile-section">
                <div class="user-profile-card">
                    <?php if (isset($user['avatar']) && $user['avatar']): ?>
                        <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="User Avatar" class="user-avatar-large">
                    <?php else: ?>
                        <div class="user-avatar-large default-avatar">
                            <i class="mdi mdi-account"></i>
                        </div>
                    <?php endif; ?>
                    <div class="user-info">
                        <h3><?php echo htmlspecialchars($user['username'] ?? 'User ' . $userID); ?></h3>
                        <p class="user-title"><?php echo htmlspecialchars($user['title'] ?? 'Beginner'); ?></p>
                        <div class="user-coins">
                            <i class="mdi mdi-coin"></i>
                            <span><?php echo $user['coins'] ?? 0; ?> coins</span>
                        </div>
                    </div>
                </div>
                
                <!-- Profile Edit Button -->
                <button class="edit-profile-btn" onclick="openProfileModal()">
                    <i class="mdi mdi-account-edit"></i>
                    Edit Profile
                </button>
            </div>
            
            <!-- AI Agents Section -->
            <div class="ai-agents-section">
                <div class="section-header">
                    <span>AI Tutors</span>
                    <i class="mdi mdi-chevron-down"></i>
                </div>
                <div class="ai-agents-list">
                    <?php foreach ($aiAgents as $agent): ?>
                        <div class="ai-agent-item" data-agent-id="<?php echo $agent['id']; ?>">
                            <div class="agent-avatar-small">
                                <img src="<?php echo htmlspecialchars($agent['avatar']); ?>" alt="<?php echo htmlspecialchars($agent['name']); ?>" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
                            </div>
                            <div class="agent-info-small">
                                <div class="agent-name"><?php echo htmlspecialchars($agent['name']); ?></div>
                                <div class="agent-title"><?php echo htmlspecialchars($agent['location']); ?></div>
                            </div>
                            <div class="agent-status-indicator online"></div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Conversations Section -->
            <div class="saved-articles-section">
                <div class="section-header">
                    <span>Recent Chats</span>
                    <i class="mdi mdi-chevron-down"></i>
                </div>
                <div class="saved-articles-list" id="conversationsList">
                    <div class="no-conversations">No conversations yet</div>
                </div>
            </div>
            
            <!-- App Controls -->
            <div class="sidebar-controls">
                <button class="new-chat-btn" onclick="startNewChat()">
                    <i class="mdi mdi-plus-circle"></i>
                    New Chat
                </button>
                <button class="settings-btn" onclick="openSettings()">
                    <i class="mdi mdi-cog"></i>
                    AI Settings
                </button>
                <a href="logout.php" class="logout-btn">
                    <i class="mdi mdi-logout"></i>
                    Logout
                </a>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <!-- Header -->
            <div class="header">
                <button class="mobile-menu-btn">
                    <i class="mdi mdi-menu"></i>
                </button>
                <div class="header-title">
                    <h1>AI Tutor</h1>
                    <span class="subtitle">Intelligent Learning Companion</span>
                </div>
                <div class="header-actions">
                    <button class="theme-toggle" id="themeToggle">
                        <i class="mdi mdi-weather-sunny"></i>
                    </button>
                    <div class="user-menu">
                        <div class="user-avatar">
                            <?php if (isset($user['avatar']) && $user['avatar']): ?>
                                <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="User Avatar">
                            <?php else: ?>
                                <i class="mdi mdi-account"></i>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Chat Area -->
            <div class="chat-container">
                <!-- Chat Messages Area -->
                <div class="chat-messages-area">
                    <div class="chat-header">
                        <div class="active-agent">
                            <div class="agent-avatar-small">
                                <img src="<?php echo $defaultAgent ? htmlspecialchars($defaultAgent['avatar']) : 'https://randomuser.me/api/portraits/women/1.jpg'; ?>" alt="Active Agent" id="activeAgentAvatar" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
                            </div>
                            <div class="agent-info">
                                <h3 id="activeAgentName"><?php echo $defaultAgent ? htmlspecialchars($defaultAgent['name']) : 'Select an AI Tutor'; ?></h3>
                                <span class="agent-status" id="agentStatus"><?php echo $defaultAgent ? 'Online' : 'Offline'; ?></span>
                            </div>
                        </div>
                        <div class="chat-actions">
                            <button class="chat-action-btn" onclick="clearCurrentChat()" title="Clear Chat">
                                <i class="mdi mdi-delete-sweep"></i>
                            </button>
                            <button class="chat-action-btn" onclick="exportConversation()" title="Export Chat">
                                <i class="mdi mdi-export"></i>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Chat Messages -->
                    <div class="chat-messages" id="chatMessages">
                        <div class="welcome-message">
                            <div class="message agent-message">
                                <div class="message-avatar">
                                    <img src="<?php echo $defaultAgent ? htmlspecialchars($defaultAgent['avatar']) : 'https://randomuser.me/api/portraits/women/1.jpg'; ?>" alt="Prime Tutor" onerror="this.src='https://randomuser.me/api/portraits/women/1.jpg'">
                                </div>
                                <div class="message-content">
                                    <div class="message-text">
                                        <p>👋 Welcome to <strong>PRIME-AI CHAT</strong>! I'm your intelligent tutoring companion.</p>
                                        <p>Select an AI Tutor from the sidebar to begin your learning journey, or start with me as your Prime Tutor!</p>
                                        <p>What would you like to learn or discuss today?</p>
                                    </div>
                                    <div class="message-time">Just now</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Message Input -->
                    <div class="message-input-container">
                        <div class="input-actions">
                            <button class="input-action-btn" onclick="toggleVoiceRecording()" title="Voice Message">
                                <i class="mdi mdi-microphone"></i>
                            </button>
                            <button class="input-action-btn" onclick="triggerImageUpload()" title="Upload Image">
                                <i class="mdi mdi-image"></i>
                            </button>
                            <input type="file" id="imageUpload" accept="image/*" style="display: none;" onchange="handleImageUpload(this.files[0])">
                        </div>
                        <div class="message-input-wrapper">
                            <textarea id="messageInput" placeholder="Type your message here..." rows="1"></textarea>
                            <button class="send-button" onclick="sendMessage()" id="sendButton">
                                <i class="mdi mdi-send"></i>
                            </button>
                        </div>
                    </div>
                    
                    <!-- Voice Recording Indicator -->
                    <div class="voice-recording-indicator" id="voiceRecordingIndicator" style="display: none;">
                        <div class="recording-animation">
                            <i class="mdi mdi-record-circle"></i>
                            <span>Recording...</span>
                        </div>
                        <button class="stop-recording-btn" onclick="stopVoiceRecording()">
                            <i class="mdi mdi-stop"></i>
                            Stop
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Profile Modal -->
    <div class="modal" id="profileModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Edit Profile</h3>
                <button class="close-modal" onclick="closeProfileModal()">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <form method="POST" enctype="multipart/form-data" class="modal-body">
                <input type="hidden" name="update_profile" value="1">
                <div class="form-group">
                    <label class="form-label">Avatar</label>
                    <div class="avatar-upload">
                        <div class="avatar-preview" id="avatarPreview">
                            <?php if (isset($user['avatar']) && $user['avatar']): ?>
                                <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Current Avatar">
                            <?php else: ?>
                                <i class="mdi mdi-account"></i>
                            <?php endif; ?>
                        </div>
                        <input type="file" name="avatar" id="avatarInput" accept="image/*" onchange="previewAvatar(this)">
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Username</label>
                    <input type="text" name="username" class="form-input" value="<?php echo htmlspecialchars($user['username'] ?? ''); ?>" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Age</label>
                    <input type="number" name="age" class="form-input" value="<?php echo htmlspecialchars($user['age'] ?? ''); ?>">
                </div>
                <div class="form-group">
                    <label class="form-label">Bio</label>
                    <textarea name="bio" class="form-textarea" rows="3"><?php echo htmlspecialchars($user['bio'] ?? ''); ?></textarea>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-secondary" onclick="closeProfileModal()">Cancel</button>
                    <button type="submit" class="btn-primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- AI Settings Modal -->
    <div class="modal" id="settingsModal">
        <div class="modal-content large-modal">
            <div class="modal-header">
                <h3>AI Personality Settings</h3>
                <button class="close-modal" onclick="closeSettings()">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <div class="modal-body" id="aiSettingsContent">
                <div class="loading-settings">
                    <i class="mdi mdi-loading mdi-spin"></i>
                    <p>Loading AI Settings...</p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Overlay -->
    <div class="sidebar-overlay" id="sidebarOverlay"></div>
    
    <script src="chat.js"></script>
</body>
</html>