<?php
// inukan.php - Admin Panel
require_once 'config.php';

// Check admin authentication
session_start();

if (!isset($_SESSION['admin_authenticated']) || $_SESSION['admin_authenticated'] !== true) {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['username']) && isset($_POST['password'])) {
        if (authenticateAdmin($_POST['username'], $_POST['password'])) {
            $_SESSION['admin_authenticated'] = true;
            $_SESSION['admin_username'] = $_POST['username'];
            header('Location: inukan.php');
            exit;
        } else {
            $error = "Invalid admin credentials!";
        }
    }
    
    // Show login form
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>PRIME-AI CHAT - Admin Login</title>
        <link rel="stylesheet" href="styles.css">
        <link href="https://cdn.jsdelivr.net/npm/@mdi/font@6.9.96/css/materialdesignicons.min.css" rel="stylesheet">
    </head>
    <body class="auth-body">
        <div class="auth-container">
            <div class="auth-card admin-login">
                <div class="auth-header">
                    <div class="logo">
                        <i class="mdi mdi-shield-account"></i>
                        <h1>PRIME-AI ADMIN</h1>
                    </div>
                    <p class="tagline">Administrative Control Panel</p>
                </div>
                
                <div class="auth-content">
                    <?php if (isset($error)): ?>
                        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
                    <?php endif; ?>
                    
                    <form method="POST" class="token-form">
                        <div class="input-group">
                            <i class="mdi mdi-account"></i>
                            <input type="text" name="username" placeholder="Admin Username" required>
                        </div>
                        <div class="input-group">
                            <i class="mdi mdi-lock"></i>
                            <input type="password" name="password" placeholder="Admin Password" required>
                        </div>
                        <button type="submit" class="auth-btn">
                            <i class="mdi mdi-login"></i>
                            Access Admin Panel
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Admin is authenticated, show admin panel
$pdo = getDBConnection();

// Handle admin actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_user'])) {
        $userID = $_POST['user_id'];
        $username = $_POST['username'];
        
        $stmt = $pdo->prepare("INSERT INTO users (user_id, username) VALUES (?, ?)");
        if ($stmt->execute([$userID, $username])) {
            $success = "User added successfully!";
        } else {
            $error = "Failed to add user.";
        }
    }
    
    if (isset($_POST['add_agent'])) {
        $name = $_POST['name'];
        $avatar = $_POST['avatar'];
        $age = $_POST['age'];
        $gender = $_POST['gender'];
        $location = $_POST['location'];
        $personality = $_POST['personality'];
        
        $stmt = $pdo->prepare("INSERT INTO ai_agents (name, avatar, age, gender, location, personality) VALUES (?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$name, $avatar, $age, $gender, $location, $personality])) {
            $success = "AI Agent added successfully!";
        } else {
            $error = "Failed to add AI Agent.";
        }
    }
    
    if (isset($_POST['update_settings'])) {
        foreach ($_POST['settings'] as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO admin_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute([$key, $value, $value]);
        }
        $success = "Settings updated successfully!";
    }
    
    // Handle user updates
    if (isset($_POST['update_user'])) {
        $user_id = $_POST['user_id'];
        $username = $_POST['username'];
        $coins = $_POST['coins'];
        $title = $_POST['title'];
        
        $stmt = $pdo->prepare("UPDATE users SET username = ?, coins = ?, title = ? WHERE id = ?");
        if ($stmt->execute([$username, $coins, $title, $user_id])) {
            $success = "User updated successfully!";
        } else {
            $error = "Failed to update user.";
        }
    }
    
    // Handle agent updates
    if (isset($_POST['update_agent'])) {
        $agent_id = $_POST['agent_id'];
        $name = $_POST['name'];
        $avatar = $_POST['avatar'];
        $age = $_POST['age'];
        $gender = $_POST['gender'];
        $location = $_POST['location'];
        $personality = $_POST['personality'];
        
        $stmt = $pdo->prepare("UPDATE ai_agents SET name = ?, avatar = ?, age = ?, gender = ?, location = ?, personality = ? WHERE id = ?");
        if ($stmt->execute([$name, $avatar, $age, $gender, $location, $personality, $agent_id])) {
            $success = "AI Agent updated successfully!";
        } else {
            $error = "Failed to update AI Agent.";
        }
    }
}

// Get data for display
$users = $pdo->query("SELECT * FROM users ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
$aiAgents = $pdo->query("SELECT * FROM ai_agents ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
$conversations = $pdo->query("
    SELECT c.*, u.username, a.name as agent_name 
    FROM conversations c 
    LEFT JOIN users u ON c.user_id = u.id 
    LEFT JOIN ai_agents a ON c.ai_agent_id = a.id 
    ORDER BY c.updated_at DESC 
    LIMIT 50
")->fetchAll(PDO::FETCH_ASSOC);

// Get message counts for conversations
foreach ($conversations as &$conv) {
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM messages WHERE conversation_id = ?");
    $stmt->execute([$conv['id']]);
    $conv['message_count'] = $stmt->fetchColumn();
}

// Fix for settings query
$settingsResult = $pdo->query("SELECT setting_key, setting_value FROM admin_settings")->fetchAll(PDO::FETCH_ASSOC);
$settings = [];
foreach ($settingsResult as $row) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PRIME-AI CHAT - Admin Panel</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@6.9.96/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body class="admin-body">
    <div class="admin-container">
        <!-- Admin Header -->
        <div class="admin-header">
            <div class="admin-header-left">
                <i class="mdi mdi-shield-account"></i>
                <h1>PRIME-AI ADMIN PANEL</h1>
            </div>
            <div class="admin-header-right">
                <span>Welcome, <?php echo htmlspecialchars($_SESSION['admin_username']); ?></span>
                <a href="logout.php" class="logout-btn">
                    <i class="mdi mdi-logout"></i>
                    Logout
                </a>
            </div>
        </div>
        
        <!-- Admin Navigation -->
        <div class="admin-nav">
            <button class="nav-btn active" data-tab="dashboard">Dashboard</button>
            <button class="nav-btn" data-tab="users">Users</button>
            <button class="nav-btn" data-tab="agents">AI Agents</button>
            <button class="nav-btn" data-tab="conversations">Conversations</button>
            <button class="nav-btn" data-tab="settings">Settings</button>
        </div>
        
        <!-- Dashboard Tab -->
        <div class="admin-tab active" id="dashboard-tab">
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-account-group"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo count($users); ?></h3>
                        <p>Total Users</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-robot"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo count($aiAgents); ?></h3>
                        <p>AI Agents</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-message-text"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo count($conversations); ?></h3>
                        <p>Conversations</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-coin"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php 
                            $totalCoins = 0;
                            foreach ($users as $user) {
                                $totalCoins += (int)$user['coins'];
                            }
                            echo $totalCoins;
                        ?></h3>
                        <p>Total Coins</p>
                    </div>
                </div>
            </div>
            
            <div class="recent-activity">
                <h3>Recent Activity</h3>
                <div class="activity-list">
                    <?php foreach (array_slice($conversations, 0, 10) as $conv): ?>
                        <div class="activity-item">
                            <div class="activity-icon">
                                <i class="mdi mdi-message-text"></i>
                            </div>
                            <div class="activity-info">
                                <p><strong><?php echo htmlspecialchars($conv['username'] ?? 'Unknown User'); ?></strong> chatted with <strong><?php echo htmlspecialchars($conv['agent_name'] ?? 'Unknown Agent'); ?></strong></p>
                                <span class="activity-time"><?php echo date('M j, Y g:i A', strtotime($conv['updated_at'])); ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    <?php if (empty($conversations)): ?>
                        <div class="no-activity">
                            <p>No recent activity</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <!-- Users Tab -->
        <div class="admin-tab" id="users-tab">
            <div class="tab-header">
                <h3>User Management</h3>
                <button class="btn-primary" onclick="openModal('addUserModal')">
                    <i class="mdi mdi-plus"></i>
                    Add User
                </button>
            </div>
            
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User ID</th>
                            <th>Username</th>
                            <th>Coins</th>
                            <th>Title</th>
                            <th>Last Login</th>
                            <th>Created</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo $user['id']; ?></td>
                                <td><?php echo htmlspecialchars($user['user_id']); ?></td>
                                <td><?php echo htmlspecialchars($user['username']); ?></td>
                                <td><?php echo $user['coins']; ?></td>
                                <td><?php echo htmlspecialchars($user['title']); ?></td>
                                <td><?php echo $user['last_login'] ? date('M j, Y g:i A', strtotime($user['last_login'])) : 'Never'; ?></td>
                                <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <button class="btn-small" onclick="editUser(<?php echo $user['id']; ?>)" title="Edit">
                                        <i class="mdi mdi-pencil"></i>
                                    </button>
                                    <button class="btn-small btn-danger" onclick="deleteUser(<?php echo $user['id']; ?>)" title="Delete">
                                        <i class="mdi mdi-delete"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- AI Agents Tab -->
        <div class="admin-tab" id="agents-tab">
            <div class="tab-header">
                <h3>AI Agent Management</h3>
                <button class="btn-primary" onclick="openModal('addAgentModal')">
                    <i class="mdi mdi-plus"></i>
                    Add AI Agent
                </button>
            </div>
            
            <div class="agents-grid">
                <?php foreach ($aiAgents as $agent): ?>
                    <div class="agent-card">
                        <div class="agent-card-header">
                            <img src="<?php echo htmlspecialchars($agent['avatar']); ?>" alt="<?php echo htmlspecialchars($agent['name']); ?>" class="agent-avatar">
                            <div class="agent-info">
                                <h4><?php echo htmlspecialchars($agent['name']); ?></h4>
                                <p><?php echo htmlspecialchars($agent['location']); ?></p>
                            </div>
                            <div class="agent-status <?php echo $agent['is_active'] ? 'active' : 'inactive'; ?>">
                                <?php echo $agent['is_active'] ? 'Active' : 'Inactive'; ?>
                            </div>
                        </div>
                        <div class="agent-card-body">
                            <p><strong>Age:</strong> <?php echo htmlspecialchars($agent['age']); ?></p>
                            <p><strong>Gender:</strong> <?php echo htmlspecialchars($agent['gender']); ?></p>
                            <p class="personality-preview"><?php echo substr(htmlspecialchars($agent['personality']), 0, 100); ?>...</p>
                        </div>
                        <div class="agent-card-actions">
                            <button class="btn-small" onclick="editAgent(<?php echo $agent['id']; ?>)" title="Edit">
                                <i class="mdi mdi-pencil"></i>
                            </button>
                            <button class="btn-small btn-danger" onclick="deleteAgent(<?php echo $agent['id']; ?>)" title="Delete">
                                <i class="mdi mdi-delete"></i>
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
                <?php if (empty($aiAgents)): ?>
                    <div class="no-agents">
                        <p>No AI agents found. Add your first agent!</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Conversations Tab -->
        <div class="admin-tab" id="conversations-tab">
            <div class="tab-header">
                <h3>Conversation Monitoring</h3>
            </div>
            
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>User</th>
                            <th>AI Agent</th>
                            <th>Title</th>
                            <th>Last Updated</th>
                            <th>Message Count</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($conversations as $conv): ?>
                            <tr>
                                <td><?php echo $conv['id']; ?></td>
                                <td><?php echo htmlspecialchars($conv['username'] ?? 'Unknown User'); ?></td>
                                <td><?php echo htmlspecialchars($conv['agent_name'] ?? 'Unknown Agent'); ?></td>
                                <td><?php echo htmlspecialchars($conv['title'] ?? 'Untitled Conversation'); ?></td>
                                <td><?php echo date('M j, Y g:i A', strtotime($conv['updated_at'])); ?></td>
                                <td><?php echo $conv['message_count']; ?></td>
                                <td>
                                    <button class="btn-small" onclick="viewConversation(<?php echo $conv['id']; ?>)" title="View">
                                        <i class="mdi mdi-eye"></i>
                                    </button>
                                    <button class="btn-small btn-danger" onclick="deleteConversation(<?php echo $conv['id']; ?>)" title="Delete">
                                        <i class="mdi mdi-delete"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if (empty($conversations)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center; padding: 2rem;">
                                    <p>No conversations found</p>
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Settings Tab -->
        <div class="admin-tab" id="settings-tab">
            <div class="tab-header">
                <h3>System Settings</h3>
            </div>
            
            <form method="POST" class="settings-form">
                <input type="hidden" name="update_settings" value="1">
                
                <div class="settings-section">
                    <h4>Gamification Settings</h4>
                    <div class="form-group">
                        <label class="form-label">Coins per Question</label>
                        <input type="number" name="settings[coins_per_question]" value="<?php echo $settings['coins_per_question'] ?? 2; ?>" class="form-input" min="0">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Coins per Correct Answer</label>
                        <input type="number" name="settings[coins_per_correct_answer]" value="<?php echo $settings['coins_per_correct_answer'] ?? 5; ?>" class="form-input" min="0">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Coins Deduction per Miss</label>
                        <input type="number" name="settings[coins_deduction_per_miss]" value="<?php echo $settings['coins_deduction_per_miss'] ?? 3; ?>" class="form-input" min="0">
                    </div>
                </div>
                
                <div class="settings-section">
                    <h4>API Settings</h4>
                    <div class="form-group">
                        <label class="form-label">Gemini API Key</label>
                        <input type="password" name="settings[gemini_api_key]" value="<?php echo GEMINI_API_KEY; ?>" class="form-input" readonly>
                        <small class="form-help">Configured in config.php - Not editable here</small>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn-primary">
                        <i class="mdi mdi-content-save"></i>
                        Save Settings
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Add User Modal -->
    <div class="modal" id="addUserModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New User</h3>
                <button class="close-modal" onclick="closeModal('addUserModal')">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <form method="POST" class="modal-body">
                <input type="hidden" name="add_user" value="1">
                <div class="form-group">
                    <label class="form-label">User ID</label>
                    <input type="text" name="user_id" class="form-input" required placeholder="e.g., PRIME12345">
                </div>
                <div class="form-group">
                    <label class="form-label">Username</label>
                    <input type="text" name="username" class="form-input" required placeholder="User display name">
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('addUserModal')">Cancel</button>
                    <button type="submit" class="btn-primary">Add User</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Edit User Modal -->
    <div class="modal" id="editUserModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Edit User</h3>
                <button class="close-modal" onclick="closeModal('editUserModal')">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <form method="POST" class="modal-body">
                <input type="hidden" name="update_user" value="1">
                <input type="hidden" name="user_id" id="editUserId">
                <div class="form-group">
                    <label class="form-label">Username</label>
                    <input type="text" name="username" id="editUsername" class="form-input" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Coins</label>
                    <input type="number" name="coins" id="editCoins" class="form-input" min="0">
                </div>
                <div class="form-group">
                    <label class="form-label">Title</label>
                    <input type="text" name="title" id="editTitle" class="form-input">
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('editUserModal')">Cancel</button>
                    <button type="submit" class="btn-primary">Update User</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Add Agent Modal -->
    <div class="modal" id="addAgentModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add New AI Agent</h3>
                <button class="close-modal" onclick="closeModal('addAgentModal')">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <form method="POST" class="modal-body">
                <input type="hidden" name="add_agent" value="1">
                <div class="form-group">
                    <label class="form-label">Agent Name</label>
                    <input type="text" name="name" class="form-input" required placeholder="e.g., Prime Tutor">
                </div>
                <div class="form-group">
                    <label class="form-label">Avatar URL</label>
                    <input type="url" name="avatar" class="form-input" required placeholder="https://example.com/avatar.jpg">
                    <small class="form-help">Use a direct image URL</small>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Age</label>
                        <input type="text" name="age" class="form-input" placeholder="e.g., 32">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Gender</label>
                        <select name="gender" class="form-input">
                            <option value="male">Male</option>
                            <option value="female" selected>Female</option>
                            <option value="other">Other</option>
                        </select>
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Location</label>
                    <input type="text" name="location" class="form-input" placeholder="e.g., Nigeria">
                </div>
                <div class="form-group">
                    <label class="form-label">Personality</label>
                    <textarea name="personality" class="form-textarea" rows="6" required placeholder="Describe the agent's personality, behavior, and capabilities..."></textarea>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('addAgentModal')">Cancel</button>
                    <button type="submit" class="btn-primary">Add Agent</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Edit Agent Modal -->
    <div class="modal" id="editAgentModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Edit AI Agent</h3>
                <button class="close-modal" onclick="closeModal('editAgentModal')">
                    <i class="mdi mdi-close"></i>
                </button>
            </div>
            <form method="POST" class="modal-body">
                <input type="hidden" name="update_agent" value="1">
                <input type="hidden" name="agent_id" id="editAgentId">
                <div class="form-group">
                    <label class="form-label">Agent Name</label>
                    <input type="text" name="name" id="editAgentName" class="form-input" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Avatar URL</label>
                    <input type="url" name="avatar" id="editAgentAvatar" class="form-input" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Age</label>
                        <input type="text" name="age" id="editAgentAge" class="form-input">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Gender</label>
                        <select name="gender" id="editAgentGender" class="form-input">
                            <option value="male">Male</option>
                            <option value="female">Female</option>
                            <option value="other">Other</option>
                        </select>
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Location</label>
                    <input type="text" name="location" id="editAgentLocation" class="form-input">
                </div>
                <div class="form-group">
                    <label class="form-label">Personality</label>
                    <textarea name="personality" id="editAgentPersonality" class="form-textarea" rows="6" required></textarea>
                </div>
                <div class="modal-actions">
                    <button type="button" class="btn-secondary" onclick="closeModal('editAgentModal')">Cancel</button>
                    <button type="submit" class="btn-primary">Update Agent</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Success/Error Messages -->
    <?php if (isset($success)): ?>
        <div class="toast toast-success" id="successToast">
            <div class="toast-content">
                <i class="mdi mdi-check-circle"></i>
                <span><?php echo htmlspecialchars($success); ?></span>
            </div>
        </div>
        <script>
            setTimeout(() => {
                const toast = document.getElementById('successToast');
                if (toast) toast.remove();
            }, 5000);
        </script>
    <?php endif; ?>

    <?php if (isset($error)): ?>
        <div class="toast toast-error" id="errorToast">
            <div class="toast-content">
                <i class="mdi mdi-alert-circle"></i>
                <span><?php echo htmlspecialchars($error); ?></span>
            </div>
        </div>
        <script>
            setTimeout(() => {
                const toast = document.getElementById('errorToast');
                if (toast) toast.remove();
            }, 5000);
        </script>
    <?php endif; ?>
    
    <script>
        // Tab navigation
        document.querySelectorAll('.nav-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                const tabName = this.dataset.tab;
                
                // Update active nav button
                document.querySelectorAll('.nav-btn').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                
                // Show selected tab
                document.querySelectorAll('.admin-tab').forEach(tab => tab.classList.remove('active'));
                document.getElementById(tabName + '-tab').classList.add('active');
            });
        });
        
        // Modal functions
        function openModal(modalId) {
            document.getElementById(modalId).style.display = 'block';
        }
        
        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }
        
        // Close modal when clicking outside
        window.addEventListener('click', function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.style.display = 'none';
            }
        });
        
        // User management functions
        async function editUser(userId) {
            try {
                const response = await fetch('admin_actions.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=get_user&user_id=' + userId
                });
                
                const data = await response.json();
                
                if (data.success) {
                    const user = data.user;
                    document.getElementById('editUserId').value = user.id;
                    document.getElementById('editUsername').value = user.username || '';
                    document.getElementById('editCoins').value = user.coins || 0;
                    document.getElementById('editTitle').value = user.title || '';
                    openModal('editUserModal');
                } else {
                    alert('Error: ' + data.error);
                }
            } catch (error) {
                alert('Error: ' + error.message);
            }
        }
        
        async function deleteUser(userId) {
            if (confirm('Are you sure you want to delete this user? This action cannot be undone.')) {
                try {
                    const response = await fetch('admin_actions.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=delete_user&user_id=' + userId
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        alert('User deleted successfully');
                        location.reload();
                    } else {
                        alert('Error deleting user: ' + data.error);
                    }
                } catch (error) {
                    alert('Error: ' + error.message);
                }
            }
        }
        
        // Agent management functions
        async function editAgent(agentId) {
            try {
                const response = await fetch('admin_actions.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: 'action=get_agent&agent_id=' + agentId
                });
                
                const data = await response.json();
                
                if (data.success) {
                    const agent = data.agent;
                    document.getElementById('editAgentId').value = agent.id;
                    document.getElementById('editAgentName').value = agent.name || '';
                    document.getElementById('editAgentAvatar').value = agent.avatar || '';
                    document.getElementById('editAgentAge').value = agent.age || '';
                    document.getElementById('editAgentGender').value = agent.gender || 'female';
                    document.getElementById('editAgentLocation').value = agent.location || '';
                    document.getElementById('editAgentPersonality').value = agent.personality || '';
                    openModal('editAgentModal');
                } else {
                    alert('Error: ' + data.error);
                }
            } catch (error) {
                alert('Error: ' + error.message);
            }
        }
        
        async function deleteAgent(agentId) {
            if (confirm('Are you sure you want to delete this AI agent? This action cannot be undone.')) {
                try {
                    const response = await fetch('admin_actions.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=delete_agent&agent_id=' + agentId
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        alert('AI Agent deleted successfully');
                        location.reload();
                    } else {
                        alert('Error deleting agent: ' + data.error);
                    }
                } catch (error) {
                    alert('Error: ' + error.message);
                }
            }
        }
        
        function viewConversation(convId) {
            alert('View conversation ' + convId + ' - This would show conversation details in a full implementation.');
        }
        
        async function deleteConversation(convId) {
            if (confirm('Are you sure you want to delete this conversation? This action cannot be undone.')) {
                try {
                    const response = await fetch('admin_actions.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=delete_conversation&conversation_id=' + convId
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        alert('Conversation deleted successfully');
                        location.reload();
                    } else {
                        alert('Error deleting conversation: ' + data.error);
                    }
                } catch (error) {
                    alert('Error: ' + error.message);
                }
            }
        }

        // Auto-close toasts after 5 seconds
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(() => {
                const toasts = document.querySelectorAll('.toast');
                toasts.forEach(toast => {
                    toast.remove();
                });
            }, 5000);
        });
    </script>
</body>
</html>