CREATE DATABASE IF NOT EXISTS fuzesale_noteai;
USE fuzesale_noteai;

CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) NOT NULL UNIQUE,
    full_name VARCHAR(100) NOT NULL,
    school VARCHAR(100),
    note_credits INT DEFAULT 2,
    role ENUM('admin', 'teacher') DEFAULT 'teacher',
    status ENUM('active', 'suspended', 'pending') DEFAULT 'active',
    last_login DATETIME,
    login_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE notes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    topic VARCHAR(255) NOT NULL,
    age_group INT NOT NULL,
    content TEXT NOT NULL,
    word_count INT DEFAULT 0,
    ai_generated TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(50) NOT NULL UNIQUE,
    setting_value TEXT NOT NULL,
    description TEXT,
    is_public TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE credit_transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    admin_id INT,
    credits INT NOT NULL,
    type ENUM('purchase', 'reward', 'penalty', 'adjustment', 'system') DEFAULT 'adjustment',
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (admin_id) REFERENCES users(id) ON DELETE SET NULL
);

CREATE TABLE user_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    session_id VARCHAR(255) NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    login_time DATETIME NOT NULL,
    last_activity DATETIME NOT NULL,
    logout_time DATETIME,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE system_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL
);

CREATE TABLE email_templates (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    subject VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    variables TEXT,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

CREATE TABLE announcements (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content TEXT NOT NULL,
    priority ENUM('low', 'medium', 'high') DEFAULT 'medium',
    is_active TINYINT(1) DEFAULT 1,
    start_date DATE,
    end_date DATE,
    created_by INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE announcement_views (
    id INT AUTO_INCREMENT PRIMARY KEY,
    announcement_id INT NOT NULL,
    user_id INT NOT NULL,
    viewed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (announcement_id) REFERENCES announcements(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_announcement_view (announcement_id, user_id)
);

-- Insert default admin account
INSERT INTO users (username, password, email, full_name, school, note_credits, role, status) 
VALUES ('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@vinesubtech.com.ng', 'System Administrator', 'VineSub Tech', 9999, 'admin', 'active');

-- Insert default teacher account
INSERT INTO users (username, password, email, full_name, school, note_credits) 
VALUES ('teacherai', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'teacher@example.com', 'Demo Teacher', 'Demo School', 2);

-- Insert app settings
INSERT INTO settings (setting_key, setting_value, description, is_public) VALUES 
('app_name', 'Vine AI Note Maker - NERDC Compliant', 'The name of the application', 1),
('app_version', '2.0', 'Current application version', 1),
('default_credits', '2', 'Default number of credits for new users', 0),
('max_notes_per_day', '10', 'Maximum number of notes a user can create per day', 0),
('ai_api_key', 'YOUR_GEMINI_API_KEY', 'Gemini AI API Key', 0),
('admin_email', 'admin@vinesubtech.com.ng', 'Administrator email address', 0),
('site_url', 'http://p.vinesubtech.com.ng/noteaiv7', 'Application URL', 1),
('allow_registrations', '1', 'Whether new user registrations are allowed', 1),
('maintenance_mode', '0', 'Put the site in maintenance mode', 1),
('logo_url', '/assets/logo.png', 'Application logo URL', 1),
('currency', '₦', 'Currency symbol', 1),
('credit_price', '500', 'Price per credit in currency units', 1),
('min_credits_purchase', '5', 'Minimum credits that can be purchased', 1),
('max_credits_purchase', '100', 'Maximum credits that can be purchased', 1);

-- Insert email templates
INSERT INTO email_templates (name, subject, content, variables) VALUES
('welcome', 'Welcome to Vine AI Note Maker', 'Hello {user_name},\n\nWelcome to Vine AI Note Maker! Your account has been successfully created.\n\nYou can now login using:\nUsername: {username}\n\nThank you for joining us!\n\nBest regards,\nVineSub Tech Team', 'user_name,username'),
('credit_purchase', 'Credit Purchase Confirmation', 'Hello {user_name},\n\nYour credit purchase has been confirmed. You have received {credits} credits.\n\nYour total available credits: {total_credits}\n\nThank you for your purchase!\n\nBest regards,\nVineSub Tech Team', 'user_name,credits,total_credits'),
('account_suspended', 'Account Suspension Notice', 'Hello {user_name},\n\nYour account has been suspended for the following reason:\n{reason}\n\nIf you believe this is a mistake, please contact support.\n\nBest regards,\nVineSub Tech Team', 'user_name,reason'),
('password_reset', 'Password Reset Request', 'Hello {user_name},\n\nYou requested a password reset. Use this code to reset your password: {reset_code}\n\nIf you didn''t request this, please ignore this email.\n\nBest regards,\nVineSub Tech Team', 'user_name,reset_code');

-- Insert sample announcement
INSERT INTO announcements (title, content, priority, created_by) VALUES
('Welcome to Vine AI Note Maker', 'We are excited to have you on board! This platform will help you create lesson notes quickly and efficiently using AI technology.', 'high', 1);