<?php
require_once 'config.php';

// Handle password reset request
if ($_POST['action'] === 'verify_characters') {
    $email = $_POST['email'];
    $characters = $_POST['characters'];
    
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT id, password FROM users WHERE email = ? AND is_active = TRUE");
        $stmt->execute([$email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            $password = $user['password'];
            $allPresent = true;
            
            // STRICT verification - check if all characters are present in the password (case-sensitive)
            foreach (str_split($characters) as $char) {
                if (strpos($password, $char) === false) {
                    $allPresent = false;
                    break;
                }
            }
            
            if ($allPresent) {
                // Characters are correct, allow password reset
                $_SESSION['reset_email'] = $email;
                $_SESSION['reset_verified'] = true;
                $_SESSION['reset_attempts'] = 0; // Reset attempts on success
                echo json_encode([
                    'success' => true,
                    'message' => 'Security verification successful!'
                ]);
            } else {
                // Track failed attempts
                $_SESSION['reset_attempts'] = ($_SESSION['reset_attempts'] ?? 0) + 1;
                
                // After 2 failed attempts, force contact admin
                if ($_SESSION['reset_attempts'] >= 2) {
                    echo json_encode([
                        'success' => false, 
                        'message' => 'Multiple verification failures detected. For security reasons, please contact the administrator at prudenceeni@gmail.com for assistance.',
                        'contact_admin' => true
                    ]);
                } else {
                    echo json_encode([
                        'success' => false,
                        'message' => 'Security verification failed. One or more characters are not present in your password. Please try again.'
                    ]);
                }
            }
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'No active account found with this email address.'
            ]);
        }
    } catch (Exception $e) {
        error_log("Password reset error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Database error. Please contact administrator at prudenceeni@gmail.com'
        ]);
    }
    exit();
}

// Handle password update
if ($_POST['action'] === 'update_password') {
    if (!isset($_SESSION['reset_verified']) || !$_SESSION['reset_verified']) {
        echo json_encode([
            'success' => false,
            'message' => 'Security verification required. Please complete character verification first.'
        ]);
        exit();
    }
    
    $email = $_SESSION['reset_email'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];
    
    if (strlen($new_password) < 6) {
        echo json_encode([
            'success' => false,
            'message' => 'Password must be at least 6 characters long.'
        ]);
        exit();
    }
    
    if ($new_password !== $confirm_password) {
        echo json_encode([
            'success' => false,
            'message' => 'Passwords do not match. Please ensure both passwords are identical.'
        ]);
        exit();
    }
    
    try {
        $db = getDB();
        $hashedPassword = password_hash($new_password, PASSWORD_DEFAULT);
        $stmt = $db->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE email = ?");
        $stmt->execute([$hashedPassword, $email]);
        
        // Log the password change for security
        error_log("Password reset successful for user: " . $email);
        
        // Clear reset session
        unset($_SESSION['reset_email']);
        unset($_SESSION['reset_verified']);
        unset($_SESSION['reset_attempts']);
        
        echo json_encode([
            'success' => true, 
            'message' => 'Password updated successfully! You can now login with your new password.'
        ]);
    } catch (Exception $e) {
        error_log("Password update error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Failed to update password. Please contact administrator at prudenceeni@gmail.com'
        ]);
    }
    exit();
}

// Clear any existing reset session when page loads
if (!isset($_POST['action'])) {
    unset($_SESSION['reset_email']);
    unset($_SESSION['reset_verified']);
    unset($_SESSION['reset_attempts']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password - Bulk AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.4.47/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body class="auth-page">
    <div class="auth-container">
        <div class="auth-header">
            <h1>🔐 Reset Password</h1>
            <p>Bulk AI Writer Account Recovery</p>
        </div>
        
        <div class="tab-content">
            <!-- Step 1: Email and Characters Verification -->
            <div id="verify-step" class="tab-pane active">
                <form id="verifyForm" class="auth-form">
                    <div class="security-notice">
                        <div class="security-icon">
                            <i class="mdi mdi-shield-lock"></i>
                        </div>
                        <div class="security-message">
                            <h4>Security Verification Required</h4>
                            <p>For your security, please verify your identity by entering characters from your current password.</p>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="reset-email">Email Address</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-email"></i>
                            <input type="email" id="reset-email" name="email" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="characters">Enter 3 characters from your current password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-form-textbox-password"></i>
                            <input type="text" id="characters" name="characters" maxlength="3" required 
                                   placeholder="Enter any 3 characters from your password"
                                   pattern="[A-Za-z0-9!@#$%^&*()_+\-=\[\]{};':\\|,.<>\/?]{3}"
                                   title="Enter exactly 3 characters from your password">
                        </div>
                        <small class="form-hint">Enter any 3 characters that are present in your current password. This is case-sensitive.</small>
                    </div>
                    
                    <button type="submit" class="btn-primary btn-full">
                        <i class="mdi mdi-shield-check"></i> Verify Characters
                    </button>
                    
                    <div class="auth-links">
                        <a href="index.php"><i class="mdi mdi-arrow-left"></i> Back to Login</a>
                    </div>
                </form>
            </div>
            
            <!-- Step 2: New Password -->
            <div id="password-step" class="tab-pane">
                <form id="passwordForm" class="auth-form">
                    <div class="security-notice success">
                        <div class="security-icon">
                            <i class="mdi mdi-check-circle"></i>
                        </div>
                        <div class="security-message">
                            <h4>Verification Successful</h4>
                            <p>You can now set your new password.</p>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="new-password">New Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock-plus"></i>
                            <input type="password" id="new-password" name="new_password" required minlength="6">
                        </div>
                        <small class="form-hint">Password must be at least 6 characters long</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm-password">Confirm New Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock-check"></i>
                            <input type="password" id="confirm-password" name="confirm_password" required>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-primary btn-full">
                        <i class="mdi mdi-key-change"></i> Update Password
                    </button>
                    
                    <div class="auth-links">
                        <a href="#" onclick="showVerifyStep()"><i class="mdi mdi-arrow-left"></i> Back to Verification</a>
                    </div>
                </form>
            </div>
            
            <!-- Contact Admin -->
            <div id="contact-admin" class="tab-pane">
                <div class="security-notice warning">
                    <div class="security-icon">
                        <i class="mdi mdi-alert-circle"></i>
                    </div>
                    <div class="security-message">
                        <h4>Security Verification Failed</h4>
                        <p>For security reasons, further password reset attempts have been disabled.</p>
                    </div>
                </div>
                
                <div class="contact-info">
                    <div class="contact-card">
                        <div class="contact-icon">
                            <i class="mdi mdi-email"></i>
                        </div>
                        <div class="contact-details">
                            <h5>Contact Administrator</h5>
                            <p class="admin-email">prudenceeni@gmail.com</p>
                            <p>Please include your registered email address and account details in your message.</p>
                        </div>
                    </div>
                </div>
                
                <div class="auth-links">
                    <a href="#" onclick="showVerifyStep()"><i class="mdi mdi-refresh"></i> Try Again</a> 
                    <span class="link-separator">|</span>
                    <a href="index.php"><i class="mdi mdi-login"></i> Back to Login</a>
                </div>
            </div>
        </div>
        
        <div id="resetResult" class="install-result"></div>
    </div>

    <script>
        let resetAttempts = 0;
        const maxAttempts = 2;

        function showVerifyStep() {
            document.querySelectorAll('.tab-pane').forEach(pane => pane.classList.remove('active'));
            document.getElementById('verify-step').classList.add('active');
            document.getElementById('resetResult').innerHTML = '';
        }

        function showPasswordStep() {
            document.querySelectorAll('.tab-pane').forEach(pane => pane.classList.remove('active'));
            document.getElementById('password-step').classList.add('active');
            document.getElementById('resetResult').innerHTML = '';
        }

        function showContactAdmin() {
            document.querySelectorAll('.tab-pane').forEach(pane => pane.classList.remove('active'));
            document.getElementById('contact-admin').classList.add('active');
            document.getElementById('resetResult').innerHTML = '';
        }

        function showMessage(message, type = 'info') {
            const resultDiv = document.getElementById('resetResult');
            resultDiv.innerHTML = `<div class="${type}">${message}</div>`;
            
            // Scroll to result
            resultDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }

        document.getElementById('verifyForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const email = document.getElementById('reset-email').value;
            const characters = document.getElementById('characters').value;
            
            if (characters.length !== 3) {
                showMessage('Please enter exactly 3 characters from your password.', 'error');
                return;
            }

            const formData = new FormData(this);
            formData.append('action', 'verify_characters');
            
            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="mdi mdi-loading spinner"></i> Verifying...';
            submitBtn.disabled = true;
            
            fetch('forgot-password.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message, 'success');
                    setTimeout(() => showPasswordStep(), 1500);
                } else {
                    showMessage(data.message, 'error');
                    resetAttempts++;
                    
                    if (data.contact_admin || resetAttempts >= maxAttempts) {
                        setTimeout(() => showContactAdmin(), 2000);
                    }
                }
            })
            .catch(error => {
                showMessage('Verification failed: ' + error, 'error');
            })
            .finally(() => {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        });

        document.getElementById('passwordForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const newPassword = document.getElementById('new-password').value;
            const confirmPassword = document.getElementById('confirm-password').value;
            
            if (newPassword.length < 6) {
                showMessage('Password must be at least 6 characters long.', 'error');
                return;
            }
            
            if (newPassword !== confirmPassword) {
                showMessage('Passwords do not match. Please ensure both passwords are identical.', 'error');
                return;
            }

            const formData = new FormData(this);
            formData.append('action', 'update_password');
            
            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="mdi mdi-loading spinner"></i> Updating...';
            submitBtn.disabled = true;
            
            fetch('forgot-password.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message + ' <a href="index.php" style="color: white; text-decoration: underline;">Click here to login</a>', 'success');
                    document.getElementById('passwordForm').reset();
                    
                    // Auto-redirect to login after 5 seconds
                    setTimeout(() => {
                        window.location.href = 'index.php';
                    }, 5000);
                } else {
                    showMessage(data.message, 'error');
                }
            })
            .catch(error => {
                showMessage('Password update failed: ' + error, 'error');
            })
            .finally(() => {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            });
        });

        // Real-time password confirmation check
        document.getElementById('confirm-password').addEventListener('input', function() {
            const newPassword = document.getElementById('new-password').value;
            const confirmPassword = this.value;
            const confirmField = this.closest('.form-group');
            
            if (confirmPassword.length > 0) {
                if (newPassword === confirmPassword) {
                    confirmField.classList.add('valid');
                    confirmField.classList.remove('invalid');
                } else {
                    confirmField.classList.add('invalid');
                    confirmField.classList.remove('valid');
                }
            } else {
                confirmField.classList.remove('valid', 'invalid');
            }
        });

        // Input validation for characters field
        document.getElementById('characters').addEventListener('input', function(e) {
            this.value = this.value.replace(/[^A-Za-z0-9!@#$%^&*()_+\-=\[\]{};':\\|,.<>\/?]/g, '');
            if (this.value.length > 3) {
                this.value = this.value.slice(0, 3);
            }
        });
    </script>
</body>
</html>