<?php
require_once 'config.php';

// Handle logout
if (isset($_GET['logout'])) {
    session_destroy();
    header("Location: index.php");
    exit();
}

// Handle login - FIXED: Check if POST data exists
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'login') {
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    
    try {
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM users WHERE email = ? AND is_active = TRUE");
        $stmt->execute([$email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['is_admin'] = $user['is_admin'] == 1;
            
            // Redirect to admin panel if admin, otherwise to bulk writer
            if ($_SESSION['is_admin']) {
                header("Location: 123admin.php");
            } else {
                header("Location: bulkwriter.php");
            }
            exit();
        } else {
            $error = "Invalid email or password";
        }
    } catch (PDOException $e) {
        $error = "Database error: " . $e->getMessage();
    }
}

// Handle registration - FIXED: Check if POST data exists
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'register') {
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    if ($password !== $confirm_password) {
        $error = "Passwords do not match";
    } else {
        try {
            $db = getDB();
            $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->fetch()) {
                $error = "Email already registered";
            } else {
                $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $db->prepare("INSERT INTO users (email, password) VALUES (?, ?)");
                $stmt->execute([$email, $hashedPassword]);
                
                $_SESSION['user_id'] = $db->lastInsertId();
                $_SESSION['user_email'] = $email;
                $_SESSION['is_admin'] = false;
                
                // Create default settings for user
                saveUserSettings($_SESSION['user_id'], [
                    'ai_api_key' => getAppSetting('default_ai_api_key')
                ]);
                
                header("Location: bulkwriter.php");
                exit();
            }
        } catch (PDOException $e) {
            $error = "Registration failed: " . $e->getMessage();
        }
    }
}

// Redirect if already logged in
if (isLoggedIn()) {
    if (isAdmin()) {
        header("Location: 123admin.php");
    } else {
        header("Location: bulkwriter.php");
    }
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Bulk AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.4.47/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body class="auth-page">
    <div class="auth-container">
        <div class="auth-header">
            <h1>🚀 Bulk AI Writer</h1>
            <p>Generate high-quality, human-like content at scale</p>
        </div>
        
        <div class="auth-tabs">
            <button class="tab-button active" data-tab="login">Login</button>
            <button class="tab-button" data-tab="register">Register</button>
        </div>
        
        <?php if (isset($error)): ?>
            <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>
        
        <div class="tab-content">
            <!-- Login Form -->
            <div id="login-tab" class="tab-pane active">
                <form method="POST" class="auth-form">
                    <input type="hidden" name="action" value="login">
                    
                    <div class="form-group">
                        <label for="login-email">Email</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-email"></i>
                            <input type="email" id="login-email" name="email" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="login-password">Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock"></i>
                            <input type="password" id="login-password" name="password" required>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-primary btn-full">Login</button>
                    
                    <div class="auth-links">
                        <a href="forgot-password.php">Forgot Password?</a>
                    </div>
                </form>
            </div>
            
            <!-- Register Form -->
            <div id="register-tab" class="tab-pane">
                <form method="POST" class="auth-form">
                    <input type="hidden" name="action" value="register">
                    
                    <div class="form-group">
                        <label for="register-email">Email</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-email"></i>
                            <input type="email" id="register-email" name="email" required>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="register-password">Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock"></i>
                            <input type="password" id="register-password" name="password" required minlength="6">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm-password">Confirm Password</label>
                        <div class="input-with-icon">
                            <i class="mdi mdi-lock-check"></i>
                            <input type="password" id="confirm-password" name="confirm_password" required>
                        </div>
                    </div>
                    
                    <button type="submit" class="btn-primary btn-full">Register</button>
                    
                    <div class="auth-links">
                        <a href="#" onclick="switchTab('login')">Already have an account? Login</a>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="script.js"></script>
</body>
</html>