<?php
// Define the JSON file path
define('JSON_FILE', 'results.json');

// Function to load data from JSON file
function loadData() {
    if (file_exists(JSON_FILE)) {
        $json = file_get_contents(JSON_FILE);
        return json_decode($json, true);
    }
    return [];
}

// Function to save data to JSON file
function saveData($data) {
    file_put_contents(JSON_FILE, json_encode($data, JSON_PRETTY_PRINT));
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        
        if ($action === 'save') {
            // Save the result data
            $studentId = $_POST['student_id'];
            $data = loadData();
            $data[$studentId] = $_POST;
            saveData($data);
            
            echo json_encode(['success' => true, 'message' => 'Result saved successfully!']);
            exit;
        } elseif ($action === 'query') {
            // Query result by student ID
            $studentId = $_POST['student_id'];
            $data = loadData();
            
            if (isset($data[$studentId])) {
                echo json_encode(['success' => true, 'data' => $data[$studentId]]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Result not found for ID: ' . $studentId]);
            }
            exit;
        } elseif ($action === 'delete') {
            // Delete a result
            $studentId = $_POST['student_id'];
            $data = loadData();
            
            if (isset($data[$studentId])) {
                unset($data[$studentId]);
                saveData($data);
                echo json_encode(['success' => true, 'message' => 'Result deleted successfully!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Result not found for ID: ' . $studentId]);
            }
            exit;
        } elseif ($action === 'list') {
            // List all results
            $data = loadData();
            echo json_encode(['success' => true, 'data' => $data]);
            exit;
        }
    }
}

// Check if we're on the result checker page
$isResultChecker = basename($_SERVER['PHP_SELF']) === 'checker.php';

// If we're not on the result checker page, load the editor
if (!$isResultChecker) {
    // Load initial data if available
    $initialData = loadData();
    $defaultStudentId = 'PRIME-C006';
    $initialValues = isset($initialData[$defaultStudentId]) ? $initialData[$defaultStudentId] : [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Result Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #3498db;
            --secondary-color: #2c3e50;
            --accent-color: #e74c3c;
            --light-color: #ecf0f1;
            --success-color: #2ecc71;
            --warning-color: #f39c12;
            --border-radius: 8px;
            --box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: #f5f7fa;
            color: #333;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            padding: 1rem;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            text-align: center;
            box-shadow: var(--box-shadow);
            position: relative;
        }
        
        h1 {
            font-size: 2.2rem;
            margin-bottom: 10px;
        }
        
        .nav-tabs {
            display: flex;
            justify-content: center;
            margin-bottom: 20px;
            background: white;
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: var(--box-shadow);
        }
        
        .nav-tab {
            padding: 15px 30px;
            text-decoration: none;
            color: var(--secondary-color);
            font-weight: bold;
            transition: var(--transition);
            flex: 1;
            text-align: center;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .nav-tab.active {
            background: var(--primary-color);
            color: white;
        }
        
        .nav-tab:hover:not(.active) {
            background: #f0f0f0;
        }
        
        .card {
            background: white;
            border-radius: var(--border-radius);
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: var(--box-shadow);
            transition: var(--transition);
        }
        
        .card:hover {
            box-shadow: 0 6px 12px rgba(0, 0, 0, 0.15);
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: var(--secondary-color);
        }
        
        input, select, textarea {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
            font-size: 16px;
            transition: var(--transition);
        }
        
        input:focus, select:focus, textarea:focus {
            border-color: var(--primary-color);
            outline: none;
            box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.2);
        }
        
        button {
            background: var(--primary-color);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: var(--border-radius);
            cursor: pointer;
            font-size: 16px;
            font-weight: bold;
            transition: var(--transition);
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        button:hover {
            background: #2980b9;
            transform: translateY(-2px);
        }
        
        .btn-print {
            background: var(--secondary-color);
        }
        
        .btn-print:hover {
            background: #1e2a36;
        }
        
        .btn-success {
            background: var(--success-color);
        }
        
        .btn-success:hover {
            background: #27ae60;
        }
        
        .btn-warning {
            background: var(--warning-color);
        }
        
        .btn-warning:hover {
            background: #e67e22;
        }
        
        .btn-danger {
            background: var(--accent-color);
        }
        
        .btn-danger:hover {
            background: #c0392b;
        }
        
        .result-table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            background: white;
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: var(--box-shadow);
        }
        
        .result-table th, .result-table td {
            padding: 12px 15px;
            text-align: left;
            border: 1px solid #ddd;
        }
        
        .result-table th {
            background: var(--secondary-color);
            color: white;
            font-weight: bold;
            position: sticky;
            top: 0;
        }
        
        .result-table tr:nth-child(even) {
            background: #f9f9f9;
        }
        
        .result-table tr:hover {
            background: #f1f1f1;
        }
        
        .editable-cell {
            background: #fffde7;
            width: 60px;
            text-align: center;
        }
        
        .editable-cell:focus {
            background: #fff9c4;
            outline: 2px solid var(--primary-color);
        }
        
        .alert {
            padding: 15px;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .hidden {
            display: none;
        }
        
        .text-center {
            text-align: center;
        }
        
        .student-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .action-buttons {
            display: flex;
            gap: 12px;
            justify-content: center;
            margin-top: 20px;
            flex-wrap: wrap;
        }
        
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }
        
        .stat-card {
            background: white;
            border-radius: var(--border-radius);
            padding: 20px;
            text-align: center;
            box-shadow: var(--box-shadow);
            transition: var(--transition);
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 15px rgba(0, 0, 0, 0.1);
        }
        
        .stat-value {
            font-size: 2.5rem;
            font-weight: bold;
            color: var(--primary-color);
            margin: 10px 0;
        }
        
        .stat-label {
            color: var(--secondary-color);
            font-weight: 600;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: white;
            border-radius: var(--border-radius);
            padding: 25px;
            width: 90%;
            max-width: 500px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }
        
        .close {
            font-size: 24px;
            cursor: pointer;
            color: #999;
            transition: var(--transition);
        }
        
        .close:hover {
            color: var(--accent-color);
        }
        
        .search-box {
            display: flex;
            gap: 12px;
            margin-bottom: 20px;
        }
        
        .search-box input {
            flex: 1;
        }
        
        .student-list {
            max-height: 400px;
            overflow-y: auto;
            border: 1px solid #ddd;
            border-radius: var(--border-radius);
        }
        
        .student-item {
            padding: 15px;
            border-bottom: 1px solid #eee;
            cursor: pointer;
            transition: var(--transition);
        }
        
        .student-item:hover {
            background: #f5f7fa;
        }
        
        .student-item:last-child {
            border-bottom: none;
        }
        
        .print-only {
            display: none;
        }
        
        .theme-selector {
            position: absolute;
            top: 15px;
            right: 15px;
            display: flex;
            gap: 8px;
        }
        
        .theme-btn {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            cursor: pointer;
            border: 2px solid white;
            box-shadow: 0 0 5px rgba(0, 0, 0, 0.2);
        }
        
        .theme-light {
            background: #f5f7fa;
        }
        
        .theme-dark {
            background: #2c3e50;
        }
        
        .theme-blue {
            background: #3498db;
        }
        
        .theme-green {
            background: #2ecc71;
        }
        
        @media (max-width: 768px) {
            .nav-tabs {
                flex-direction: column;
            }
            
            .result-table {
                display: block;
                overflow-x: auto;
            }
            
            .student-info {
                grid-template-columns: 1fr;
            }
            
            .action-buttons {
                flex-direction: column;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .theme-selector {
                position: static;
                justify-content: center;
                margin-top: 15px;
            }
        }
        
        @media print {
            body * {
                visibility: hidden;
            }
            
            .result-section, .result-section * {
                visibility: visible;
            }
            
            .result-section {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
            }
            
            .no-print {
                display: none;
            }
            
            .print-only {
                display: block;
            }
            
            .card, .result-table {
                box-shadow: none;
            }
        }
        
        /* Dark theme */
        body.dark-theme {
            background-color: #1a1a1a;
            color: #f0f0f0;
        }
        
        body.dark-theme .card,
        body.dark-theme .stat-card {
            background: #2d2d2d;
            color: #f0f0f0;
        }
        
        body.dark-theme input,
        body.dark-theme select,
        body.dark-theme textarea {
            background: #3d3d3d;
            color: #f0f0f0;
            border-color: #4d4d4d;
        }
        
        body.dark-theme .result-table {
            background: #2d2d2d;
            color: #f0f0f0;
        }
        
        body.dark-theme .result-table th {
            background: #3d3d3d;
        }
        
        body.dark-theme .result-table tr:nth-child(even) {
            background: #353535;
        }
        
        body.dark-theme .result-table tr:hover {
            background: #404040;
        }
        
        /* Blue theme */
        body.blue-theme {
            background-color: #e6f2ff;
            color: #2c3e50;
        }
        
        body.blue-theme .card,
        body.blue-theme .stat-card {
            background: #d4e6ff;
            color: #2c3e50;
        }
        
        body.blue-theme header {
            background: linear-gradient(135deg, #2980b9, #2c3e50);
        }
        
        /* Green theme */
        body.green-theme {
            background-color: #eafaf1;
            color: #2c3e50;
        }
        
        body.green-theme .card,
        body.green-theme .stat-card {
            background: #d5f5e3;
            color: #2c3e50;
        }
        
        body.green-theme header {
            background: linear-gradient(135deg, #27ae60, #2c3e50);
        }
        
        /* Animation */
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        
        /* Loading spinner */
        .spinner {
            border: 4px solid rgba(0, 0, 0, 0.1);
            border-left: 4px solid var(--primary-color);
            border-radius: 50%;
            width: 30px;
            height: 30px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body class="light-theme">
    <div class="container">
        <header>
            <h1>Student Result Management System</h1>
            <p>Excellence in all ramifications</p>
            
            <div class="theme-selector">
                <div class="theme-btn theme-light" title="Light Theme" onclick="setTheme('light')"></div>
                <div class="theme-btn theme-dark" title="Dark Theme" onclick="setTheme('dark')"></div>
                <div class="theme-btn theme-blue" title="Blue Theme" onclick="setTheme('blue')"></div>
                <div class="theme-btn theme-green" title="Green Theme" onclick="setTheme('green')"></div>
            </div>
        </header>
        
        <div class="nav-tabs">
            <a href="index.php" class="nav-tab <?= !$isResultChecker ? 'active' : '' ?>">
                <i class="fas fa-edit"></i> Result Editor
            </a>
            <a href="checker.php" class="nav-tab <?= $isResultChecker ? 'active' : '' ?>">
                <i class="fas fa-search"></i> Result Checker
            </a>
            <a href="#" class="nav-tab" onclick="openStudentManager()">
                <i class="fas fa-users"></i> Student Manager
            </a>
            <a href="#" class="nav-tab" onclick="openStatistics()">
                <i class="fas fa-chart-bar"></i> Statistics
            </a>
        </div>
        
        <?php if (!$isResultChecker): ?>
        <!-- Result Editor Section -->
        <div class="card fade-in">
            <h2><i class="fas fa-edit"></i> Academic E-Report Sheet</h2>
            
            <div id="message"></div>
            
            <div class="student-info">
                <div class="form-group">
                    <label for="student_id"><i class="fas fa-id-card"></i> Student ID:</label>
                    <input type="text" id="student_id" value="PRIME-C006">
                </div>
                
                <div class="form-group">
                    <label for="name"><i class="fas fa-user"></i> Name:</label>
                    <input type="text" id="name" value="AREGBESOLA OLUWADAMILOLA MARVELOUS">
                </div>
                
                <div class="form-group">
                    <label for="class"><i class="fas fa-graduation-cap"></i> Class:</label>
                    <input type="text" id="class" value="JSS 3">
                </div>
                
                <div class="form-group">
                    <label for="year"><i class="fas fa-calendar"></i> Year:</label>
                    <input type="text" id="year" value="2024/2025">
                </div>
            </div>
            
            <table class="result-table">
                <thead>
                    <tr>
                        <th>SUBJECT</th>
                        <th>ASMNT (10)</th>
                        <th>TEST 2 (20)</th>
                        <th>EXAM (70)</th>
                        <th>TOTAL (100)</th>
                        <th>CUMM. (100)</th>
                        <th>GRADE</th>
                        <th>REMARK</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $subjects = [
                        'MATHEMATICS', 'ENGLISH LANGUAGE', 'BST', 'PVS', 
                        'NATIONAL VALUES', 'BUSINESS STUDIES', 'YORUBA LANGUAGE', 
                        'CRS', 'PHONICS', 'INFO TECH'
                    ];
                    
                    foreach ($subjects as $subject) {
                        $subjectKey = str_replace(' ', '_', strtolower($subject));
                        echo "<tr data-subject='$subjectKey'>";
                        echo "<td>$subject</td>";
                        echo "<td><input type='number' class='editable-cell asmnt' min='0' max='10' value='".($initialValues[$subjectKey.'_asmnt'] ?? '')."'></td>";
                        echo "<td><input type='number' class='editable-cell test' min='0' max='20' value='".($initialValues[$subjectKey.'_test'] ?? '')."'></td>";
                        echo "<td><input type='number' class='editable-cell exam' min='0' max='70' value='".($initialValues[$subjectKey.'_exam'] ?? '')."'></td>";
                        echo "<td class='total'>0</td>";
                        echo "<td class='cumm'>0</td>";
                        echo "<td class='grade'></td>";
                        echo "<td class='remark'></td>";
                        echo "</tr>";
                    }
                    ?>
                </tbody>
                <tfoot>
                    <tr>
                        <td colspan="4" class="text-center"><strong>OVERALL GRADE (%)</strong></td>
                        <td id="overall-percentage">0</td>
                        <td id="overall-grade">F</td>
                        <td colspan="2" id="overall-remark">FAIL</td>
                    </tr>
                </tfoot>
            </table>
            
            <div class="student-info">
                <div class="form-group">
                    <label for="promoted_to"><i class="fas fa-arrow-up"></i> Promoted To:</label>
                    <input type="text" id="promoted_to" value="JSS 3">
                </div>
                
                <div class="form-group">
                    <label for="next_term_begins"><i class="fas fa-calendar-alt"></i> Next Term Begins:</label>
                    <input type="text" id="next_term_begins" value="16 -- SEPT -- 2024">
                </div>
                
                <div class="form-group">
                    <label for="teacher_comment"><i class="fas fa-comment"></i> Teacher's Comment:</label>
                    <textarea id="teacher_comment" rows="2" placeholder="Enter teacher's comment">Good performance, keep it up!</textarea>
                </div>
                
                <div class="form-group">
                    <label for="principal_comment"><i class="fas fa-comment-dots"></i> Principal's Comment:</label>
                    <textarea id="principal_comment" rows="2" placeholder="Enter principal's comment">Excellent result, promoted to next class</textarea>
                </div>
            </div>
            
            <div class="action-buttons">
                <button id="save-btn" class="btn-success">
                    <i class="fas fa-save"></i> Save Result
                </button>
                <button id="new-btn" class="btn-warning">
                    <i class="fas fa-file"></i> New Result
                </button>
                <button id="print-btn" class="btn-print">
                    <i class="fas fa-print"></i> Print Result
                </button>
                <button id="delete-btn" class="btn-danger">
                    <i class="fas fa-trash"></i> Delete Result
                </button>
            </div>
        </div>
        <?php else: ?>
        <!-- Result Checker Section -->
        <div class="card fade-in">
            <h2><i class="fas fa-search"></i> Result Checker</h2>
            
            <div id="message"></div>
            
            <div class="search-box">
                <input type="text" id="query_student_id" placeholder="Enter Student ID (e.g., PRIME-C006)">
                <button id="query-btn">
                    <i class="fas fa-search"></i> Check Result
                </button>
            </div>
            
            <div id="result-display" class="hidden">
                <div class="result-section">
                    <h2 class="text-center">Academic E-Report Sheet</h2>
                    
                    <div class="student-info">
                        <div>
                            <strong><i class="fas fa-id-card"></i> Student ID:</strong> <span id="display-student-id"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-user"></i> Name:</strong> <span id="display-name"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-graduation-cap"></i> Class:</strong> <span id="display-class"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-calendar"></i> Year:</strong> <span id="display-year"></span>
                        </div>
                    </div>
                    
                    <table class="result-table">
                        <thead>
                            <tr>
                                <th>SUBJECT</th>
                                <th>ASMNT (10)</th>
                                <th>TEST 2 (20)</th>
                                <th>EXAM (70)</th>
                                <th>TOTAL (100)</th>
                                <th>CUMM. (100)</th>
                                <th>GRADE</th>
                                <th>REMARK</th>
                            </tr>
                        </thead>
                        <tbody id="result-body">
                            <!-- Results will be populated here -->
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="4" class="text-center"><strong>OVERALL GRADE (%)</strong></td>
                                <td id="display-overall-percentage">0</td>
                                <td id="display-overall-grade">F</td>
                                <td colspan="2" id="display-overall-remark">FAIL</td>
                            </tr>
                        </tfoot>
                    </table>
                    
                    <div class="student-info">
                        <div>
                            <strong><i class="fas fa-arrow-up"></i> Promoted To:</strong> <span id="display-promoted-to"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-calendar-alt"></i> Next Term Begins:</strong> <span id="display-next-term"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-comment"></i> Teacher's Comment:</strong> <span id="display-teacher-comment"></span>
                        </div>
                        <div>
                            <strong><i class="fas fa-comment-dots"></i> Principal's Comment:</strong> <span id="display-principal-comment"></span>
                        </div>
                    </div>
                    
                    <div class="text-center no-print">
                        <button id="display-print-btn" class="btn-print">
                            <i class="fas fa-print"></i> Print Result
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Student Manager Modal -->
    <div id="studentManagerModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-users"></i> Student Manager</h3>
                <span class="close" onclick="closeModal('studentManagerModal')">&times;</span>
            </div>
            <div class="search-box">
                <input type="text" id="searchStudent" placeholder="Search students...">
                <button onclick="searchStudents()">
                    <i class="fas fa-search"></i> Search
                </button>
            </div>
            <div class="student-list" id="studentList">
                <div class="spinner" id="studentSpinner"></div>
            </div>
        </div>
    </div>

    <!-- Statistics Modal -->
    <div id="statisticsModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-chart-bar"></i> Result Statistics</h3>
                <span class="close" onclick="closeModal('statisticsModal')">&times;</span>
            </div>
            <div id="statsContent">
                <div class="spinner" id="statsSpinner"></div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Load theme preference
            const savedTheme = localStorage.getItem('theme') || 'light';
            setTheme(savedTheme);
            
            <?php if (!$isResultChecker): ?>
            // Result Editor Functionality
            const editableCells = document.querySelectorAll('.editable-cell');
            const saveBtn = document.getElementById('save-btn');
            const newBtn = document.getElementById('new-btn');
            const printBtn = document.getElementById('print-btn');
            const deleteBtn = document.getElementById('delete-btn');
            const messageDiv = document.getElementById('message');
            
            // Calculate grades and remarks when values change
            editableCells.forEach(cell => {
                cell.addEventListener('input', calculateResults);
            });
            
            // Initial calculation
            calculateResults();
            
            // Save result
            saveBtn.addEventListener('click', function() {
                const studentId = document.getElementById('student_id').value;
                const name = document.getElementById('name').value;
                const className = document.getElementById('class').value;
                const year = document.getElementById('year').value;
                const promotedTo = document.getElementById('promoted_to').value;
                const nextTermBegins = document.getElementById('next_term_begins').value;
                const teacherComment = document.getElementById('teacher_comment').value;
                const principalComment = document.getElementById('principal_comment').value;
                
                if (!studentId || !name) {
                    showMessage('Student ID and Name are required!', 'error');
                    return;
                }
                
                // Prepare data object
                const data = {
                    action: 'save',
                    student_id: studentId,
                    name: name,
                    class: className,
                    year: year,
                    promoted_to: promotedTo,
                    next_term_begins: nextTermBegins,
                    teacher_comment: teacherComment,
                    principal_comment: principalComment
                };
                
                // Add subject scores
                document.querySelectorAll('tr[data-subject]').forEach(row => {
                    const subject = row.getAttribute('data-subject');
                    data[subject + '_asmnt'] = row.querySelector('.asmnt').value;
                    data[subject + '_test'] = row.querySelector('.test').value;
                    data[subject + '_exam'] = row.querySelector('.exam').value;
                    data[subject + '_total'] = row.querySelector('.total').textContent;
                    data[subject + '_cumm'] = row.querySelector('.cumm').textContent;
                    data[subject + '_grade'] = row.querySelector('.grade').textContent;
                    data[subject + '_remark'] = row.querySelector('.remark').textContent;
                });
                
                // Add overall scores
                data.overall_percentage = document.getElementById('overall-percentage').textContent;
                data.overall_grade = document.getElementById('overall-grade').textContent;
                data.overall_remark = document.getElementById('overall-remark').textContent;
                
                // Show loading
                saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
                saveBtn.disabled = true;
                
                // Send data to server
                fetch('index.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams(data)
                })
                .then(response => response.json())
                .then(result => {
                    if (result.success) {
                        showMessage(result.message, 'success');
                    } else {
                        showMessage(result.message, 'error');
                    }
                })
                .catch(error => {
                    showMessage('Error saving result: ' + error, 'error');
                })
                .finally(() => {
                    saveBtn.innerHTML = '<i class="fas fa-save"></i> Save Result';
                    saveBtn.disabled = false;
                });
            });
            
            // New result
            newBtn.addEventListener('click', function() {
                if (confirm('Create a new result? Any unsaved changes will be lost.')) {
                    document.getElementById('student_id').value = '';
                    document.getElementById('name').value = '';
                    document.getElementById('class').value = 'JSS 3';
                    document.getElementById('year').value = '2024/2025';
                    document.getElementById('promoted_to').value = 'JSS 3';
                    document.getElementById('next_term_begins').value = '16 -- SEPT -- 2024';
                    document.getElementById('teacher_comment').value = '';
                    document.getElementById('principal_comment').value = '';
                    
                    document.querySelectorAll('.editable-cell').forEach(cell => {
                        cell.value = '';
                    });
                    
                    calculateResults();
                    
                    showMessage('New result form prepared. Fill in the details and save.', 'success');
                }
            });
            
            // Print result
            printBtn.addEventListener('click', function() {
                window.print();
            });
            
            // Delete result
            deleteBtn.addEventListener('click', function() {
                const studentId = document.getElementById('student_id').value;
                
                if (!studentId) {
                    showMessage('Please enter a Student ID to delete', 'error');
                    return;
                }
                
                if (confirm(`Are you sure you want to delete the result for ${studentId}? This action cannot be undone.`)) {
                    // Send delete request to server
                    fetch('index.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            action: 'delete',
                            student_id: studentId
                        })
                    })
                    .then(response => response.json())
                    .then(result => {
                        if (result.success) {
                            showMessage(result.message, 'success');
                            // Clear the form
                            newBtn.click();
                        } else {
                            showMessage(result.message, 'error');
                        }
                    })
                    .catch(error => {
                        showMessage('Error deleting result: ' + error, 'error');
                    });
                }
            });
            
            // Calculate results function
            function calculateResults() {
                let totalPercentage = 0;
                let subjectCount = 0;
                
                document.querySelectorAll('tr[data-subject]').forEach(row => {
                    const asmnt = parseFloat(row.querySelector('.asmnt').value) || 0;
                    const test = parseFloat(row.querySelector('.test').value) || 0;
                    const exam = parseFloat(row.querySelector('.exam').value) || 0;
                    
                    // Calculate total (sum of asmnt, test, and exam)
                    const total = asmnt + test + exam;
                    row.querySelector('.total').textContent = total;
                    
                    // Calculate cumulative (average of total, but formula seems to be different in Excel)
                    // Using the formula from Excel: =SUM(F17,G17+H17)/1 which seems to be just the total
                    const cumm = total;
                    row.querySelector('.cumm').textContent = cumm;
                    
                    // Calculate grade
                    let grade = 'F';
                    if (cumm >= 70) grade = 'A';
                    else if (cumm >= 60) grade = 'B';
                    else if (cumm >= 50) grade = 'C';
                    else if (cumm >= 40) grade = 'D';
                    row.querySelector('.grade').textContent = grade;
                    
                    // Calculate remark
                    let remark = 'FAIL';
                    if (cumm >= 69.5) remark = 'EXCELLENT';
                    else if (cumm >= 59.5) remark = 'GOOD';
                    else if (cumm >= 49.5) remark = 'CREDIT';
                    else if (cumm >= 39.5) remark = 'PASS';
                    row.querySelector('.remark').textContent = remark;
                    
                    // Add to overall percentage calculation
                    totalPercentage += cumm;
                    subjectCount++;
                });
                
                // Calculate overall percentage (average of all subject cumm values)
                const overallPercentage = subjectCount > 0 ? (totalPercentage / subjectCount).toFixed(2) : 0;
                document.getElementById('overall-percentage').textContent = overallPercentage;
                
                // Calculate overall grade
                let overallGrade = 'F';
                if (overallPercentage >= 70) overallGrade = 'A';
                else if (overallPercentage >= 60) overallGrade = 'B';
                else if (overallPercentage >= 50) overallGrade = 'C';
                else if (overallPercentage >= 40) overallGrade = 'D';
                document.getElementById('overall-grade').textContent = overallGrade;
                
                // Calculate overall remark
                let overallRemark = 'FAIL';
                if (overallPercentage >= 69.5) overallRemark = 'EXCELLENT';
                else if (overallPercentage >= 59.5) overallRemark = 'GOOD';
                else if (overallPercentage >= 49.5) overallRemark = 'CREDIT';
                else if (overallPercentage >= 39.5) overallRemark = 'PASS';
                document.getElementById('overall-remark').textContent = overallRemark;
            }
            
            <?php else: ?>
            // Result Checker Functionality
            const queryBtn = document.getElementById('query-btn');
            const messageDiv = document.getElementById('message');
            const resultDisplay = document.getElementById('result-display');
            const printBtn = document.getElementById('display-print-btn');
            
            queryBtn.addEventListener('click', function() {
                const studentId = document.getElementById('query_student_id').value;
                
                if (!studentId) {
                    showMessage('Please enter a Student ID', 'error');
                    return;
                }
                
                // Show loading
                queryBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Searching...';
                queryBtn.disabled = true;
                
                // Query result from server
                fetch('checker.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'query',
                        student_id: studentId
                    })
                })
                .then(response => response.json())
                .then(result => {
                    if (result.success) {
                        displayResult(result.data);
                        showMessage('Result found successfully!', 'success');
                    } else {
                        resultDisplay.classList.add('hidden');
                        showMessage(result.message, 'error');
                    }
                })
                .catch(error => {
                    showMessage('Error querying result: ' + error, 'error');
                })
                .finally(() => {
                    queryBtn.innerHTML = '<i class="fas fa-search"></i> Check Result';
                    queryBtn.disabled = false;
                });
            });
            
            printBtn.addEventListener('click', function() {
                window.print();
            });
            
            function displayResult(data) {
                // Display student info
                document.getElementById('display-student-id').textContent = data.student_id;
                document.getElementById('display-name').textContent = data.name;
                document.getElementById('display-class').textContent = data.class;
                document.getElementById('display-year').textContent = data.year;
                document.getElementById('display-promoted-to').textContent = data.promoted_to;
                document.getElementById('display-next-term').textContent = data.next_term_begins;
                document.getElementById('display-teacher-comment').textContent = data.teacher_comment || 'No comment';
                document.getElementById('display-principal-comment').textContent = data.principal_comment || 'No comment';
                
                // Display subject results
                const resultBody = document.getElementById('result-body');
                resultBody.innerHTML = '';
                
                const subjects = [
                    'mathematics', 'english_language', 'bst', 'pvs', 
                    'national_values', 'business_studies', 'yoruba_language', 
                    'crs', 'phonics', 'info_tech'
                ];
                
                subjects.forEach(subject => {
                    const subjectName = subject.toUpperCase().replace('_', ' ');
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td>${subjectName}</td>
                        <td>${data[subject + '_asmnt'] || 0}</td>
                        <td>${data[subject + '_test'] || 0}</td>
                        <td>${data[subject + '_exam'] || 0}</td>
                        <td>${data[subject + '_total'] || 0}</td>
                        <td>${data[subject + '_cumm'] || 0}</td>
                        <td>${data[subject + '_grade'] || 'F'}</td>
                        <td>${data[subject + '_remark'] || 'FAIL'}</td>
                    `;
                    resultBody.appendChild(row);
                });
                
                // Display overall results
                document.getElementById('display-overall-percentage').textContent = data.overall_percentage || 0;
                document.getElementById('display-overall-grade').textContent = data.overall_grade || 'F';
                document.getElementById('display-overall-remark').textContent = data.overall_remark || 'FAIL';
                
                // Show the result display
                resultDisplay.classList.remove('hidden');
            }
            <?php endif; ?>
            
            function showMessage(message, type) {
                messageDiv.innerHTML = `
                    <div class="alert alert-${type}">
                        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'}"></i>
                        ${message}
                    </div>
                `;
                setTimeout(() => {
                    messageDiv.innerHTML = '';
                }, 5000);
            }
        });
        
        // Theme functions
        function setTheme(theme) {
            document.body.className = `${theme}-theme`;
            localStorage.setItem('theme', theme);
        }
        
        // Modal functions
        function openStudentManager() {
            document.getElementById('studentManagerModal').style.display = 'flex';
            loadStudents();
        }
        
        function openStatistics() {
            document.getElementById('statisticsModal').style.display = 'flex';
            loadStatistics();
        }
        
        function closeModal(modalId) {
            document.getElementById(modalId).style.display = 'none';
        }
        
        // Load students for management
        function loadStudents() {
            const studentList = document.getElementById('studentList');
            studentList.innerHTML = '<div class="spinner" id="studentSpinner"></div>';
            
            fetch('index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'list'
                })
            })
            .then(response => response.json())
            .then(result => {
                if (result.success && Object.keys(result.data).length > 0) {
                    let html = '';
                    Object.keys(result.data).forEach(studentId => {
                        const student = result.data[studentId];
                        html += `
                            <div class="student-item" onclick="loadStudent('${studentId}')">
                                <strong>${studentId}</strong> - ${student.name} (${student.class})
                            </div>
                        `;
                    });
                    studentList.innerHTML = html;
                } else {
                    studentList.innerHTML = '<div class="alert alert-error">No students found</div>';
                }
            })
            .catch(error => {
                studentList.innerHTML = `<div class="alert alert-error">Error loading students: ${error}</div>`;
            });
        }
        
        function searchStudents() {
            const searchTerm = document.getElementById('searchStudent').value.toLowerCase();
            const studentItems = document.querySelectorAll('.student-item');
            
            studentItems.forEach(item => {
                const text = item.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    item.style.display = 'block';
                } else {
                    item.style.display = 'none';
                }
            });
        }
        
        function loadStudent(studentId) {
            fetch('index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'query',
                    student_id: studentId
                })
            })
            .then(response => response.json())
            .then(result => {
                if (result.success) {
                    const data = result.data;
                    
                    // Fill the form with student data
                    document.getElementById('student_id').value = data.student_id;
                    document.getElementById('name').value = data.name;
                    document.getElementById('class').value = data.class;
                    document.getElementById('year').value = data.year;
                    document.getElementById('promoted_to').value = data.promoted_to;
                    document.getElementById('next_term_begins').value = data.next_term_begins;
                    document.getElementById('teacher_comment').value = data.teacher_comment || '';
                    document.getElementById('principal_comment').value = data.principal_comment || '';
                    
                    // Fill subject scores
                    const subjects = [
                        'mathematics', 'english_language', 'bst', 'pvs', 
                        'national_values', 'business_studies', 'yoruba_language', 
                        'crs', 'phonics', 'info_tech'
                    ];
                    
                    subjects.forEach(subject => {
                        const row = document.querySelector(`tr[data-subject="${subject}"]`);
                        if (row) {
                            row.querySelector('.asmnt').value = data[subject + '_asmnt'] || '';
                            row.querySelector('.test').value = data[subject + '_test'] || '';
                            row.querySelector('.exam').value = data[subject + '_exam'] || '';
                        }
                    });
                    
                    // Calculate results
                    calculateResults();
                    
                    // Close modal
                    closeModal('studentManagerModal');
                    
                    showMessage(`Loaded data for ${data.name}`, 'success');
                } else {
                    showMessage('Error loading student data', 'error');
                }
            })
            .catch(error => {
                showMessage('Error loading student: ' + error, 'error');
            });
        }
        
        function loadStatistics() {
            const statsContent = document.getElementById('statsContent');
            statsContent.innerHTML = '<div class="spinner" id="statsSpinner"></div>';
            
            fetch('index.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'list'
                })
            })
            .then(response => response.json())
            .then(result => {
                if (result.success && Object.keys(result.data).length > 0) {
                    const data = result.data;
                    const totalStudents = Object.keys(data).length;
                    
                    let totalPercentage = 0;
                    let gradeDistribution = { A: 0, B: 0, C: 0, D: 0, F: 0 };
                    
                    Object.values(data).forEach(student => {
                        totalPercentage += parseFloat(student.overall_percentage) || 0;
                        
                        if (student.overall_grade) {
                            gradeDistribution[student.overall_grade] = (gradeDistribution[student.overall_grade] || 0) + 1;
                        }
                    });
                    
                    const averagePercentage = (totalPercentage / totalStudents).toFixed(2);
                    
                    let html = `
                        <div class="stats-container">
                            <div class="stat-card">
                                <div class="stat-label">Total Students</div>
                                <div class="stat-value">${totalStudents}</div>
                            </div>
                            <div class="stat-card">
                                <div class="stat-label">Average Score</div>
                                <div class="stat-value">${averagePercentage}%</div>
                            </div>
                        </div>
                        
                        <h3>Grade Distribution</h3>
                        <table class="result-table">
                            <thead>
                                <tr>
                                    <th>Grade</th>
                                    <th>Count</th>
                                    <th>Percentage</th>
                                </tr>
                            </thead>
                            <tbody>
                    `;
                    
                    Object.keys(gradeDistribution).forEach(grade => {
                        const count = gradeDistribution[grade];
                        const percentage = ((count / totalStudents) * 100).toFixed(1);
                        html += `
                            <tr>
                                <td>${grade}</td>
                                <td>${count}</td>
                                <td>${percentage}%</td>
                            </tr>
                        `;
                    });
                    
                    html += `
                            </tbody>
                        </table>
                    `;
                    
                    statsContent.innerHTML = html;
                } else {
                    statsContent.innerHTML = '<div class="alert alert-error">No data available for statistics</div>';
                }
            })
            .catch(error => {
                statsContent.innerHTML = `<div class="alert alert-error">Error loading statistics: ${error}</div>`;
            });
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            if (event.target.classList.contains('modal')) {
                event.target.style.display = 'none';
            }
        }
    </script>
</body>
</html>