document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const searchInput = document.getElementById('search-input');
    const searchBtn = document.getElementById('search-btn');
    const suggestions = document.getElementById('suggestions');
    const showDetails = document.getElementById('show-details');
    const episodesContainer = document.getElementById('episodes-container');
    const castContainer = document.getElementById('cast-container');
    const crewContainer = document.getElementById('crew-container');
    const seasonsContainer = document.getElementById('seasons-container');
    const similarShows = document.getElementById('similar-shows');
    const trendingShows = document.getElementById('trending-shows');
    const loading = document.getElementById('loading');
    const themeIcon = document.getElementById('theme-icon');
    
    // Theme Toggle
    themeIcon.addEventListener('click', toggleTheme);
    
    // Check for saved theme preference
    if (localStorage.getItem('theme') === 'dark') {
        document.documentElement.setAttribute('data-theme', 'dark');
        themeIcon.classList.remove('fa-moon');
        themeIcon.classList.add('fa-sun');
    }
    
    // Load trending shows on page load
    loadTrendingShows();
    
    // Event Listeners
    searchBtn.addEventListener('click', () => searchShow(searchInput.value));
    searchInput.addEventListener('keyup', (e) => {
        if (e.key === 'Enter') {
            searchShow(searchInput.value);
        } else {
            fetchSuggestions(searchInput.value);
        }
    });
    
    // Close suggestions when clicking outside
    document.addEventListener('click', (e) => {
        if (!searchInput.contains(e.target) && !suggestions.contains(e.target)) {
            suggestions.style.display = 'none';
        }
    });
    
    // Functions
    function toggleTheme() {
        const currentTheme = document.documentElement.getAttribute('data-theme');
        if (currentTheme === 'dark') {
            document.documentElement.removeAttribute('data-theme');
            localStorage.setItem('theme', 'light');
            themeIcon.classList.remove('fa-sun');
            themeIcon.classList.add('fa-moon');
        } else {
            document.documentElement.setAttribute('data-theme', 'dark');
            localStorage.setItem('theme', 'dark');
            themeIcon.classList.remove('fa-moon');
            themeIcon.classList.add('fa-sun');
        }
    }
    
    async function loadTrendingShows() {
        try {
            loading.style.display = 'flex';
            const response = await fetch('https://api.tvmaze.com/shows');
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            // Sort by rating and get top 10
            const trending = data
                .filter(show => show.rating?.average) // Filter out shows without ratings
                .sort((a, b) => (b.rating.average || 0) - (a.rating.average || 0))
                .slice(0, 10);
            
            trendingShows.innerHTML = trending.map(show => `
                <div class="trending-show" data-id="${show.id}">
                    ${show.image ? `<img src="${show.image.medium}" alt="${show.name}" loading="lazy">` : '<div class="no-image"><i class="fas fa-tv"></i></div>'}
                    <p>${show.name}</p>
                </div>
            `).join('');
            
            // Add click event to trending shows
            document.querySelectorAll('.trending-show').forEach(show => {
                show.addEventListener('click', () => {
                    const showId = show.getAttribute('data-id');
                    fetchShowDetails(showId);
                    searchInput.value = '';
                    suggestions.style.display = 'none';
                    window.scrollTo({ top: 0, behavior: 'smooth' });
                });
            });
        } catch (error) {
            console.error('Error loading trending shows:', error);
            trendingShows.innerHTML = `
                <div class="error-message">
                    <i class="fas fa-exclamation-circle"></i>
                    <p>Failed to load trending shows. Please try again later.</p>
                </div>
            `;
        } finally {
            loading.style.display = 'none';
        }
    }
    
    async function fetchSuggestions(query) {
        if (query.length < 2) {
            suggestions.style.display = 'none';
            return;
        }
        
        try {
            const response = await fetch(`https://api.tvmaze.com/search/shows?q=${encodeURIComponent(query)}`);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.length > 0) {
                suggestions.innerHTML = data.map(item => `
                    <div class="suggestion-item" data-id="${item.show.id}">
                        ${item.show.image ? `<img src="${item.show.image.medium}" alt="${item.show.name}" width="50" loading="lazy">` : '<i class="fas fa-tv"></i>'}
                        <div class="suggestion-text">
                            <span class="suggestion-title">${item.show.name}</span>
                            ${item.show.premiered ? `<span class="suggestion-year">(${item.show.premiered.split('-')[0]})</span>` : ''}
                        </div>
                    </div>
                `).join('');
                
                suggestions.style.display = 'block';
                
                // Add click event to suggestions
                document.querySelectorAll('.suggestion-item').forEach(item => {
                    item.addEventListener('click', () => {
                        const showId = item.getAttribute('data-id');
                        fetchShowDetails(showId);
                        searchInput.value = item.querySelector('.suggestion-title').textContent;
                        suggestions.style.display = 'none';
                    });
                });
            } else {
                suggestions.innerHTML = '<div class="suggestion-item no-results">No results found</div>';
                suggestions.style.display = 'block';
            }
        } catch (error) {
            console.error('Error fetching suggestions:', error);
            suggestions.innerHTML = '<div class="suggestion-item no-results">Error loading suggestions</div>';
            suggestions.style.display = 'block';
        }
    }
    
    function searchShow(query) {
        if (query.trim() === '') {
            showErrorMessage('Please enter a show name to search');
            return;
        }
        
        loading.style.display = 'flex';
        
        fetch(`https://api.tvmaze.com/singlesearch/shows?q=${encodeURIComponent(query)}`)
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                fetchShowDetails(data.id);
                searchInput.value = data.name;
                suggestions.style.display = 'none';
            })
            .catch(error => {
                console.error('Error searching show:', error);
                showErrorMessage('No show found with that name. Please try another search.');
                clearSections();
            })
            .finally(() => {
                loading.style.display = 'none';
            });
    }
    
    async function fetchShowDetails(showId) {
        loading.style.display = 'flex';
        
        try {
            // Show details
            const showResponse = await fetch(`https://api.tvmaze.com/shows/${showId}`);
            if (!showResponse.ok) throw new Error(`Show details error: ${showResponse.status}`);
            const showData = await showResponse.json();
            
            // Parallel requests for all additional data
            const endpoints = [
                `shows/${showId}/cast`,
                `shows/${showId}/crew`,
                `shows/${showId}/episodes`,
                `shows/${showId}/seasons`,
                `shows/${showId}/similar`
            ];
            
            const responses = await Promise.allSettled(
                endpoints.map(endpoint => 
                    fetch(`https://api.tvmaze.com/${endpoint}`)
                        .then(res => res.ok ? res.json() : Promise.reject(res.status))
                )
            );
            
            // Process responses
            const [castData, crewData, episodesData, seasonsData, similarData] = responses.map(res => {
                return res.status === 'fulfilled' ? res.value : null;
            });
            
            // Display all data
            displayShowDetails(showData);
            displayCast(castData);
            displayCrew(crewData);
            displayEpisodes(episodesData);
            displaySeasons(seasonsData);
            displaySimilarShows(similarData);
            
            // Scroll to top
            window.scrollTo({ top: 0, behavior: 'smooth' });
            
        } catch (error) {
            console.error('Error fetching show details:', error);
            showErrorMessage('There was an error loading the show details. Please try again later.');
            clearSections();
        } finally {
            loading.style.display = 'none';
        }
    }
    
    function displayShowDetails(show) {
        if (!show) return;
        
        // Format schedule days
        let scheduleText = '';
        if (show.schedule?.days?.length > 0) {
            scheduleText = show.schedule.days.join(', ');
            if (show.schedule.time) {
                scheduleText += ` at ${show.schedule.time}`;
            }
        }
        
        showDetails.innerHTML = `
            <div class="show-poster">
                ${show.image ? `<img src="${show.image.original}" alt="${show.name}" loading="lazy">` : '<div class="no-image"><i class="fas fa-tv"></i></div>'}
            </div>
            <div class="show-info">
                <h1 class="show-title">${show.name}</h1>
                
                <div class="show-meta">
                    ${show.premiered ? `<div class="meta-item"><i class="fas fa-calendar-alt"></i> ${show.premiered.split('-')[0]}</div>` : ''}
                    ${show.ended ? `<div class="meta-item"><i class="fas fa-calendar-times"></i> ${show.ended.split('-')[0]}</div>` : ''}
                    ${show.runtime ? `<div class="meta-item"><i class="fas fa-clock"></i> ${show.runtime} min</div>` : ''}
                    ${show.language ? `<div class="meta-item"><i class="fas fa-language"></i> ${show.language}</div>` : ''}
                    ${show.status ? `<div class="meta-item"><i class="fas fa-info-circle"></i> ${show.status}</div>` : ''}
                    ${show.rating?.average ? `<div class="meta-item"><i class="fas fa-star"></i> ${show.rating.average}/10</div>` : ''}
                </div>
                
                ${show.genres?.length > 0 ? `
                    <div class="show-genres">
                        ${show.genres.map(genre => `<span class="genre">${genre}</span>`).join('')}
                    </div>
                ` : ''}
                
                ${show.summary ? `
                    <div class="show-summary">
                        <h3>Summary</h3>
                        ${show.summary}
                    </div>
                ` : ''}
                
                <div class="show-stats">
                    ${scheduleText ? `
                        <div class="stat-card">
                            <h3>${scheduleText}</h3>
                            <p>Schedule</p>
                        </div>
                    ` : ''}
                    
                    ${show.network ? `
                        <div class="stat-card">
                            <h3>${show.network.name}</h3>
                            <p>Network</p>
                        </div>
                    ` : show.webChannel ? `
                        <div class="stat-card">
                            <h3>${show.webChannel.name}</h3>
                            <p>Web Channel</p>
                        </div>
                    ` : ''}
                    
                    ${show.weight ? `
                        <div class="stat-card">
                            <h3>${show.weight}</h3>
                            <p>Popularity</p>
                        </div>
                    ` : ''}
                    
                    ${show.type ? `
                        <div class="stat-card">
                            <h3>${show.type}</h3>
                            <p>Show Type</p>
                        </div>
                    ` : ''}
                </div>
                
                ${show.officialSite ? `
                    <a href="${show.officialSite}" class="official-site" target="_blank" rel="noopener noreferrer">
                        <i class="fas fa-external-link-alt"></i> Official Site
                    </a>
                ` : ''}
                
                ${show.externals?.imdb ? `
                    <a href="https://www.imdb.com/title/${show.externals.imdb}" class="imdb-link" target="_blank" rel="noopener noreferrer">
                        <i class="fab fa-imdb"></i> View on IMDb
                    </a>
                ` : ''}
            </div>
        `;
    }
    
    function displayCast(cast) {
        if (!cast || cast.length === 0) {
            castContainer.innerHTML = '<p class="no-data">No cast information available</p>';
            return;
        }
        
        castContainer.innerHTML = `
            <h2 class="section-title">Main Cast</h2>
            <div class="cast-grid">
                ${cast.slice(0, 12).map(person => `
                    <div class="cast-card">
                        ${person.person.image ? `<img src="${person.person.image.medium}" alt="${person.person.name}" loading="lazy">` : '<div class="no-image"><i class="fas fa-user"></i></div>'}
                        <div class="cast-info">
                            <h3>${person.person.name}</h3>
                            <p>as ${person.character.name}</p>
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
    }
    
    function displayCrew(crew) {
        if (!crew || crew.length === 0) {
            crewContainer.innerHTML = '<p class="no-data">No crew information available</p>';
            return;
        }
        
        // Group crew by type
        const crewByType = {};
        crew.forEach(member => {
            if (!crewByType[member.type]) {
                crewByType[member.type] = [];
            }
            crewByType[member.type].push(member);
        });
        
        let crewHTML = '<h2 class="section-title">Crew</h2>';
        
        for (const [type, members] of Object.entries(crewByType)) {
            crewHTML += `
                <h3 class="crew-type-title">${type}</h3>
                <div class="crew-grid">
                    ${members.slice(0, 6).map(member => `
                        <div class="crew-card">
                            ${member.person.image ? `<img src="${member.person.image.medium}" alt="${member.person.name}" loading="lazy">` : '<div class="no-image"><i class="fas fa-user"></i></div>'}
                            <div class="crew-info">
                                <h3>${member.person.name}</h3>
                                <p>${member.type}</p>
                            </div>
                        </div>
                    `).join('')}
                </div>
            `;
        }
        
        crewContainer.innerHTML = crewHTML;
    }
    
    function displayEpisodes(episodes) {
        if (!episodes || episodes.length === 0) {
            episodesContainer.innerHTML = '<p class="no-data">No episode information available</p>';
            return;
        }
        
        // Group episodes by season
        const episodesBySeason = {};
        episodes.forEach(episode => {
            if (!episodesBySeason[episode.season]) {
                episodesBySeason[episode.season] = [];
            }
            episodesBySeason[episode.season].push(episode);
        });
        
        let episodesHTML = '<h2 class="section-title">Episodes</h2>';
        
        for (const [season, seasonEpisodes] of Object.entries(episodesBySeason)) {
            episodesHTML += `
                <h3 class="season-title">Season ${season}</h3>
                <div class="episode-list">
                    ${seasonEpisodes.map(episode => `
                        <div class="episode-card">
                            <div class="episode-header">
                                <div class="episode-title">
                                    ${episode.number ? `<span class="episode-number">${episode.number}.</span>` : ''}
                                    <span class="episode-name">${episode.name || 'Untitled'}</span>
                                </div>
                                <div class="episode-meta">
                                    ${episode.airdate ? `<span><i class="fas fa-calendar-alt"></i> ${episode.airdate}</span>` : ''}
                                    ${episode.runtime ? `<span><i class="fas fa-clock"></i> ${episode.runtime} min</span>` : ''}
                                    ${episode.rating?.average ? `<span><i class="fas fa-star"></i> ${episode.rating.average}</span>` : ''}
                                </div>
                            </div>
                            ${episode.summary ? `
                                <div class="episode-summary">
                                    ${episode.summary}
                                </div>
                            ` : ''}
                        </div>
                    `).join('')}
                </div>
            `;
        }
        
        episodesContainer.innerHTML = episodesHTML;
        
        // Add click event to episode headers
        document.querySelectorAll('.episode-header').forEach(header => {
            header.addEventListener('click', () => {
                const summary = header.nextElementSibling;
                summary.classList.toggle('show');
                header.querySelector('.episode-title').classList.toggle('expanded');
            });
        });
    }
    
    function displaySeasons(seasons) {
        if (!seasons || seasons.length === 0) {
            seasonsContainer.innerHTML = '<p class="no-data">No season information available</p>';
            return;
        }
        
        seasonsContainer.innerHTML = `
            <h2 class="section-title">Seasons</h2>
            <div class="seasons-grid">
                ${seasons.map(season => `
                    <div class="season-card">
                        <div class="season-poster">
                            ${season.image ? `<img src="${season.image.medium}" alt="Season ${season.number}" loading="lazy">` : '<i class="fas fa-tv"></i>'}
                        </div>
                        <div class="season-info">
                            <h3>Season ${season.number}</h3>
                            <p>${season.episodeOrder} episodes</p>
                            ${season.premiereDate ? `<p>Premiere: ${season.premiereDate}</p>` : ''}
                            ${season.endDate ? `<p>End: ${season.endDate}</p>` : ''}
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
    }
    
    function displaySimilarShows(similar) {
        if (!similar || similar.length === 0) {
            similarShows.innerHTML = '<p class="no-data">No similar shows found</p>';
            return;
        }
        
        similarShows.innerHTML = `
            <h2 class="section-title">Similar Shows</h2>
            <div class="similar-grid">
                ${similar.slice(0, 6).map(show => `
                    <div class="similar-card" data-id="${show.id}">
                        ${show.image ? `<img src="${show.image.medium}" alt="${show.name}" loading="lazy">` : '<div class="no-image"><i class="fas fa-tv"></i></div>'}
                        <div class="similar-info">
                            <h3>${show.name}</h3>
                            <p>${show.rating?.average ? `${show.rating.average}/10` : 'No rating'}</p>
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
        
        // Add click event to similar shows
        document.querySelectorAll('.similar-card').forEach(show => {
            show.addEventListener('click', () => {
                const showId = show.getAttribute('data-id');
                fetchShowDetails(showId);
                window.scrollTo({ top: 0, behavior: 'smooth' });
            });
        });
    }
    
    function showErrorMessage(message) {
        showDetails.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i>
                <p>${message}</p>
            </div>
        `;
    }
    
    function clearSections() {
        episodesContainer.innerHTML = '';
        castContainer.innerHTML = '';
        crewContainer.innerHTML = '';
        seasonsContainer.innerHTML = '';
        similarShows.innerHTML = '';
    }
});