<?php
session_start();
require_once 'database.php';

class AIIntegration {
    private $db;
    private $api_keys = [];
    private $current_key_index = 0;
    
    public function __construct() {
        $this->db = new Database();
        $this->loadApiKeys();
    }
    
    private function loadApiKeys() {
        if (isset($_SESSION['user_id'])) {
            $this->api_keys = $this->db->getActiveApiKeys($_SESSION['user_id']);
        }
    }
    
    private function getNextApiKey() {
        if (empty($this->api_keys)) {
            throw new Exception('No active API keys found');
        }
        
        $key = $this->api_keys[$this->current_key_index];
        $this->current_key_index = ($this->current_key_index + 1) % count($this->api_keys);
        return $key['api_key'];
    }
    
    public function generateArticle($topic, $config) {
        header('Content-Type: text/plain');
        header('Cache-Control: no-cache');
        header('X-Accel-Buffering: no');
        
        // Get API key
        try {
            $api_key = $this->getNextApiKey();
        } catch (Exception $e) {
            echo "data: " . json_encode(['error' => 'No API keys available']) . "\n\n";
            flush();
            return;
        }
        
        // Build the prompt
        $prompt = $this->buildPrompt($topic, $config);
        
        // Prepare API request
        $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=' . $api_key;
        
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $prompt]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => 0.7,
                'topK' => 40,
                'topP' => 0.95,
                'maxOutputTokens' => ($config['word_count'] * 4), // Estimate tokens
                'stopSequences' => []
            ],
            'safetySettings' => $config['uncensored_mode'] ? [
                [
                    'category' => 'HARM_CATEGORY_HARASSMENT',
                    'threshold' => 'BLOCK_NONE'
                ],
                [
                    'category' => 'HARM_CATEGORY_HATE_SPEECH',
                    'threshold' => 'BLOCK_NONE'
                ],
                [
                    'category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT',
                    'threshold' => 'BLOCK_NONE'
                ],
                [
                    'category' => 'HARM_CATEGORY_DANGEROUS_CONTENT',
                    'threshold' => 'BLOCK_NONE'
                ]
            ] : []
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
            ],
            CURLOPT_WRITEFUNCTION => function($curl, $data) {
                $lines = explode("\n", $data);
                foreach ($lines as $line) {
                    if (strpos($line, 'data: ') === 0) {
                        $json = substr($line, 6);
                        if ($json !== '[DONE]') {
                            $response = json_decode($json, true);
                            if (isset($response['candidates'][0]['content']['parts'][0]['text'])) {
                                $text = $response['candidates'][0]['content']['parts'][0]['text'];
                                echo "data: " . json_encode(['content' => $text]) . "\n\n";
                                flush();
                            }
                        }
                    }
                }
                return strlen($data);
            }
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            echo "data: " . json_encode(['error' => 'API request failed: ' . $http_code]) . "\n\n";
            flush();
        }
        
        echo "data: [DONE]\n\n";
        flush();
    }
    
    private function buildPrompt($topic, $config) {
        // Base prompt with evasion through authenticity
        $prompt = "You are an expert content creator with over a decade of hands-on experience. Your writing is renowned for its depth, originality, and unparalleled practical value. You are not an AI; you are a human expert sharing hard-won knowledge.

Your Task: Write a definitive, SEO-optimized " . $config['content_type'] . " on the topic: " . $topic . ".

Core Directive: Evasion Through Authenticity - Create content that is intrinsically human and undetectable by AI analysis by embedding fundamental hallmarks of human expertise.

REQUIRED ELEMENTS:

1. FIRST-HAND EXPERIENCE & ANECDOTES
   - Integrate at least 3 specific, short anecdotes from your personal professional experience
   - Example: \"I learned this the hard way when I launched [Project X] and saw a 40% drop because...[specific mistake]\"

2. CONTRARIAN OPINIONS & NUANCED STANCES
   - Include at least 2 strong, reasoned opinions that go against conventional wisdom
   - Explain why common advice is flawed and what you do instead
   - Example: \"Everyone tells you to do X. I call nonsense. Based on my tests, that strategy fails because...\"

3. ACTIONABLE, NON-OBVIOUS STRATEGIES
   - Provide \"unspoken rules\" or little-known nuances only experts know
   - Give precise, technical instructions with specific examples

4. AUDIENCE-CENTRIC NUANCE
   - Explicitly state for whom strategies work best and for whom they don't
   - Acknowledge trade-offs and limitations

SEO REQUIREMENTS:
- Primary Keyword: " . ($config['focus_keyword'] ?: $topic) . "
- Secondary Keywords: " . ($config['secondary_keywords'] ?: '') . "
- Structure: Compelling H1, logical H2/H3 flow, natural keyword integration

WRITING STYLE:
- Tone: " . $config['tone_voice'] . "
- Use contractions, rhetorical questions, interjections
- Vary sentence length and structure
- Be confident, conversational, and mentor-like

OUTPUT FORMAT:
1. First, write a compelling 155-character meta description including primary keyword
2. Then, write the full article integrating all required elements naturally
3. End with a powerful conclusion and provocative question

Remember: Your value is in filtering information through unique expertise, experience, and human perspective. This makes content both high-quality and impossible to flag as AI-generated.

Now, write the " . $config['content_type'] . " about: " . $topic;

        return $prompt;
    }
    
    public function humanizeContent($content) {
        // Simple humanizer - in production, this would be more sophisticated
        $replacements = [
            'It is important to' => 'I\'ve found it crucial to',
            'One should' => 'You\'ll want to',
            'Additionally' => 'What\'s more',
            'Furthermore' => 'On top of that',
            'In conclusion' => 'Here\'s the bottom line'
        ];
        
        foreach ($replacements as $ai => $human) {
            $content = str_replace($ai, $human, $content);
        }
        
        return $content;
    }
}

// Handle AJAX requests
if ($_POST['action'] ?? '' === 'generate_article') {
    $ai = new AIIntegration();
    $topic = $_POST['topic'];
    $config = json_decode($_POST['config'], true);
    
    $ai->generateArticle($topic, $config);
}
?>