<?php
session_start();
require_once 'database.php';

// Check authentication for all API calls
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'error' => 'Not authenticated']);
    exit;
}

$db = new Database();
$user_id = $_SESSION['user_id'];

// Set JSON header
header('Content-Type: application/json');

// Get the raw POST data
$input = file_get_contents('php://input');
$data = [];

// Try to parse as JSON first
if (!empty($input)) {
    $data = json_decode($input, true) ?: [];
}

// If no JSON data, use form data
if (empty($data)) {
    $data = $_POST;
}

$action = $data['action'] ?? '';

// Handle API key operations
if ($action === 'add_api_key') {
    $api_key = $data['api_key'] ?? '';
    
    if (empty(trim($api_key))) {
        echo json_encode(['success' => false, 'error' => 'API key is required']);
        exit;
    }
    
    $result = $db->addApiKey($user_id, trim($api_key));
    echo json_encode($result);
    exit;
}

if ($action === 'delete_api_key') {
    $key_id = $data['key_id'] ?? '';
    
    if (empty($key_id)) {
        echo json_encode(['success' => false, 'error' => 'Key ID is required']);
        exit;
    }
    
    $success = $db->deleteApiKey($user_id, $key_id);
    echo json_encode(['success' => $success]);
    exit;
}

if ($action === 'get_api_keys') {
    $api_keys = $db->getActiveApiKeys($user_id);
    echo json_encode(['success' => true, 'keys' => $api_keys]);
    exit;
}

if ($action === 'save_content') {
    $project_id = $data['project_id'] ?? '';
    $topic = $data['topic'] ?? '';
    $content = $data['content'] ?? '';
    
    if (empty($project_id) || empty($topic) || empty($content)) {
        echo json_encode(['success' => false, 'error' => 'Missing required fields']);
        exit;
    }
    
    $config = $data['config'] ?? [];
    $content_id = $db->saveGeneratedContent($project_id, $topic, $content, $config);
    echo json_encode(['success' => (bool)$content_id, 'content_id' => $content_id]);
    exit;
}

if ($action === 'update_content') {
    // This would update existing content in production
    // For now, just return success
    echo json_encode(['success' => true, 'message' => 'Content updated successfully']);
    exit;
}

if ($action === 'generate_article') {
    $topic = $data['topic'] ?? '';
    $config = $data['config'] ?? [];
    
    if (empty($topic)) {
        echo json_encode(['success' => false, 'error' => 'Topic is required']);
        exit;
    }
    
    // Get active API keys
    $api_keys = $db->getActiveApiKeys($user_id);
    if (empty($api_keys)) {
        echo json_encode(['success' => false, 'error' => 'No active API keys found']);
        exit;
    }
    
    // Try each API key
    foreach ($api_keys as $api_key_data) {
        try {
            $content = $this->generateWithGemini($api_key_data['api_key'], $topic, $config);
            if ($content) {
                // Save the content
                $content_id = $db->saveGeneratedContent($config['project_id'], $topic, $content, $config);
                
                echo json_encode([
                    'success' => true,
                    'content' => $content,
                    'content_id' => $content_id
                ]);
                exit;
            }
        } catch (Exception $e) {
            // Try next API key
            continue;
        }
    }
    
    echo json_encode(['success' => false, 'error' => 'All API keys failed']);
    exit;
}

// Unknown action
echo json_encode(['success' => false, 'error' => 'Unknown action: ' . $action]);
exit;

// Gemini API generation function
function generateWithGemini($apiKey, $topic, $config) {
    $prompt = buildAdvancedPrompt($topic, $config);
    
    $url = "https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=" . $apiKey;
    
    $postData = [
        'contents' => [
            [
                'parts' => [
                    ['text' => $prompt]
                ]
            ]
        ],
        'generationConfig' => [
            'temperature' => 0.7,
            'topK' => 40,
            'topP' => 0.95,
            'maxOutputTokens' => $config['word_count'] * 4,
            'stopSequences' => []
        ]
    ];
    
    // Add safety settings for uncensored mode
    if ($config['uncensored_mode'] ?? false) {
        $postData['safetySettings'] = [
            ['category' => 'HARM_CATEGORY_HARASSMENT', 'threshold' => 'BLOCK_NONE'],
            ['category' => 'HARM_CATEGORY_HATE_SPEECH', 'threshold' => 'BLOCK_NONE'],
            ['category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT', 'threshold' => 'BLOCK_NONE'],
            ['category' => 'HARM_CATEGORY_DANGEROUS_CONTENT', 'threshold' => 'BLOCK_NONE']
        ];
    }
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($postData),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
        ],
        CURLOPT_TIMEOUT => 60
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        throw new Exception('API request failed with code: ' . $httpCode);
    }
    
    $data = json_decode($response, true);
    
    if (!isset($data['candidates'][0]['content']['parts'][0]['text'])) {
        throw new Exception('No content generated by AI');
    }
    
    return $data['candidates'][0]['content']['parts'][0]['text'];
}

function buildAdvancedPrompt($topic, $config) {
    $prompt = "You are {$config['ai_persona']}, an expert content creator with deep industry knowledge. ";
    
    if (!empty($config['ai_prompt'])) {
        $prompt .= $config['ai_prompt'] . " ";
    }
    
    $prompt .= "\n\nTOPIC: {$topic}";
    $prompt .= "\nCONTENT TYPE: {$config['content_type']}";
    $prompt .= "\nTONE: {$config['tone_voice']}";
    $prompt .= "\nTARGET WORD COUNT: {$config['word_count']}";
    
    $prompt .= "\n\nCORE REQUIREMENTS:";
    $prompt .= "\n1. Include 3+ specific, real-world anecdotes from your professional experience";
    $prompt .= "\n2. Provide 2+ contrarian opinions that challenge conventional wisdom";
    $prompt .= "\n3. Share actionable, non-obvious strategies with precise instructions";
    $prompt .= "\n4. Specify exactly who this strategy works for and who it doesn't";
    $prompt .= "\n5. Use a {$config['tone_voice']} tone throughout";
    
    if (!empty($config['focus_keyword'])) {
        $prompt .= "\n\nSEO REQUIREMENTS:";
        $prompt .= "\n- Primary Keyword: {$config['focus_keyword']}";
        if (!empty($config['secondary_keywords'])) {
            $prompt .= "\n- Secondary Keywords: {$config['secondary_keywords']}";
        }
    }
    
    $prompt .= "\n\nADDITIONAL INSTRUCTIONS:";
    if (!empty($config['extra_prompt'])) {
        $prompt .= "\n{$config['extra_prompt']}";
    } else {
        $prompt .= "\nNone provided";
    }
    
    $prompt .= "\n\nOUTPUT FORMAT:";
    $prompt .= "\n1. Start with a compelling meta description (under 155 characters)";
    $prompt .= "\n2. Write the full article with proper HTML formatting (h1, h2, h3, p, strong, em, ul/ol)";
    $prompt .= "\n3. End with a thought-provoking conclusion and question";
    
    $prompt .= "\n\nIMPORTANT: This content must pass AI detection by being deeply human, opinionated, and based on real experience. Avoid generic advice and focus on unique insights.";
    
    $prompt .= "\n\nNow, create the {$config['content_type']} about: {$topic}";
    
    return $prompt;
}
?>