<?php
session_start();
require_once 'database.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$project_id = $_GET['project_id'] ?? null;
if (!$project_id) {
    header('Location: project_manager.php');
    exit;
}

$db = new Database();
$user_id = $_SESSION['user_id'];

// Verify project ownership
$project = $db->getProject($user_id, $project_id);
if (!$project) {
    header('Location: project_manager.php');
    exit;
}

// Handle topic saving
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_topics') {
    $topics_data = $_POST['topics'] ?? [];
    $db->saveProjectTopics($project_id, $topics_data);
    $_SESSION['message'] = 'topics_saved';
    header('Location: content_generator.php?project_id=' . $project_id);
    exit;
}

// Handle AI agent saving
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_ai_agent') {
    $agent_name = $_POST['agent_name'] ?? '';
    $agent_prompt = $_POST['agent_prompt'] ?? '';
    
    if (!empty($agent_name) && !empty($agent_prompt)) {
        // Save to session for demo (in production, save to database)
        if (!isset($_SESSION['ai_agents'])) {
            $_SESSION['ai_agents'] = [];
        }
        $_SESSION['ai_agents'][] = [
            'name' => $agent_name,
            'prompt' => $agent_prompt,
            'id' => uniqid()
        ];
        $_SESSION['message'] = 'agent_saved';
    }
    header('Location: content_generator.php?project_id=' . $project_id);
    exit;
}

// Get project content
$topics = $db->getProjectTopics($project_id);
$generated_content = $db->getProjectContent($project_id);

// Get AI agents from session
$ai_agents = $_SESSION['ai_agents'] ?? [];
// Add default Lily SEO agent
$default_agents = [
    [
        'name' => 'Lily SEO',
        'prompt' => 'You are Lilly SEO, a content strategist and SEO expert with over 10 years of hands-on experience. Your voice is direct, conversational, and mentor-like. Include 3 specific anecdotes from your professional history, 2 strong contrarian opinions, actionable non-obvious strategies, and audience-specific nuance.',
        'id' => 'lily-seo'
    ]
];
$ai_agents = array_merge($default_agents, $ai_agents);

// Check for messages
$message = $_SESSION['message'] ?? '';
unset($_SESSION['message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Content Generator - <?php echo htmlspecialchars($project['name']); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.2.96/css/materialdesignicons.min.css" rel="stylesheet">
    <link rel="stylesheet" href="styles.css">
</head>
<body class="enhanced-generator">
    <?php include 'ui_components.php'; echo createHeader('generator'); ?>
    
    <div class="generator-container">
        <?php if ($message === 'topics_saved'): ?>
            <div class="alert alert-success">
                <i class="mdi mdi-check-circle"></i>
                <span>Topics saved successfully!</span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            </div>
        <?php elseif ($message === 'agent_saved'): ?>
            <div class="alert alert-success">
                <i class="mdi mdi-check-circle"></i>
                <span>AI Agent saved successfully!</span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            </div>
        <?php endif; ?>
        
        <div class="advanced-generator-layout">
            <!-- Left Sidebar - Configuration -->
            <div class="config-sidebar">
                <!-- Topics Management -->
                <div class="modern-card">
                    <div class="modern-card-header">
                        <h2><i class="mdi mdi-format-list-bulleted"></i> Topics</h2>
                    </div>
                    <div class="modern-card-body">
                        <form id="topics-form" method="POST">
                            <input type="hidden" name="action" value="save_topics">
                            <div id="topics-container">
                                <?php if (!empty($topics)): ?>
                                    <?php foreach ($topics as $index => $topic): ?>
                                        <div class="topic-input-group">
                                            <input type="text" name="topics[]" class="enhanced-input topic-input" value="<?php echo htmlspecialchars($topic); ?>" placeholder="Enter topic/title">
                                            <button type="button" class="btn-remove-topic">
                                                <i class="mdi mdi-close"></i>
                                            </button>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="topic-input-group">
                                        <input type="text" name="topics[]" class="enhanced-input topic-input" placeholder="Enter topic/title">
                                        <button type="button" class="btn-remove-topic" style="display: none;">
                                            <i class="mdi mdi-close"></i>
                                        </button>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div style="display: flex; gap: 10px; margin-top: 15px;">
                                <button type="button" id="add-topic" class="enhanced-btn enhanced-btn-secondary">
                                    <i class="mdi mdi-plus"></i> Add Topic
                                </button>
                                <button type="submit" class="enhanced-btn enhanced-btn-primary">
                                    <i class="mdi mdi-content-save"></i> Save Topics
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- AI Agents Management -->
                <div class="modern-card" style="margin-top: 2rem;">
                    <div class="modern-card-header">
                        <h2><i class="mdi mdi-robot"></i> AI Agents</h2>
                    </div>
                    <div class="modern-card-body">
                        <div class="ai-agents-section">
                            <?php foreach ($ai_agents as $agent): ?>
                                <div class="agent-item">
                                    <div class="agent-header">
                                        <span class="agent-name"><?php echo htmlspecialchars($agent['name']); ?></span>
                                        <div class="agent-actions">
                                            <?php if ($agent['id'] !== 'lily-seo'): ?>
                                                <button type="button" class="btn-danger btn-sm delete-agent" data-agent-id="<?php echo $agent['id']; ?>">
                                                    <i class="mdi mdi-delete"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="agent-prompt"><?php echo htmlspecialchars(substr($agent['prompt'], 0, 100)) . '...'; ?></div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <div class="collapsible-section">
                            <div class="collapsible-header">
                                <h4><i class="mdi mdi-plus-circle"></i> Add New AI Agent</h4>
                                <i class="mdi mdi-chevron-down"></i>
                            </div>
                            <div class="collapsible-content">
                                <form id="ai-agent-form" method="POST">
                                    <input type="hidden" name="action" value="save_ai_agent">
                                    <div class="enhanced-form-group">
                                        <label>Agent Name</label>
                                        <input type="text" name="agent_name" class="enhanced-input" placeholder="e.g., Technical Writer" required>
                                    </div>
                                    <div class="enhanced-form-group">
                                        <label>Agent Persona & Prompt</label>
                                        <textarea name="agent_prompt" class="enhanced-textarea" placeholder="Describe the AI agent's persona, writing style, expertise areas, and any specific instructions..." required></textarea>
                                    </div>
                                    <button type="submit" class="enhanced-btn enhanced-btn-primary">
                                        <i class="mdi mdi-robot"></i> Save Agent
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Content Settings -->
                <div class="modern-card" style="margin-top: 2rem;">
                    <div class="modern-card-header">
                        <h2><i class="mdi mdi-tune"></i> Content Settings</h2>
                    </div>
                    <div class="modern-card-body">
                        <div class="enhanced-form-group">
                            <label><i class="mdi mdi-format-list-bulleted-type"></i> Content Type</label>
                            <select id="content-type" class="enhanced-select">
                                <option value="article">Article</option>
                                <option value="blog-post">Blog Post</option>
                                <option value="listicle">Listicle</option>
                                <option value="how-to-guide">How-To Guide</option>
                                <option value="tutorial">Tutorial</option>
                                <option value="case-study">Case Study</option>
                                <option value="news-post">News Post</option>
                                <option value="product-description">Product Description</option>
                            </select>
                        </div>

                        <div class="enhanced-form-group">
                            <label><i class="mdi mdi-account-voice"></i> AI Writer Persona</label>
                            <select id="ai-persona" class="enhanced-select">
                                <?php foreach ($ai_agents as $agent): ?>
                                    <option value="<?php echo $agent['id']; ?>" <?php echo $agent['id'] === 'lily-seo' ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($agent['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="enhanced-form-group">
                            <label><i class="mdi mdi-microphone"></i> Tone/Voice</label>
                            <select id="tone-voice" class="enhanced-select">
                                <option value="professional">Professional</option>
                                <option value="conversational" selected>Conversational</option>
                                <option value="authoritative">Authoritative</option>
                                <option value="friendly">Friendly</option>
                                <option value="humorous">Humorous</option>
                                <option value="storytelling">Storytelling</option>
                            </select>
                        </div>

                        <div class="enhanced-form-group">
                            <label><i class="mdi mdi-counter"></i> Word Count</label>
                            <input type="number" id="word-count" class="enhanced-input" value="1500" min="500" max="5000">
                        </div>
                    </div>
                </div>

                <!-- SEO & Advanced Settings -->
                <div class="modern-card" style="margin-top: 2rem;">
                    <div class="modern-card-header">
                        <h2><i class="mdi mdi-seo"></i> SEO & Advanced</h2>
                    </div>
                    <div class="modern-card-body">
                        <div class="enhanced-form-group">
                            <label>Focus Keyword</label>
                            <input type="text" id="focus-keyword" class="enhanced-input" placeholder="Primary keyword (optional)">
                        </div>
                        
                        <div class="enhanced-form-group">
                            <label>Secondary Keywords</label>
                            <textarea id="secondary-keywords" class="enhanced-textarea" placeholder="Comma-separated secondary keywords (optional)"></textarea>
                        </div>

                        <div class="enhanced-form-group">
                            <label>Extra Instructions</label>
                            <textarea id="extra-prompt" class="enhanced-textarea" placeholder="Any additional instructions for the AI..."></textarea>
                        </div>

                        <div class="enhanced-form-group">
                            <label>Internal Links per Article</label>
                            <input type="number" id="min-links" class="enhanced-input" value="5" min="3" max="20">
                        </div>

                        <div class="checkbox-group">
                            <label>
                                <input type="checkbox" id="uncensored-mode">
                                <span class="checkmark"></span>
                                Uncensored Mode
                            </label>
                            <small>Generate content without restrictions</small>
                        </div>

                        <div class="checkbox-group">
                            <label>
                                <input type="checkbox" id="enable-linking" checked>
                                <span class="checkmark"></span>
                                Enable Intelligent Linking
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="modern-card" style="margin-top: 2rem;">
                    <div class="modern-card-body">
                        <button id="generate-all" class="enhanced-btn enhanced-btn-primary enhanced-btn-large w-full">
                            <i class="mdi mdi-auto-fix"></i> Generate All Content
                        </button>
                        <button id="export-all" class="enhanced-btn enhanced-btn-secondary w-full" style="margin-top: 1rem;" <?php echo empty($generated_content) ? 'disabled' : ''; ?>>
                            <i class="mdi mdi-export"></i> Export All Articles
                        </button>
                    </div>
                </div>
            </div>

            <!-- Main Content Area -->
            <div class="content-area">
                <div class="generation-header slide-up">
                    <h1><?php echo htmlspecialchars($project['name']); ?></h1>
                    <p>Configure your content settings and generate high-quality, SEO-optimized articles</p>
                </div>

                <!-- Generation Progress -->
                <div id="generation-progress" class="enhanced-progress" style="display: none;">
                    <div class="progress-header">
                        <h3><i class="mdi mdi-progress-clock"></i> Generating Content</h3>
                        <span id="progress-text">Initializing...</span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" id="progress-fill"></div>
                    </div>
                    <div class="progress-stats">
                        <div class="stat-item">
                            <span class="stat-value" id="stat-completed">0</span>
                            <span class="stat-label">Completed</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-value" id="stat-remaining">0</span>
                            <span class="stat-label">Remaining</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-value" id="stat-success">0</span>
                            <span class="stat-label">Success</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-value" id="stat-failed">0</span>
                            <span class="stat-label">Failed</span>
                        </div>
                    </div>
                </div>

                <!-- Content Output -->
                <div id="content-output" class="fade-in">
                    <?php if (!empty($generated_content)): ?>
                        <?php foreach ($generated_content as $content): ?>
                            <div class="enhanced-article" data-article-id="<?php echo $content['id']; ?>" data-topic="<?php echo htmlspecialchars($content['topic']); ?>">
                                <div class="enhanced-article-header">
                                    <h3 class="article-title"><?php echo htmlspecialchars($content['topic']); ?></h3>
                                    <div class="article-meta-enhanced">
                                        <span><i class="mdi mdi-counter"></i> <?php echo $content['word_count']; ?> words</span>
                                        <span><i class="mdi mdi-calendar"></i> <?php echo date('M j, H:i', strtotime($content['created_at'])); ?></span>
                                        <span class="badge badge-success">Generated</span>
                                    </div>
                                </div>
                                <div class="enhanced-article-content">
                                    <?php echo $content['content']; ?>
                                </div>
                                <div class="enhanced-article-actions">
                                    <button class="article-action-btn action-primary copy-article" data-content="<?php echo htmlspecialchars($content['content']); ?>">
                                        <i class="mdi mdi-content-copy"></i> Copy
                                    </button>
                                    <button class="article-action-btn action-secondary export-article" data-content="<?php echo htmlspecialchars($content['content']); ?>" data-title="<?php echo htmlspecialchars($content['topic']); ?>">
                                        <i class="mdi mdi-export"></i> Export
                                    </button>
                                    <button class="article-action-btn action-warning regenerate-article" data-article-id="<?php echo $content['id']; ?>" data-topic="<?php echo htmlspecialchars($content['topic']); ?>">
                                        <i class="mdi mdi-refresh"></i> Regenerate
                                    </button>
                                    <button class="article-action-btn action-secondary pause-article" style="display: none;">
                                        <i class="mdi mdi-pause"></i> Pause
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="mdi mdi-file-document-outline" style="font-size: 4rem;"></i>
                            <h3>No Content Generated Yet</h3>
                            <p>Configure your settings and click "Generate All Content" to start creating amazing content</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Floating Action Buttons -->
    <div class="floating-actions">
        <button class="floating-btn" id="scroll-to-top" title="Scroll to Top">
            <i class="mdi mdi-chevron-up"></i>
        </button>
        <button class="floating-btn" id="scroll-to-bottom" title="Scroll to Bottom">
            <i class="mdi mdi-chevron-down"></i>
        </button>
    </div>

    <script src="utilities.js"></script>
    <script>
        // Initialize theme manager
        const themeManager = new ThemeManager();
        
        // Initialize content manager
        const contentManager = new ContentGenerationManager();
        
        // Collapsible sections
        document.querySelectorAll('.collapsible-header').forEach(header => {
            header.addEventListener('click', function() {
                const section = this.closest('.collapsible-section');
                section.classList.toggle('active');
                
                const icon = this.querySelector('.mdi-chevron-down');
                if (icon) {
                    icon.className = section.classList.contains('active') ? 
                        'mdi mdi-chevron-up' : 'mdi mdi-chevron-down';
                }
            });
        });

        // Topic management
        document.getElementById('add-topic').addEventListener('click', function() {
            const container = document.getElementById('topics-container');
            const newGroup = document.createElement('div');
            newGroup.className = 'topic-input-group';
            newGroup.innerHTML = `
                <input type="text" name="topics[]" class="enhanced-input topic-input" placeholder="Enter topic/title">
                <button type="button" class="btn-remove-topic">
                    <i class="mdi mdi-close"></i>
                </button>
            `;
            container.appendChild(newGroup);
            
            // Add remove functionality
            newGroup.querySelector('.btn-remove-topic').addEventListener('click', function() {
                if (container.children.length > 1) {
                    container.removeChild(newGroup);
                }
            });
        });

        // Initialize remove buttons for existing topics
        document.querySelectorAll('.btn-remove-topic').forEach(btn => {
            btn.addEventListener('click', function() {
                const container = document.getElementById('topics-container');
                if (container.children.length > 1) {
                    this.closest('.topic-input-group').remove();
                }
            });
        });

        // Generate All Content with Real AI
        document.getElementById('generate-all').addEventListener('click', function() {
            // Get ALL topic inputs including dynamically added ones
            const topicInputs = document.querySelectorAll('.topic-input');
            const topics = Array.from(topicInputs)
                .map(input => input.value.trim())
                .filter(topic => topic !== '');
            
            console.log('Found topics:', topics); // Debug log
            
            if (topics.length === 0) {
                showNotification('Please add at least one topic', 'error');
                return;
            }

            // Get selected AI agent
            const aiPersonaSelect = document.getElementById('ai-persona');
            const selectedAgentId = aiPersonaSelect.value;
            const selectedAgentName = aiPersonaSelect.options[aiPersonaSelect.selectedIndex].text;
            const aiAgents = <?php echo json_encode($ai_agents); ?>;

            const selectedAgent = aiAgents.find(agent => agent.id === selectedAgentId);
            
            const config = {
                project_id: <?php echo $project_id; ?>,
                topics: topics,
                content_type: document.getElementById('content-type').value,
                tone_voice: document.getElementById('tone-voice').value,
                word_count: parseInt(document.getElementById('word-count').value),
                ai_persona: selectedAgentName,
                ai_prompt: selectedAgent ? selectedAgent.prompt : '',
                focus_keyword: document.getElementById('focus-keyword').value,
                secondary_keywords: document.getElementById('secondary-keywords').value,
                extra_prompt: document.getElementById('extra-prompt').value,
                min_links: parseInt(document.getElementById('min-links').value),
                uncensored_mode: document.getElementById('uncensored-mode').checked,
                enable_linking: document.getElementById('enable-linking').checked
            };

            console.log('Generation config:', config); // Debug log
            contentManager.generateAllContent(config);
        });

        // Export All Articles
        document.getElementById('export-all').addEventListener('click', function() {
            contentManager.exportAllArticles(<?php echo $project_id; ?>);
        });

        // Copy article functionality
        document.addEventListener('click', function(e) {
            if (e.target.closest('.copy-article')) {
                const btn = e.target.closest('.copy-article');
                const content = btn.getAttribute('data-content');
                
                navigator.clipboard.writeText(content).then(() => {
                    showNotification('Article copied to clipboard!', 'success');
                }).catch(() => {
                    showNotification('Failed to copy article', 'error');
                });
            }
        });

        // Export single article
        document.addEventListener('click', function(e) {
            if (e.target.closest('.export-article')) {
                const btn = e.target.closest('.export-article');
                const content = btn.getAttribute('data-content');
                const title = btn.getAttribute('data-title');
                
                contentManager.exportSingleArticle(content, title);
            }
        });

        // Regenerate article
        document.addEventListener('click', function(e) {
            if (e.target.closest('.regenerate-article')) {
                const btn = e.target.closest('.regenerate-article');
                const articleId = btn.getAttribute('data-article-id');
                const topic = btn.getAttribute('data-topic');
                
                // Get current configuration
                const aiPersonaSelect = document.getElementById('ai-persona');
                const selectedAgentName = aiPersonaSelect.options[aiPersonaSelect.selectedIndex].text;
                const aiAgents = <?php echo json_encode($ai_agents); ?>;
                const selectedAgent = aiAgents.find(agent => agent.id === aiPersonaSelect.value);

                const config = {
                    project_id: <?php echo $project_id; ?>,
                    topics: [topic],
                    content_type: document.getElementById('content-type').value,
                    tone_voice: document.getElementById('tone-voice').value,
                    word_count: parseInt(document.getElementById('word-count').value),
                    ai_persona: selectedAgentName,
                    ai_prompt: selectedAgent ? selectedAgent.prompt : '',
                    focus_keyword: document.getElementById('focus-keyword').value,
                    secondary_keywords: document.getElementById('secondary-keywords').value,
                    extra_prompt: document.getElementById('extra-prompt').value,
                    min_links: parseInt(document.getElementById('min-links').value),
                    uncensored_mode: document.getElementById('uncensored-mode').checked,
                    enable_linking: document.getElementById('enable-linking').checked
                };

                contentManager.regenerateArticle(articleId, topic, config);
            }
        });

        // Pause article generation
        document.addEventListener('click', function(e) {
            if (e.target.closest('.pause-article')) {
                const btn = e.target.closest('.pause-article');
                const articleElement = btn.closest('.enhanced-article');
                const topic = articleElement.getAttribute('data-topic');
                
                contentManager.togglePauseGeneration(topic, btn);
            }
        });

        // Delete AI Agent
        document.querySelectorAll('.delete-agent').forEach(btn => {
            btn.addEventListener('click', function() {
                const agentId = this.getAttribute('data-agent-id');
                if (confirm('Are you sure you want to delete this AI agent?')) {
                    // Remove from UI
                    this.closest('.agent-item').remove();
                    showNotification('AI agent deleted', 'success');
                    
                    // In production, make API call to delete from database
                    console.log('Delete agent:', agentId);
                }
            });
        });

        // Floating actions
        document.getElementById('scroll-to-top').addEventListener('click', () => {
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });

        document.getElementById('scroll-to-bottom').addEventListener('click', () => {
            window.scrollTo({ top: document.body.scrollHeight, behavior: 'smooth' });
        });

        // Notification function
        function showNotification(message, type = 'info') {
            // Remove existing notifications
            document.querySelectorAll('.alert').forEach(alert => alert.remove());
            
            const notification = document.createElement('div');
            notification.className = `alert alert-${type}`;
            notification.innerHTML = `
                <i class="mdi mdi-${getNotificationIcon(type)}"></i>
                <span>${message}</span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            `;
            
            const container = document.querySelector('.generator-container');
            container.insertBefore(notification, container.firstChild);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 5000);
            
            // Close button functionality
            notification.querySelector('.alert-close').addEventListener('click', () => {
                notification.parentNode.removeChild(notification);
            });
        }

        function getNotificationIcon(type) {
            const icons = {
                info: 'information',
                success: 'check-circle',
                warning: 'alert',
                error: 'alert-circle'
            };
            return icons[type] || 'information';
        }

        // Close alert buttons
        document.querySelectorAll('.alert-close').forEach(btn => {
            btn.addEventListener('click', function() {
                this.closest('.alert').remove();
            });
        });

        // Enable export button when content exists
        if (document.querySelectorAll('.enhanced-article').length > 0) {
            document.getElementById('export-all').disabled = false;
        }

        // Debug: Log all topic inputs on page load
        document.addEventListener('DOMContentLoaded', function() {
            const topicInputs = document.querySelectorAll('.topic-input');
            console.log('Initial topic inputs:', topicInputs.length);
            topicInputs.forEach((input, index) => {
                console.log(`Topic ${index + 1}:`, input.value);
            });
        });
    </script>
</body>
</html>