<?php
session_start();

class Database {
    private $pdo;
    private $db_file = 'content_writer.db';
    
    public function __construct() {
        $this->connect();
    }
    
    private function connect() {
        try {
            $this->pdo = new PDO("sqlite:" . $this->db_file);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            die("Database connection failed. Please check permissions.");
        }
    }
    
    public function autoSetup() {
        try {
            // Users table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                email VARCHAR(255) UNIQUE NOT NULL,
                password_hash VARCHAR(255) NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                theme_preference VARCHAR(10) DEFAULT 'light'
            )");
            
            // API keys table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS api_keys (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                api_key VARCHAR(255) NOT NULL,
                is_active BOOLEAN DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                last_used DATETIME,
                usage_count INTEGER DEFAULT 0,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )");
            
            // Projects table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS projects (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                name VARCHAR(255) NOT NULL,
                description TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
            )");
            
            // Project topics table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS project_topics (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                project_id INTEGER NOT NULL,
                topic TEXT NOT NULL,
                sort_order INTEGER DEFAULT 0,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE
            )");
            
            // Generated content table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS generated_content (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                project_id INTEGER NOT NULL,
                topic TEXT NOT NULL,
                content TEXT NOT NULL,
                word_count INTEGER DEFAULT 0,
                meta_description TEXT,
                focus_keyword TEXT,
                secondary_keywords TEXT,
                internal_links TEXT,
                ai_model_used VARCHAR(50),
                generation_time INTEGER,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE
            )");
            
            // Usage statistics table
            $this->pdo->exec("CREATE TABLE IF NOT EXISTS usage_stats (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                project_id INTEGER,
                api_key_id INTEGER,
                tokens_used INTEGER DEFAULT 0,
                articles_generated INTEGER DEFAULT 0,
                date_recorded DATE DEFAULT CURRENT_DATE,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE,
                FOREIGN KEY (api_key_id) REFERENCES api_keys(id) ON DELETE CASCADE
            )");
            
            return true;
        } catch (PDOException $e) {
            error_log("Database setup error: " . $e->getMessage());
            return false;
        }
    }
    
    public function registerUser($email, $password) {
        try {
            // Check if email already exists
            $stmt = $this->pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            
            if ($stmt->fetch()) {
                return ['success' => false, 'error' => 'Email already registered'];
            }
            
            // Create user
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $this->pdo->prepare("INSERT INTO users (email, password_hash) VALUES (?, ?)");
            $stmt->execute([$email, $password_hash]);
            
            $user_id = $this->pdo->lastInsertId();
            
            return ['success' => true, 'user_id' => $user_id];
        } catch (PDOException $e) {
            error_log("Registration error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Registration failed: ' . $e->getMessage()];
        }
    }
    
    public function loginUser($email, $password) {
        try {
            $stmt = $this->pdo->prepare("SELECT id, email, password_hash FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password_hash'])) {
                return $user;
            }
            return false;
        } catch (PDOException $e) {
            error_log("Login error: " . $e->getMessage());
            return false;
        }
    }
    
    public function createProject($user_id, $name, $description = '') {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO projects (user_id, name, description) VALUES (?, ?, ?)");
            $stmt->execute([$user_id, $name, $description]);
            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Project creation error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getProject($user_id, $project_id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM projects WHERE id = ? AND user_id = ?");
            $stmt->execute([$project_id, $user_id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Get project error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getUserProjects($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT p.*, 
                       (SELECT COUNT(*) FROM generated_content WHERE project_id = p.id) as article_count,
                       (SELECT COUNT(*) FROM project_topics WHERE project_id = p.id) as topic_count
                FROM projects p 
                WHERE p.user_id = ? 
                ORDER BY p.updated_at DESC
            ");
            $stmt->execute([$user_id]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Get user projects error: " . $e->getMessage());
            return [];
        }
    }
    
    public function deleteProject($user_id, $project_id) {
        try {
            $this->pdo->beginTransaction();
            
            // Delete related data first
            $stmt = $this->pdo->prepare("DELETE FROM generated_content WHERE project_id = ?");
            $stmt->execute([$project_id]);
            
            $stmt = $this->pdo->prepare("DELETE FROM project_topics WHERE project_id = ?");
            $stmt->execute([$project_id]);
            
            $stmt = $this->pdo->prepare("DELETE FROM projects WHERE id = ? AND user_id = ?");
            $stmt->execute([$project_id, $user_id]);
            
            $this->pdo->commit();
            return true;
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Delete project error: " . $e->getMessage());
            return false;
        }
    }
    
    public function addApiKey($user_id, $api_key) {
        try {
            // Validate API key format (basic check)
            if (empty($api_key) || strlen($api_key) < 10) {
                return ['success' => false, 'error' => 'Invalid API key format'];
            }
            
            // Check if this API key already exists for the user
            $stmt = $this->pdo->prepare("SELECT id FROM api_keys WHERE user_id = ? AND api_key = ?");
            $stmt->execute([$user_id, $api_key]);
            if ($stmt->fetch()) {
                return ['success' => false, 'error' => 'This API key is already added'];
            }
            
            $stmt = $this->pdo->prepare("INSERT INTO api_keys (user_id, api_key) VALUES (?, ?)");
            $stmt->execute([$user_id, $api_key]);
            
            return ['success' => true, 'key_id' => $this->pdo->lastInsertId()];
        } catch (PDOException $e) {
            error_log("Add API key error: " . $e->getMessage());
            return ['success' => false, 'error' => 'Failed to add API key: ' . $e->getMessage()];
        }
    }
    
public function getActiveApiKeys($user_id) {
    try {
        $stmt = $this->pdo->prepare("
            SELECT id, api_key, is_active, usage_count, last_used 
            FROM api_keys 
            WHERE user_id = ? AND is_active = 1 
            ORDER BY last_used ASC, usage_count ASC
        ");
        $stmt->execute([$user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Get active API keys error: " . $e->getMessage());
        return [];
    }
}
    
    public function getUserApiKeys($user_id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM api_keys WHERE user_id = ? ORDER BY created_at DESC");
            $stmt->execute([$user_id]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Get user API keys error: " . $e->getMessage());
            return [];
        }
    }
    
    public function deleteApiKey($user_id, $key_id) {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM api_keys WHERE id = ? AND user_id = ?");
            $stmt->execute([$key_id, $user_id]);
            return $stmt->rowCount() > 0;
        } catch (PDOException $e) {
            error_log("Delete API key error: " . $e->getMessage());
            return false;
        }
    }
    
    public function updateApiKeyUsage($key_id) {
        try {
            $stmt = $this->pdo->prepare("UPDATE api_keys SET last_used = CURRENT_TIMESTAMP, usage_count = usage_count + 1 WHERE id = ?");
            $stmt->execute([$key_id]);
            return true;
        } catch (PDOException $e) {
            error_log("Update API key usage error: " . $e->getMessage());
            return false;
        }
    }
    
    public function saveGeneratedContent($project_id, $topic, $content, $metadata = []) {
        try {
            $word_count = str_word_count(strip_tags($content));
            
            $stmt = $this->pdo->prepare("
                INSERT INTO generated_content 
                (project_id, topic, content, word_count, focus_keyword, secondary_keywords, ai_model_used, generation_time) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $project_id,
                $topic,
                $content,
                $word_count,
                $metadata['focus_keyword'] ?? '',
                $metadata['secondary_keywords'] ?? '',
                $metadata['ai_model'] ?? 'gemini-2.0-flash',
                $metadata['generation_time'] ?? 0
            ]);
            
            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Save generated content error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getProjectContent($project_id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM generated_content WHERE project_id = ? ORDER BY created_at DESC");
            $stmt->execute([$project_id]);
            return $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("Get project content error: " . $e->getMessage());
            return [];
        }
    }
    
    public function getProjectTopics($project_id) {
        try {
            $stmt = $this->pdo->prepare("SELECT topic FROM project_topics WHERE project_id = ? ORDER BY sort_order ASC");
            $stmt->execute([$project_id]);
            return $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
        } catch (PDOException $e) {
            error_log("Get project topics error: " . $e->getMessage());
            return [];
        }
    }
    
    public function saveProjectTopics($project_id, $topics) {
        try {
            $this->pdo->beginTransaction();
            
            // Clear existing topics
            $stmt = $this->pdo->prepare("DELETE FROM project_topics WHERE project_id = ?");
            $stmt->execute([$project_id]);
            
            // Insert new topics
            $stmt = $this->pdo->prepare("INSERT INTO project_topics (project_id, topic, sort_order) VALUES (?, ?, ?)");
            $sort_order = 0;
            foreach ($topics as $topic) {
                if (trim($topic) !== '') {
                    $stmt->execute([$project_id, trim($topic), $sort_order++]);
                }
            }
            
            $this->pdo->commit();
            return true;
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            error_log("Save project topics error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getUserUsageStats($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT 
                    COUNT(DISTINCT p.id) as total_projects,
                    COUNT(gc.id) as total_articles,
                    COALESCE(SUM(gc.word_count), 0) as total_words,
                    COUNT(DISTINCT ak.id) as total_api_keys
                FROM users u
                LEFT JOIN projects p ON p.user_id = u.id
                LEFT JOIN generated_content gc ON gc.project_id = p.id
                LEFT JOIN api_keys ak ON ak.user_id = u.id
                WHERE u.id = ?
            ");
            $stmt->execute([$user_id]);
            return $stmt->fetch();
        } catch (PDOException $e) {
            error_log("Get usage stats error: " . $e->getMessage());
            return ['total_projects' => 0, 'total_articles' => 0, 'total_words' => 0, 'total_api_keys' => 0];
        }
    }
    
    public function recordUsage($user_id, $project_id, $api_key_id, $tokens_used) {
        try {
            // Check if record exists for today
            $stmt = $this->pdo->prepare("
                SELECT id FROM usage_stats 
                WHERE user_id = ? AND date_recorded = DATE('now')
            ");
            $stmt->execute([$user_id]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Update existing record
                $stmt = $this->pdo->prepare("
                    UPDATE usage_stats 
                    SET tokens_used = tokens_used + ?, articles_generated = articles_generated + 1
                    WHERE user_id = ? AND date_recorded = DATE('now')
                ");
                $stmt->execute([$tokens_used, $user_id]);
            } else {
                // Insert new record
                $stmt = $this->pdo->prepare("
                    INSERT INTO usage_stats (user_id, project_id, api_key_id, tokens_used, articles_generated)
                    VALUES (?, ?, ?, ?, 1)
                ");
                $stmt->execute([$user_id, $project_id, $api_key_id, $tokens_used]);
            }
            
            return true;
        } catch (PDOException $e) {
            error_log("Record usage error: " . $e->getMessage());
            return false;
        }
    }
}

?>