<?php
session_start();
require_once 'database.php';

class ExportManager {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function exportArticleAsHTML($content, $title, $metadata = []) {
        $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . htmlspecialchars($title) . '</title>
    <style>
        body { 
            font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif; 
            line-height: 1.7; 
            margin: 0;
            padding: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            box-shadow: 0 20px 60px rgba(0,0,0,0.1);
            border-radius: 20px;
            overflow: hidden;
            margin: 40px auto;
        }
        .header {
            background: linear-gradient(135deg, #2c3e50, #3498db);
            color: white;
            padding: 3rem 2rem;
            text-align: center;
        }
        .header h1 {
            margin: 0;
            font-size: 2.5rem;
            font-weight: 700;
            line-height: 1.2;
        }
        .meta-info {
            background: #f8f9fa;
            padding: 1.5rem 2rem;
            border-bottom: 1px solid #e9ecef;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            font-size: 0.9rem;
            color: #6c757d;
        }
        .meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        .meta-item i {
            color: #3498db;
        }
        .content {
            padding: 3rem 2rem;
            font-size: 1.1rem;
            color: #2c3e50;
        }
        .content h1 {
            color: #2c3e50;
            border-bottom: 3px solid #3498db;
            padding-bottom: 0.5rem;
            margin-top: 2rem;
            margin-bottom: 1.5rem;
            font-size: 2.2rem;
        }
        .content h2 {
            color: #34495e;
            margin-top: 2rem;
            margin-bottom: 1rem;
            font-size: 1.8rem;
        }
        .content h3 {
            color: #7f8c8d;
            margin-top: 1.5rem;
            margin-bottom: 1rem;
            font-size: 1.4rem;
        }
        .content p {
            margin-bottom: 1.5rem;
            text-align: justify;
        }
        .content ul, .content ol {
            margin: 1.5rem 0;
            padding-left: 2rem;
        }
        .content li {
            margin-bottom: 0.75rem;
            line-height: 1.6;
        }
        .content strong {
            color: #2c3e50;
            font-weight: 700;
        }
        .content em {
            color: #7f8c8d;
            font-style: italic;
        }
        .internal-link {
            color: #e74c3c;
            text-decoration: none;
            font-weight: 600;
            border-bottom: 2px dotted #e74c3c;
            transition: all 0.3s ease;
        }
        .internal-link:hover {
            color: #c0392b;
            border-bottom-style: solid;
        }
        .footer {
            background: #2c3e50;
            color: white;
            text-align: center;
            padding: 2rem;
            font-size: 0.9rem;
        }
        .footer a {
            color: #3498db;
            text-decoration: none;
        }
        .footer a:hover {
            text-decoration: underline;
        }
        @media (max-width: 768px) {
            .container {
                margin: 20px;
                border-radius: 15px;
            }
            .header {
                padding: 2rem 1.5rem;
            }
            .header h1 {
                font-size: 2rem;
            }
            .content {
                padding: 2rem 1.5rem;
            }
            .meta-info {
                grid-template-columns: 1fr;
                padding: 1rem 1.5rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>' . htmlspecialchars($title) . '</h1>
        </div>';
        
        if (!empty($metadata)) {
            $html .= '<div class="meta-info">
                <div class="meta-item">
                    <i>📄</i>
                    <span><strong>Document:</strong> ' . htmlspecialchars($title) . '</span>
                </div>';
            if (!empty($metadata['word_count'])) {
                $html .= '<div class="meta-item">
                    <i>📊</i>
                    <span><strong>Word Count:</strong> ' . $metadata['word_count'] . '</span>
                </div>';
            }
            if (!empty($metadata['created_at'])) {
                $html .= '<div class="meta-item">
                    <i>📅</i>
                    <span><strong>Generated:</strong> ' . date('F j, Y g:i A', strtotime($metadata['created_at'])) . '</span>
                </div>';
            }
            $html .= '<div class="meta-item">
                    <i>⚡</i>
                    <span><strong>Source:</strong> Bulk AI Content Writer</span>
                </div>
            </div>';
        }
        
        $html .= '<div class="content">' . $content . '</div>
        <div class="footer">
            <p>Generated with ❤️ using <strong>Bulk AI Content Writer</strong> | ' . date('F Y') . '</p>
            <p>This content was AI-generated and professionally optimized for SEO and readability.</p>
        </div>
    </div>
</body>
</html>';
        
        return $html;
    }
    
    public function exportArticleAsDOC($content, $title) {
        // Enhanced DOC format with better styling
        $doc = '<!DOCTYPE html>
<html xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:w="urn:schemas-microsoft-com:office:word" xmlns="http://www.w3.org/TR/REC-html40">
<head>
    <meta charset="UTF-8">
    <title>' . htmlspecialchars($title) . '</title>
    <style>
        @page {
            size: A4;
            margin: 2cm;
        }
        body {
            font-family: "Calibri", sans-serif;
            line-height: 1.5;
            margin: 0;
            padding: 0;
            color: #2c3e50;
        }
        .doc-container {
            max-width: 21cm;
            margin: 0 auto;
            padding: 2cm;
        }
        .doc-header {
            text-align: center;
            border-bottom: 3px double #3498db;
            padding-bottom: 1cm;
            margin-bottom: 1.5cm;
        }
        .doc-title {
            font-size: 28pt;
            font-weight: bold;
            color: #2c3e50;
            margin: 0 0 0.5cm 0;
        }
        .doc-meta {
            font-size: 11pt;
            color: #7f8c8d;
            margin: 0.5cm 0;
        }
        .doc-content {
            font-size: 12pt;
        }
        .doc-content h1 {
            font-size: 18pt;
            color: #2c3e50;
            border-bottom: 1px solid #3498db;
            padding-bottom: 0.3cm;
            margin-top: 1.5cm;
            margin-bottom: 0.8cm;
        }
        .doc-content h2 {
            font-size: 16pt;
            color: #34495e;
            margin-top: 1.2cm;
            margin-bottom: 0.6cm;
        }
        .doc-content h3 {
            font-size: 14pt;
            color: #7f8c8d;
            margin-top: 1cm;
            margin-bottom: 0.5cm;
        }
        .doc-content p {
            margin-bottom: 0.6cm;
            text-align: justify;
        }
        .doc-content ul, .doc-content ol {
            margin: 0.6cm 0;
            padding-left: 1.5cm;
        }
        .doc-content li {
            margin-bottom: 0.3cm;
        }
        .doc-content strong {
            font-weight: bold;
            color: #2c3e50;
        }
        .doc-content em {
            font-style: italic;
            color: #7f8c8d;
        }
        .internal-link {
            color: #e74c3c;
            text-decoration: underline;
        }
        .doc-footer {
            text-align: center;
            font-size: 10pt;
            color: #95a5a6;
            margin-top: 2cm;
            padding-top: 0.5cm;
            border-top: 1px solid #bdc3c7;
        }
    </style>
</head>
<body>
    <div class="doc-container">
        <div class="doc-header">
            <h1 class="doc-title">' . htmlspecialchars($title) . '</h1>
            <div class="doc-meta">
                Generated: ' . date('F j, Y') . ' | Source: Bulk AI Content Writer
            </div>
        </div>
        <div class="doc-content">' . $content . '</div>
        <div class="doc-footer">
            <p>Generated with Bulk AI Content Writer • Professional AI Content Generation</p>
        </div>
    </div>
</body>
</html>';
        
        return $doc;
    }
    
    public function exportArticleAsPDF($content, $title, $metadata = []) {
        // HTML-based PDF (in production, use TCPDF or Dompdf)
        $pdf_html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>' . htmlspecialchars($title) . '</title>
    <style>
        body {
            font-family: "Helvetica", "Arial", sans-serif;
            line-height: 1.4;
            margin: 0;
            padding: 0;
            color: #333;
            font-size: 11pt;
        }
        .pdf-container {
            max-width: 100%;
            margin: 0;
            padding: 2cm;
        }
        .pdf-header {
            text-align: center;
            border-bottom: 2px solid #3498db;
            padding-bottom: 1cm;
            margin-bottom: 1.5cm;
        }
        .pdf-title {
            font-size: 20pt;
            font-weight: bold;
            color: #2c3e50;
            margin: 0 0 0.3cm 0;
        }
        .pdf-meta {
            font-size: 9pt;
            color: #666;
            margin: 0.3cm 0;
        }
        .pdf-content {
            font-size: 10pt;
        }
        .pdf-content h1 {
            font-size: 16pt;
            color: #2c3e50;
            border-bottom: 1px solid #bdc3c7;
            padding-bottom: 0.2cm;
            margin-top: 1.2cm;
            margin-bottom: 0.6cm;
        }
        .pdf-content h2 {
            font-size: 14pt;
            color: #34495e;
            margin-top: 1cm;
            margin-bottom: 0.5cm;
        }
        .pdf-content h3 {
            font-size: 12pt;
            color: #7f8c8d;
            margin-top: 0.8cm;
            margin-bottom: 0.4cm;
        }
        .pdf-content p {
            margin-bottom: 0.5cm;
            text-align: justify;
        }
        .pdf-content ul, .pdf-content ol {
            margin: 0.5cm 0;
            padding-left: 1.2cm;
        }
        .pdf-content li {
            margin-bottom: 0.2cm;
        }
        .pdf-footer {
            text-align: center;
            font-size: 8pt;
            color: #999;
            margin-top: 1.5cm;
            padding-top: 0.3cm;
            border-top: 1px solid #ddd;
        }
        .page-break {
            page-break-before: always;
        }
    </style>
</head>
<body>
    <div class="pdf-container">
        <div class="pdf-header">
            <h1 class="pdf-title">' . htmlspecialchars($title) . '</h1>
            <div class="pdf-meta">';
        
        if (!empty($metadata)) {
            if (!empty($metadata['word_count'])) {
                $pdf_html .= 'Word Count: ' . $metadata['word_count'] . ' | ';
            }
            $pdf_html .= 'Generated: ' . date('F j, Y') . ' | Source: Bulk AI Content Writer';
        }
        
        $pdf_html .= '</div>
        </div>
        <div class="pdf-content">' . $content . '</div>
        <div class="pdf-footer">
            <p>Page 1 of 1 | Generated with Bulk AI Content Writer</p>
        </div>
    </div>
</body>
</html>';
        
        return $pdf_html;
    }
    
    public function generatePDFReport($project_id, $content_items) {
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Content Generation Report</title>
    <style>
        body { 
            font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif; 
            margin: 0;
            padding: 0;
            background: #f8f9fa;
        }
        .report-container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        .report-header {
            background: linear-gradient(135deg, #2c3e50, #3498db);
            color: white;
            padding: 3rem 2rem;
            text-align: center;
        }
        .report-header h1 {
            margin: 0;
            font-size: 2.5rem;
            font-weight: 700;
        }
        .report-header p {
            margin: 1rem 0 0 0;
            font-size: 1.1rem;
            opacity: 0.9;
        }
        .report-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            padding: 2rem;
            background: white;
        }
        .stat-card {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 1.5rem;
            border-radius: 15px;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 2.5rem;
            font-weight: 700;
            display: block;
            margin-bottom: 0.5rem;
        }
        .stat-label {
            font-size: 0.9rem;
            opacity: 0.9;
        }
        .articles-section {
            padding: 2rem;
        }
        .article-item {
            background: white;
            border: 1px solid #e9ecef;
            border-radius: 12px;
            padding: 2rem;
            margin-bottom: 1.5rem;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            transition: transform 0.2s ease;
        }
        .article-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        .article-title {
            font-size: 1.4rem;
            font-weight: 600;
            color: #2c3e50;
            margin: 0 0 1rem 0;
        }
        .article-meta {
            display: flex;
            gap: 1.5rem;
            color: #6c757d;
            font-size: 0.9rem;
            margin-bottom: 1rem;
        }
        .article-content {
            line-height: 1.6;
            color: #495057;
        }
        .article-content h1, .article-content h2, .article-content h3 {
            color: #2c3e50;
            margin: 1.5rem 0 1rem 0;
        }
        .report-footer {
            background: #2c3e50;
            color: white;
            text-align: center;
            padding: 2rem;
            margin-top: 2rem;
        }
        @media (max-width: 768px) {
            .report-header {
                padding: 2rem 1rem;
            }
            .report-header h1 {
                font-size: 2rem;
            }
            .report-stats {
                grid-template-columns: 1fr;
                padding: 1rem;
            }
            .articles-section {
                padding: 1rem;
            }
            .article-item {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <div class="report-container">
        <div class="report-header">
            <h1>Content Generation Report</h1>
            <p>Comprehensive overview of all generated content</p>
        </div>
        
        <div class="report-stats">
            <div class="stat-card">
                <span class="stat-value">' . count($content_items) . '</span>
                <span class="stat-label">Total Articles</span>
            </div>
            <div class="stat-card">
                <span class="stat-value">' . array_sum(array_column($content_items, 'word_count')) . '</span>
                <span class="stat-label">Total Words</span>
            </div>
            <div class="stat-card">
                <span class="stat-value">' . date('M j, Y') . '</span>
                <span class="stat-label">Report Date</span>
            </div>
        </div>
        
        <div class="articles-section">';
        
        foreach ($content_items as $item) {
            $html .= '
            <div class="article-item">
                <h2 class="article-title">' . htmlspecialchars($item['topic']) . '</h2>
                <div class="article-meta">
                    <span>📊 ' . $item['word_count'] . ' words</span>
                    <span>📅 ' . date('M j, Y g:i A', strtotime($item['created_at'])) . '</span>
                </div>
                <div class="article-content">' . $item['content'] . '</div>
            </div>';
        }
        
        $html .= '</div>
        
        <div class="report-footer">
            <p>Generated with ❤️ using <strong>Bulk AI Content Writer</strong></p>
            <p>Report generated on ' . date('F j, Y \a\t g:i A') . '</p>
        </div>
    </div>
</body>
</html>';
        
        return $html;
    }
    
    public function exportAllArticles($project_id, $format = 'html') {
        $content_items = $this->db->getProjectContent($project_id);
        $export_files = [];
        
        foreach ($content_items as $item) {
            $filename = $this->sanitizeFilename($item['topic']) . '.' . $format;
            
            switch ($format) {
                case 'html':
                    $content = $this->exportArticleAsHTML($item['content'], $item['topic'], [
                        'word_count' => $item['word_count'],
                        'created_at' => $item['created_at']
                    ]);
                    break;
                    
                case 'doc':
                    $content = $this->exportArticleAsDOC($item['content'], $item['topic']);
                    break;
                    
                case 'pdf':
                    $content = $this->exportArticleAsPDF($item['content'], $item['topic'], [
                        'word_count' => $item['word_count'],
                        'created_at' => $item['created_at']
                    ]);
                    break;
                    
                default:
                    $content = $item['content'];
            }
            
            $export_files[$filename] = $content;
        }
        
        return $export_files;
    }
    
    public function exportAllArticlesAsZip($project_id, $format = 'html') {
        $content_items = $this->db->getProjectContent($project_id);
        
        if (empty($content_items)) {
            return false;
        }
        
        // Create a temporary directory
        $temp_dir = sys_get_temp_dir() . '/content_export_' . uniqid();
        if (!mkdir($temp_dir, 0755, true)) {
            throw new Exception('Failed to create temporary directory');
        }
        
        $files = [];
        
        try {
            foreach ($content_items as $item) {
                $filename = $this->sanitizeFilename($item['topic']) . '.' . $format;
                $filepath = $temp_dir . '/' . $filename;
                
                switch ($format) {
                    case 'html':
                        $content = $this->exportArticleAsHTML($item['content'], $item['topic'], [
                            'word_count' => $item['word_count'],
                            'created_at' => $item['created_at']
                        ]);
                        break;
                        
                    case 'doc':
                        $content = $this->exportArticleAsDOC($item['content'], $item['topic']);
                        break;
                        
                    case 'pdf':
                        $content = $this->exportArticleAsPDF($item['content'], $item['topic'], [
                            'word_count' => $item['word_count'],
                            'created_at' => $item['created_at']
                        ]);
                        break;
                        
                    default:
                        $content = $item['content'];
                }
                
                if (file_put_contents($filepath, $content) === false) {
                    throw new Exception('Failed to write file: ' . $filename);
                }
                $files[] = $filepath;
            }
            
            // Create ZIP file
            $zip_path = $temp_dir . '/content_export.zip';
            $zip = new ZipArchive();
            
            if ($zip->open($zip_path, ZipArchive::CREATE) !== TRUE) {
                throw new Exception('Cannot create ZIP file');
            }
            
            foreach ($files as $file) {
                if (!$zip->addFile($file, basename($file))) {
                    throw new Exception('Failed to add file to ZIP: ' . basename($file));
                }
            }
            
            if (!$zip->close()) {
                throw new Exception('Failed to close ZIP file');
            }
            
            // Read ZIP content
            $zip_content = file_get_contents($zip_path);
            if ($zip_content === false) {
                throw new Exception('Failed to read ZIP file');
            }
            
            return $zip_content;
            
        } catch (Exception $e) {
            // Clean up on error
            $this->cleanupTempFiles($files, $temp_dir);
            throw $e;
        }
    }
    
    private function cleanupTempFiles($files, $dir) {
        // Clean up files
        foreach ($files as $file) {
            if (file_exists($file)) {
                unlink($file);
            }
        }
        
        // Clean up directory
        if (file_exists($dir . '/content_export.zip')) {
            unlink($dir . '/content_export.zip');
        }
        
        if (is_dir($dir)) {
            rmdir($dir);
        }
    }
    
    public function generateLinkGraphHTML($graph_data) {
        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Content Link Graph - Bulk AI Content Writer</title>
    <script src="https://d3js.org/d3.v7.min.js"></script>
    <style>
        body {
            font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 0;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        .graph-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }
        .graph-header {
            text-align: center;
            color: white;
            margin-bottom: 2rem;
        }
        .graph-header h1 {
            font-size: 2.5rem;
            margin: 0 0 1rem 0;
            font-weight: 700;
        }
        .graph-header p {
            font-size: 1.1rem;
            opacity: 0.9;
            margin: 0;
        }
        .graph-svg-container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.2);
            padding: 2rem;
            margin-bottom: 2rem;
        }
        .links line {
            stroke: #999;
            stroke-opacity: 0.6;
            stroke-width: 2px;
        }
        .nodes circle {
            stroke: #fff;
            stroke-width: 2px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .nodes circle:hover {
            stroke-width: 3px;
            filter: brightness(1.2);
        }
        .node-label {
            font-size: 12px;
            pointer-events: none;
            font-weight: 600;
        }
        .graph-stats {
            background: white;
            border-radius: 15px;
            padding: 1.5rem;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }
        .stat-item {
            text-align: center;
            padding: 1rem;
            background: #f8f9fa;
            border-radius: 10px;
        }
        .stat-value {
            font-size: 1.5rem;
            font-weight: 700;
            color: #3498db;
            display: block;
        }
        .stat-label {
            font-size: 0.9rem;
            color: #6c757d;
            margin-top: 0.5rem;
        }
        .legend {
            display: flex;
            justify-content: center;
            gap: 2rem;
            margin-top: 1rem;
            flex-wrap: wrap;
        }
        .legend-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.9rem;
            color: #333;
        }
        .legend-color {
            width: 16px;
            height: 16px;
            border-radius: 50%;
        }
    </style>
</head>
<body>
    <div class="graph-container">
        <div class="graph-header">
            <h1>Content Link Graph</h1>
            <p>Visual representation of internal linking between your articles</p>
        </div>
        
        <div class="graph-svg-container">
            <div id="graph-svg"></div>
        </div>
        
        <div class="graph-stats">
            <div class="stats-grid">
                <div class="stat-item">
                    <span class="stat-value" id="node-count">0</span>
                    <span class="stat-label">Total Articles</span>
                </div>
                <div class="stat-item">
                    <span class="stat-value" id="link-count">0</span>
                    <span class="stat-label">Internal Links</span>
                </div>
                <div class="stat-item">
                    <span class="stat-value" id="density">0</span>
                    <span class="stat-label">Link Density</span>
                </div>
            </div>
            
            <div class="legend">
                <div class="legend-item">
                    <div class="legend-color" style="background: #e74c3c;"></div>
                    <span>How-To Guides</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #3498db;"></div>
                    <span>Explanatory</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #2ecc71;"></div>
                    <span>Benefits/Why</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #f39c12;"></div>
                    <span>Comparisons</span>
                </div>
                <div class="legend-item">
                    <div class="legend-color" style="background: #9b59b6;"></div>
                    <span>Reviews</span>
                </div>
            </div>
        </div>
    </div>

    <script>
        const graph = ' . json_encode($graph_data) . ';
        
        // Update stats
        document.getElementById("node-count").textContent = graph.nodes.length;
        document.getElementById("link-count").textContent = graph.links.length;
        document.getElementById("density").textContent = (graph.links.length / graph.nodes.length).toFixed(1);
        
        const width = 1200, height = 800;
        const svg = d3.select("#graph-svg").append("svg")
            .attr("width", width)
            .attr("height", height);
            
        const simulation = d3.forceSimulation(graph.nodes)
            .force("link", d3.forceLink(graph.links).id(d => d.id).distance(100))
            .force("charge", d3.forceManyBody().strength(-300))
            .force("center", d3.forceCenter(width / 2, height / 2))
            .force("collision", d3.forceCollide().radius(50));
            
        const link = svg.append("g")
            .attr("class", "links")
            .selectAll("line")
            .data(graph.links)
            .enter().append("line")
            .attr("stroke-width", d => Math.sqrt(d.value));
            
        const node = svg.append("g")
            .attr("class", "nodes")
            .selectAll("circle")
            .data(graph.nodes)
            .enter().append("circle")
            .attr("r", d => d.value * 4)
            .attr("fill", d => {
                const colors = { 
                    "how-to": "#e74c3c", 
                    "what-is": "#3498db", 
                    "why": "#2ecc71", 
                    "comparison": "#f39c12", 
                    "review": "#9b59b6", 
                    "news": "#1abc9c", 
                    "general": "#95a5a6" 
                };
                return colors[d.group] || "#95a5a6";
            })
            .call(d3.drag()
                .on("start", dragstarted)
                .on("drag", dragged)
                .on("end", dragended));
                
        const label = svg.append("g")
            .attr("class", "node-label")
            .selectAll("text")
            .data(graph.nodes)
            .enter().append("text")
            .text(d => d.name)
            .attr("text-anchor", "middle")
            .attr("dy", -15);
            
        simulation.on("tick", () => {
            link.attr("x1", d => d.source.x)
                .attr("y1", d => d.source.y)
                .attr("x2", d => d.target.x)
                .attr("y2", d => d.target.y);
                
            node.attr("cx", d => d.x)
                .attr("cy", d => d.y);
                
            label.attr("x", d => d.x)
                 .attr("y", d => d.y);
        });
        
        function dragstarted(event, d) {
            if (!event.active) simulation.alphaTarget(0.3).restart();
            d.fx = d.x;
            d.fy = d.y;
        }
        
        function dragged(event, d) {
            d.fx = event.x;
            d.fy = event.y;
        }
        
        function dragended(event, d) {
            if (!event.active) simulation.alphaTarget(0);
            d.fx = null;
            d.fy = null;
        }
        
        // Add click interaction
        node.on("click", function(event, d) {
            alert("Article: " + d.name + "\\nGroup: " + d.group + "\\nConnections: " + 
                graph.links.filter(link => link.source.id === d.id || link.target.id === d.id).length);
        });
    </script>
</body>
</html>';
        
        return $html;
    }
    
    public function sanitizeFilename($filename) {
        $filename = preg_replace('/[^a-zA-Z0-9_-]/', '_', $filename);
        $filename = preg_replace('/_{2,}/', '_', $filename);
        $filename = trim($filename, '_');
        return $filename ?: 'untitled';
    }
}

// Handle export requests
if ($_GET['action'] ?? '' === 'export_article') {
    $export_manager = new ExportManager();
    $content = $_POST['content'] ?? '';
    $title = $_POST['title'] ?? 'Untitled';
    $format = $_GET['format'] ?? 'html';
    
    switch ($format) {
        case 'html':
            $output = $export_manager->exportArticleAsHTML($content, $title);
            header('Content-Type: text/html');
            break;
        case 'doc':
            $output = $export_manager->exportArticleAsDOC($content, $title);
            header('Content-Type: application/msword');
            break;
        case 'pdf':
            $output = $export_manager->exportArticleAsPDF($content, $title);
            header('Content-Type: application/pdf');
            break;
        default:
            $output = $content;
            header('Content-Type: text/plain');
    }
    
    header('Content-Disposition: attachment; filename="' . $export_manager->sanitizeFilename($title) . '.' . $format . '"');
    echo $output;
    exit;
}

if ($_GET['action'] ?? '' === 'export_all') {
    $project_id = $_GET['project_id'] ?? '';
    $format = $_GET['format'] ?? 'html';
    
    if (empty($project_id)) {
        http_response_code(400);
        echo "Project ID is required";
        exit;
    }
    
    $export_manager = new ExportManager();
    
    try {
        if ($_GET['type'] === 'zip') {
            // Export as ZIP
            $zip_content = $export_manager->exportAllArticlesAsZip($project_id, $format);
            
            if ($zip_content === false) {
                http_response_code(404);
                echo "No content found to export";
                exit;
            }
            
            header('Content-Type: application/zip');
            header('Content-Disposition: attachment; filename="content_export_' . date('Y-m-d') . '.zip"');
            header('Content-Length: ' . strlen($zip_content));
            echo $zip_content;
            
        } else {
            // Export single file
            $files = $export_manager->exportAllArticles($project_id, $format);
            
            if (empty($files)) {
                http_response_code(404);
                echo "No content found to export";
                exit;
            }
            
            if (count($files) === 1) {
                // Single file - export directly
                $filename = key($files);
                $content = current($files);
                
                switch ($format) {
                    case 'html':
                        header('Content-Type: text/html');
                        break;
                    case 'doc':
                        header('Content-Type: application/msword');
                        break;
                    case 'pdf':
                        header('Content-Type: application/pdf');
                        break;
                    default:
                        header('Content-Type: text/plain');
                }
                
                header('Content-Disposition: attachment; filename="' . $filename . '"');
                echo $content;
            } else {
                // Multiple files - create ZIP
                $zip_content = $export_manager->exportAllArticlesAsZip($project_id, $format);
                
                header('Content-Type: application/zip');
                header('Content-Disposition: attachment; filename="content_export_' . date('Y-m-d') . '.zip"');
                header('Content-Length: ' . strlen($zip_content));
                echo $zip_content;
            }
        }
    } catch (Exception $e) {
        http_response_code(500);
        echo "Export failed: " . $e->getMessage();
    }
    exit;
}

if ($_GET['action'] === 'export_report') {
    $project_id = $_GET['project_id'] ?? '';
    
    if (empty($project_id)) {
        http_response_code(400);
        echo "Project ID is required";
        exit;
    }
    
    $export_manager = new ExportManager();
    $db = new Database();
    
    // Verify project ownership
    session_start();
    if (!isset($_SESSION['user_id'])) {
        http_response_code(401);
        echo "Not authenticated";
        exit;
    }
    
    $content_items = $db->getProjectContent($project_id);
    
    if (empty($content_items)) {
        http_response_code(404);
        echo "No content found for this project";
        exit;
    }
    
    $report_html = $export_manager->generatePDFReport($project_id, $content_items);
    
    header('Content-Type: text/html');
    header('Content-Disposition: attachment; filename="content_report_' . date('Y-m-d') . '.html"');
    echo $report_html;
    exit;
}

if ($_GET['action'] === 'export_link_graph') {
    $project_id = $_GET['project_id'] ?? '';
    
    if (empty($project_id)) {
        http_response_code(400);
        echo "Project ID is required";
        exit;
    }
    
    $export_manager = new ExportManager();
    $linking_engine = new LinkingEngine(); // You'll need to include this class
    
    // Generate graph data (you'll need to implement this based on your linking engine)
    $graph_data = []; // Placeholder - implement based on your linking engine
    
    $graph_html = $export_manager->generateLinkGraphHTML($graph_data);
    
    header('Content-Type: text/html');
    header('Content-Disposition: attachment; filename="link_graph_' . date('Y-m-d') . '.html"');
    echo $graph_html;
    exit;
}
?>