<?php
class LinkingEngine {
    private $topics = [];
    private $generated_content = [];
    private $min_links_per_article;
    
    public function __construct($min_links = 5) {
        $this->min_links_per_article = $min_links;
    }
    
    public function setTopics($topics) {
        $this->topics = $topics;
    }
    
    public function setGeneratedContent($content) {
        $this->generated_content = $content;
    }
    
    public function analyzeTopics() {
        $analysis = [];
        
        foreach ($this->topics as $topic) {
            $analysis[$topic] = [
                'keywords' => $this->extractKeywords($topic),
                'semantic_group' => $this->categorizeTopic($topic),
                'word_count' => str_word_count($topic),
                'complexity' => $this->assessComplexity($topic)
            ];
        }
        
        return $analysis;
    }
    
    public function generateInternalLinks($content, $current_topic) {
        if (count($this->generated_content) < 2) {
            return $content; // Not enough content for linking
        }
        
        $topic_analysis = $this->analyzeTopics();
        $current_analysis = $topic_analysis[$current_topic];
        
        // Find best matching topics for internal links
        $potential_links = $this->findRelevantLinks($current_topic, $current_analysis, $topic_analysis);
        
        if (count($potential_links) < $this->min_links_per_article) {
            // Add more links from broader categories
            $potential_links = array_merge(
                $potential_links,
                $this->findSupplementaryLinks($current_topic, $topic_analysis)
            );
        }
        
        // Insert links into content
        $linked_content = $this->insertLinks($content, $potential_links, $current_topic);
        
        return $linked_content;
    }
    
    private function extractKeywords($text) {
        $stopwords = ['the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by'];
        $words = str_word_count(strtolower($text), 1);
        $keywords = array_diff($words, $stopwords);
        
        return array_values($keywords);
    }
    
    private function categorizeTopic($topic) {
        $categories = [
            'how-to' => ['how to', 'guide', 'tutorial', 'step by step'],
            'what-is' => ['what is', 'definition', 'explained'],
            'why' => ['why', 'benefits', 'advantages'],
            'comparison' => ['vs', 'versus', 'comparison', 'difference'],
            'review' => ['review', 'best', 'top'],
            'news' => ['2024', 'update', 'latest', 'new']
        ];
        
        $topic_lower = strtolower($topic);
        
        foreach ($categories as $category => $indicators) {
            foreach ($indicators as $indicator) {
                if (strpos($topic_lower, $indicator) !== false) {
                    return $category;
                }
            }
        }
        
        return 'general';
    }
    
    private function assessComplexity($topic) {
        $word_count = str_word_count($topic);
        $has_technical = preg_match('/\b(api|seo|algorithm|framework|protocol)\b/i', $topic);
        
        if ($word_count > 4 || $has_technical) {
            return 'advanced';
        } elseif ($word_count > 2) {
            return 'intermediate';
        } else {
            return 'basic';
        }
    }
    
    private function findRelevantLinks($current_topic, $current_analysis, $all_analysis) {
        $links = [];
        $current_keywords = $current_analysis['keywords'];
        
        foreach ($all_analysis as $topic => $analysis) {
            if ($topic === $current_topic) continue;
            
            $score = 0;
            $matching_keywords = array_intersect($current_keywords, $analysis['keywords']);
            
            // Score based on keyword matches
            $score += count($matching_keywords) * 3;
            
            // Score based on semantic group
            if ($analysis['semantic_group'] === $current_analysis['semantic_group']) {
                $score += 2;
            }
            
            // Score based on complexity match
            if ($analysis['complexity'] === $current_analysis['complexity']) {
                $score += 1;
            }
            
            if ($score > 0) {
                $links[$topic] = $score;
            }
        }
        
        // Sort by score and return top links
        arsort($links);
        return array_slice($links, 0, $this->min_links_per_article + 2, true);
    }
    
    private function findSupplementaryLinks($current_topic, $all_analysis) {
        $links = [];
        $current_category = $all_analysis[$current_topic]['semantic_group'];
        
        foreach ($all_analysis as $topic => $analysis) {
            if ($topic === $current_topic) continue;
            
            if ($analysis['semantic_group'] === $current_category) {
                $links[$topic] = 1; // Basic score for same category
            }
        }
        
        return $links;
    }
    
    private function insertLinks($content, $potential_links, $current_topic) {
        $sentences = $this->splitIntoSentences($content);
        $used_links = [];
        $link_count = 0;
        
        foreach ($sentences as &$sentence) {
            if ($link_count >= $this->min_links_per_article) break;
            
            foreach ($potential_links as $link_topic => $score) {
                if (in_array($link_topic, $used_links)) continue;
                
                $keywords = $this->extractKeywords($link_topic);
                foreach ($keywords as $keyword) {
                    if (strpos(strtolower($sentence), strtolower($keyword)) !== false) {
                        // Found a good place to insert link
                        $linked_sentence = $this->createLink($sentence, $keyword, $link_topic);
                        if ($linked_sentence !== $sentence) {
                            $sentence = $linked_sentence;
                            $used_links[] = $link_topic;
                            $link_count++;
                            break 2; // Break both loops
                        }
                    }
                }
            }
        }
        
        // If we still need more links, add them strategically
        if ($link_count < $this->min_links_per_article) {
            $remaining_links = array_diff(array_keys($potential_links), $used_links);
            $sentences = $this->addStrategicLinks($sentences, $remaining_links, $link_count);
        }
        
        return implode(' ', $sentences);
    }
    
    private function splitIntoSentences($content) {
        // Simple sentence splitting - in production, use more robust method
        return preg_split('/(?<=[.?!])\s+/', $content, -1, PREG_SPLIT_NO_EMPTY);
    }
    
    private function createLink($sentence, $keyword, $link_topic) {
        $pattern = '/\b' . preg_quote($keyword, '/') . '\b/i';
        $replacement = '<a href="#' . urlencode($link_topic) . '" class="internal-link">' . $keyword . '</a>';
        
        return preg_replace($pattern, $replacement, $sentence, 1);
    }
    
    private function addStrategicLinks($sentences, $remaining_links, &$link_count) {
        $min_links = $this->min_links_per_article;
        
        foreach ($remaining_links as $link_topic) {
            if ($link_count >= $min_links) break;
            
            // Find a suitable sentence to add contextual link
            foreach ($sentences as &$sentence) {
                if (strlen($sentence) > 50 && !preg_match('/<a /', $sentence)) {
                    $keywords = $this->extractKeywords($link_topic);
                    $primary_keyword = $keywords[0] ?? '';
                    
                    if ($primary_keyword) {
                        $sentence .= " This relates to our discussion of <a href=\"#" . urlencode($link_topic) . "\" class=\"internal-link\">" . $link_topic . "</a>.";
                        $link_count++;
                        break;
                    }
                }
            }
        }
        
        return $sentences;
    }
    
    public function generateLinkGraph() {
        $graph = [
            'nodes' => [],
            'links' => []
        ];
        
        foreach ($this->topics as $topic) {
            $graph['nodes'][] = [
                'id' => urlencode($topic),
                'name' => $topic,
                'group' => $this->categorizeTopic($topic),
                'value' => $this->assessComplexity($topic) === 'advanced' ? 3 : 
                          ($this->assessComplexity($topic) === 'intermediate' ? 2 : 1)
            ];
        }
        
        // Generate links between nodes based on relevance
        $topic_analysis = $this->analyzeTopics();
        
        foreach ($this->topics as $source_topic) {
            $source_analysis = $topic_analysis[$source_topic];
            $relevant_links = $this->findRelevantLinks($source_topic, $source_analysis, $topic_analysis);
            
            foreach ($relevant_links as $target_topic => $score) {
                if ($score > 1) { // Only include meaningful links
                    $graph['links'][] = [
                        'source' => urlencode($source_topic),
                        'target' => urlencode($target_topic),
                        'value' => min($score, 5)
                    ];
                }
            }
        }
        
        return $graph;
    }
}
?>