<?php
session_start();
require_once 'database.php';

// Check authentication
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$db = new Database();
$user_id = $_SESSION['user_id'];

// Handle project creation via POST
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create_project') {
    $project_name = $_POST['project_name'] ?? '';
    $project_description = $_POST['project_description'] ?? '';
    
    if (empty(trim($project_name))) {
        $_SESSION['error'] = "Project name is required";
    } else {
        $project_id = $db->createProject($user_id, trim($project_name), trim($project_description));
        if ($project_id) {
            header('Location: content_generator.php?project_id=' . $project_id);
            exit;
        } else {
            $_SESSION['error'] = "Failed to create project";
        }
    }
}

// Handle project deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete_project') {
    $project_id = $_POST['project_id'] ?? '';
    if ($project_id) {
        $db->deleteProject($user_id, $project_id);
        $_SESSION['message'] = 'project_deleted';
        header('Location: project_manager.php');
        exit;
    }
}

// Get user's data
$projects = $db->getUserProjects($user_id);
$api_keys = $db->getUserApiKeys($user_id);
$usage_stats = $db->getUserUsageStats($user_id);

// Check for messages
$message = $_SESSION['message'] ?? '';
$error = $_SESSION['error'] ?? '';

// Clear messages
unset($_SESSION['message'], $_SESSION['error']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Project Manager - Bulk AI Content Writer</title>
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.2.96/css/materialdesignicons.min.css" rel="stylesheet">
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <?php include 'ui_components.php'; echo createHeader('project'); ?>
    
    <div class="container">
        <?php if ($message === 'project_deleted'): ?>
            <div class="alert alert-success">
                <i class="mdi mdi-check-circle"></i>
                <span>Project deleted successfully</span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            </div>
        <?php endif; ?>
        
        <?php if (!empty($error)): ?>
            <div class="alert alert-error">
                <i class="mdi mdi-alert-circle"></i>
                <span><?php echo htmlspecialchars($error); ?></span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            </div>
        <?php endif; ?>
        
        <div class="dashboard-grid">
            <!-- Quick Stats -->
            <div class="stats-container">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-folder-multiple"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo count($projects); ?></h3>
                        <p>Total Projects</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-key-chain"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo count($api_keys); ?></h3>
                        <p>API Keys</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-file-document-multiple"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo $usage_stats['total_articles'] ?? 0; ?></h3>
                        <p>Articles Generated</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="mdi mdi-counter"></i>
                    </div>
                    <div class="stat-info">
                        <h3><?php echo number_format($usage_stats['total_words'] ?? 0); ?></h3>
                        <p>Total Words</p>
                    </div>
                </div>
            </div>

            <!-- API Key Management -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="mdi mdi-key-chain"></i> API Key Management</h2>
                </div>
                <div class="card-body">
                    <form id="api-key-form" class="api-key-form">
                        <div class="form-group">
                            <label>Gemini 2.0 Flash API Key</label>
                            <input type="text" name="api_key" required placeholder="Enter your Gemini API key">
                            <small>Add multiple keys for failover and increased limits. Your keys are stored securely.</small>
                        </div>
                        <button type="submit" class="btn-primary">
                            <i class="mdi mdi-plus"></i> Add API Key
                        </button>
                    </form>
                    
                    <?php if (!empty($api_keys)): ?>
                    <div class="api-keys-list">
                        <h4>Your API Keys</h4>
                        <?php foreach ($api_keys as $key): ?>
                        <div class="api-key-item">
                            <code>••••<?php echo substr($key['api_key'], -4); ?></code>
                            <span class="key-status <?php echo $key['is_active'] ? 'active' : 'inactive'; ?>">
                                <?php echo $key['is_active'] ? 'Active' : 'Inactive'; ?>
                            </span>
                            <span class="usage-count">Used: <?php echo $key['usage_count']; ?> times</span>
                            <button type="button" class="btn-danger btn-sm delete-api-key" data-key-id="<?php echo $key['id']; ?>">
                                <i class="mdi mdi-delete"></i>
                            </button>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php else: ?>
                    <div class="empty-state">
                        <i class="mdi mdi-key-remove"></i>
                        <h3>No API Keys Added</h3>
                        <p>Add your first Gemini API key to start generating content</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Project Creation -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="mdi mdi-plus-circle"></i> Create New Project</h2>
                </div>
                <div class="card-body">
                    <form method="POST" class="project-form">
                        <input type="hidden" name="action" value="create_project">
                        <div class="form-group">
                            <label>Project Name</label>
                            <input type="text" name="project_name" required placeholder="e.g., My SEO Blog Content">
                        </div>
                        <div class="form-group">
                            <label>Description (Optional)</label>
                            <textarea name="project_description" placeholder="Brief description of your project..."></textarea>
                        </div>
                        <button type="submit" class="btn-primary">
                            <i class="mdi mdi-creation"></i> Create Project
                        </button>
                    </form>
                </div>
            </div>

            <!-- Existing Projects -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="mdi mdi-folder-multiple"></i> Your Projects</h2>
                </div>
                <div class="card-body">
                    <?php if (empty($projects)): ?>
                        <div class="empty-state">
                            <i class="mdi mdi-folder-open-outline"></i>
                            <h3>No Projects Yet</h3>
                            <p>Create your first project to start generating content</p>
                        </div>
                    <?php else: ?>
                        <div class="projects-grid">
                            <?php foreach ($projects as $project): ?>
                            <div class="project-card">
                                <div class="project-info">
                                    <h3><?php echo htmlspecialchars($project['name']); ?></h3>
                                    <p><?php echo htmlspecialchars($project['description']); ?></p>
                                    <div class="project-meta">
                                        <span><i class="mdi mdi-file-document"></i> <?php echo $project['article_count']; ?> articles</span>
                                        <span><i class="mdi mdi-calendar"></i> <?php echo date('M j, Y', strtotime($project['created_at'])); ?></span>
                                    </div>
                                </div>
                                <div class="project-actions">
                                    <a href="content_generator.php?project_id=<?php echo $project['id']; ?>" class="btn-primary">
                                        <i class="mdi mdi-pencil"></i> Open
                                    </a>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="delete_project">
                                        <input type="hidden" name="project_id" value="<?php echo $project['id']; ?>">
                                        <button type="submit" class="btn-danger" onclick="return confirm('Are you sure you want to delete this project? All generated content will be lost.')">
                                            <i class="mdi mdi-delete"></i>
                                        </button>
                                    </form>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <script>
        // API Key Management
        document.getElementById('api-key-form').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const apiKey = this.querySelector('input[name="api_key"]').value.trim();
            
            if (!apiKey) {
                showNotification('Please enter an API key', 'error');
                return;
            }
            
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            // Show loading state
            submitBtn.innerHTML = '<div class="loading-spinner small"><div class="spinner"></div></div> Adding...';
            submitBtn.disabled = true;
            
            try {
                const response = await fetch('api.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'add_api_key',
                        api_key: apiKey
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showNotification('API key added successfully!', 'success');
                    this.reset();
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(result.error || 'Failed to add API key', 'error');
                }
            } catch (error) {
                showNotification('Network error. Please try again.', 'error');
                console.error('API Key Error:', error);
            } finally {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }
        });
        
        // Delete API Key
        document.querySelectorAll('.delete-api-key').forEach(btn => {
            btn.addEventListener('click', async function() {
                const keyId = this.getAttribute('data-key-id');
                if (!confirm('Are you sure you want to delete this API key?')) {
                    return;
                }
                
                try {
                    const response = await fetch('api.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            action: 'delete_api_key',
                            key_id: keyId
                        })
                    });
                    
                    const result = await response.json();
                    
                    if (result.success) {
                        showNotification('API key deleted successfully!', 'success');
                        setTimeout(() => {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotification(result.error || 'Failed to delete API key', 'error');
                    }
                } catch (error) {
                    showNotification('Network error. Please try again.', 'error');
                    console.error('Delete API Key Error:', error);
                }
            });
        });
        
        // Notification function
        function showNotification(message, type = 'info') {
            // Remove existing notifications
            document.querySelectorAll('.alert').forEach(alert => alert.remove());
            
            const notification = document.createElement('div');
            notification.className = `alert alert-${type}`;
            notification.innerHTML = `
                <i class="mdi mdi-${getNotificationIcon(type)}"></i>
                <span>${message}</span>
                <button class="alert-close"><i class="mdi mdi-close"></i></button>
            `;
            
            const container = document.querySelector('.container');
            container.insertBefore(notification, container.firstChild);
            
            // Auto-remove after 5 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 5000);
            
            // Close button functionality
            notification.querySelector('.alert-close').addEventListener('click', () => {
                notification.parentNode.removeChild(notification);
            });
        }
        
        function getNotificationIcon(type) {
            const icons = {
                info: 'information',
                success: 'check-circle',
                warning: 'alert',
                error: 'alert-circle'
            };
            return icons[type] || 'information';
        }
        
        // Close alert buttons
        document.querySelectorAll('.alert-close').forEach(btn => {
            btn.addEventListener('click', function() {
                this.closest('.alert').remove();
            });
        });
    </script>
</body>
</html>