// Enhanced Theme Manager
class ThemeManager {
    constructor() {
        this.currentTheme = localStorage.getItem('theme') || 'light';
        this.init();
    }

    init() {
        this.applyTheme(this.currentTheme);
        this.setupEventListeners();
    }

    applyTheme(theme) {
        document.documentElement.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
        this.updateThemeButton(theme);
    }

    toggleTheme() {
        this.currentTheme = this.currentTheme === 'light' ? 'dark' : 'light';
        this.applyTheme(this.currentTheme);
        
        // Show notification
        this.showThemeNotification(this.currentTheme);
    }

    updateThemeButton(theme) {
        const themeBtn = document.getElementById('theme-toggle');
        if (themeBtn) {
            const icon = themeBtn.querySelector('i');
            if (icon) {
                icon.className = theme === 'light' ? 'mdi mdi-weather-sunny' : 'mdi mdi-weather-night';
            }
        }
    }

    showThemeNotification(theme) {
        const notification = document.createElement('div');
        notification.className = `alert alert-success`;
        notification.innerHTML = `
            <i class="mdi mdi-${theme === 'light' ? 'weather-sunny' : 'weather-night'}"></i>
            <span>Switched to ${theme} theme</span>
            <button class="alert-close"><i class="mdi mdi-close"></i></button>
        `;
        
        const container = document.querySelector('.container, .generator-container');
        if (container) {
            container.insertBefore(notification, container.firstChild);
            
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 3000);
            
            notification.querySelector('.alert-close').addEventListener('click', () => {
                notification.parentNode.removeChild(notification);
            });
        }
    }

    setupEventListeners() {
        const themeBtn = document.getElementById('theme-toggle');
        if (themeBtn) {
            themeBtn.addEventListener('click', () => this.toggleTheme());
        }
    }
}

// Enhanced API Manager with Real Gemini Integration
class APIManager {
    constructor() {
        this.baseURL = window.location.origin;
        this.isGenerating = false;
        this.currentGenerations = new Map();
        this.activePauses = new Set();
        this.apiKeys = [];
        this.loadApiKeys();
    }

    async loadApiKeys() {
        try {
            // Get API keys from the page (they're rendered in the HTML)
            const apiKeyElements = document.querySelectorAll('.api-key-item code');
            this.apiKeys = Array.from(apiKeyElements).map(element => {
                // Extract the actual API key from the hidden storage
                // In a real implementation, you'd get this from a secure endpoint
                const keyId = element.closest('.api-key-item').querySelector('.delete-api-key').getAttribute('data-key-id');
                // For now, we'll use a different approach - get from session
                return this.getApiKeyFromStorage(keyId);
            }).filter(key => key);
            
            console.log('Loaded API keys:', this.apiKeys.length);
        } catch (error) {
            console.error('Error loading API keys:', error);
            this.apiKeys = [];
        }
    }

    getApiKeyFromStorage(keyId) {
        // This is a simplified version. In production, you'd get this from a secure API endpoint
        // For now, we'll rely on the user having added keys through the interface
        try {
            const storedKeys = JSON.parse(localStorage.getItem('gemini_api_keys') || '{}');
            return storedKeys[keyId];
        } catch (error) {
            return null;
        }
    }

    async generateWithGemini(topic, config) {
        return new Promise(async (resolve, reject) => {
            try {
                // Get active API keys
                const apiKeys = await this.getActiveApiKeys();
                console.log('Available API keys:', apiKeys.length);
                
                if (apiKeys.length === 0) {
                    reject('No active API keys found. Please add your Gemini API keys in the Project Manager.');
                    return;
                }

                // Try each API key until one works
                for (let i = 0; i < apiKeys.length; i++) {
                    const apiKey = apiKeys[i];
                    console.log(`Trying API key ${i + 1}/${apiKeys.length}`);
                    
                    try {
                        const content = await this.tryGenerateWithKey(apiKey, topic, config);
                        console.log(`Successfully generated content with API key ${i + 1}`);
                        resolve(content);
                        return;
                    } catch (error) {
                        console.warn(`API key ${i + 1} failed: ${error.message}`);
                        if (i === apiKeys.length - 1) {
                            // This was the last key, throw the error
                            throw error;
                        }
                        // Continue to next key
                        continue;
                    }
                }
                
            } catch (error) {
                reject(error.message || 'Generation failed');
            }
        });
    }

    async getActiveApiKeys() {
        try {
            // Try to get API keys from the server
            const response = await fetch('api.php?action=get_api_keys');
            if (response.ok) {
                const result = await response.json();
                if (result.success && result.keys && result.keys.length > 0) {
                    return result.keys;
                }
            }
            
            // Fallback: try to get from localStorage (for demo purposes)
            try {
                const storedKeys = JSON.parse(localStorage.getItem('gemini_api_keys') || '{}');
                const keys = Object.values(storedKeys).filter(key => key && key.trim() !== '');
                if (keys.length > 0) {
                    return keys;
                }
            } catch (e) {
                console.warn('Could not load API keys from localStorage:', e);
            }
            
            return [];
        } catch (error) {
            console.error('Failed to get API keys:', error);
            return [];
        }
    }

    async tryGenerateWithKey(apiKey, topic, config) {
        if (!apiKey || apiKey.trim() === '') {
            throw new Error('Invalid API key');
        }

        const prompt = this.buildAdvancedPrompt(topic, config);
        console.log('Sending request to Gemini API...');
        
        const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                contents: [{
                    parts: [{
                        text: prompt
                    }]
                }],
                generationConfig: {
                    temperature: 0.7,
                    topK: 40,
                    topP: 0.95,
                    maxOutputTokens: config.word_count * 4,
                    stopSequences: []
                },
                safetySettings: config.uncensored_mode ? [
                    { category: 'HARM_CATEGORY_HARASSMENT', threshold: 'BLOCK_NONE' },
                    { category: 'HARM_CATEGORY_HATE_SPEECH', threshold: 'BLOCK_NONE' },
                    { category: 'HARM_CATEGORY_SEXUALLY_EXPLICIT', threshold: 'BLOCK_NONE' },
                    { category: 'HARM_CATEGORY_DANGEROUS_CONTENT', threshold: 'BLOCK_NONE' }
                ] : []
            })
        });

        if (!response.ok) {
            const errorText = await response.text();
            let errorMessage = `API request failed: ${response.status}`;
            
            try {
                const errorData = JSON.parse(errorText);
                errorMessage = errorData.error?.message || errorMessage;
            } catch (e) {
                errorMessage = errorText || errorMessage;
            }
            
            throw new Error(errorMessage);
        }

        const data = await response.json();
        
        if (!data.candidates?.[0]?.content?.parts?.[0]?.text) {
            throw new Error('No content generated by AI. The response was empty.');
        }

        return data.candidates[0].content.parts[0].text;
    }

    buildAdvancedPrompt(topic, config) {
        let prompt = `You are ${config.ai_persona}, an expert content creator with deep industry knowledge. ${config.ai_prompt || ''}

TOPIC: ${topic}
CONTENT TYPE: ${config.content_type}
TONE: ${config.tone_voice}
TARGET WORD COUNT: ${config.word_count}

CORE REQUIREMENTS:
1. Include 3+ specific, real-world anecdotes from your professional experience
2. Provide 2+ contrarian opinions that challenge conventional wisdom
3. Share actionable, non-obvious strategies with precise instructions
4. Specify exactly who this strategy works for and who it doesn't
5. Use a ${config.tone_voice} tone throughout

SEO REQUIREMENTS:${config.focus_keyword ? `\n- Primary Keyword: ${config.focus_keyword}` : ''}${config.secondary_keywords ? `\n- Secondary Keywords: ${config.secondary_keywords}` : ''}

ADDITIONAL INSTRUCTIONS:${config.extra_prompt ? `\n${config.extra_prompt}` : '\nNone provided'}

OUTPUT FORMAT:
1. Start with a compelling meta description (under 155 characters)
2. Write the full article with proper HTML formatting (h1, h2, h3, p, strong, em, ul/ol)
3. End with a thought-provoking conclusion and question

IMPORTANT: This content must pass AI detection by being deeply human, opinionated, and based on real experience. Avoid generic advice and focus on unique insights.

Now, create the ${config.content_type} about: ${topic}`;

        return prompt;
    }

    pauseGeneration(topic) {
        this.activePauses.add(topic);
    }

    resumeGeneration(topic) {
        this.activePauses.delete(topic);
    }

    isPaused(topic) {
        return this.activePauses.has(topic);
    }

    showError(message) {
        this.showNotification(message, 'error');
    }

    showSuccess(message) {
        this.showNotification(message, 'success');
    }

    showInfo(message) {
        this.showNotification(message, 'info');
    }

    showNotification(message, type = 'info') {
        // Remove existing notifications of the same type
        document.querySelectorAll(`.alert-${type}`).forEach(alert => alert.remove());
        
        const notification = document.createElement('div');
        notification.className = `alert alert-${type}`;
        notification.innerHTML = `
            <i class="mdi mdi-${this.getNotificationIcon(type)}"></i>
            <span>${message}</span>
            <button class="alert-close"><i class="mdi mdi-close"></i></button>
        `;

        const container = document.querySelector('.container, .generator-container');
        if (container) {
            container.insertBefore(notification, container.firstChild);

            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 5000);

            notification.querySelector('.alert-close').addEventListener('click', () => {
                notification.parentNode.removeChild(notification);
            });
        }
    }

    getNotificationIcon(type) {
        const icons = {
            info: 'information',
            success: 'check-circle',
            warning: 'alert',
            error: 'alert-circle'
        };
        return icons[type] || 'information';
    }
}

// Enhanced Content Generation Manager
class ContentGenerationManager {
    constructor() {
        this.apiManager = new APIManager();
        this.isBulkGenerating = false;
        this.generationQueue = [];
        this.currentProgress = 0;
        this.generationStats = {
            completed: 0,
            success: 0,
            failed: 0,
            total: 0
        };
    }

    async generateAllContent(config) {
        if (this.isBulkGenerating) {
            this.apiManager.showError('Bulk generation already in progress');
            return;
        }

        const topics = config.topics;
        if (!topics || topics.length === 0) {
            this.apiManager.showError('No topics provided');
            return;
        }

        this.isBulkGenerating = true;
        this.generationQueue = [...topics];
        this.currentProgress = 0;
        
        this.generationStats = {
            completed: 0,
            success: 0,
            failed: 0,
            total: topics.length
        };

        this.showProgressContainer();
        this.updateProgress(0, 'Starting bulk generation...');
        this.updateStats();

        // Create article containers for each topic
        this.createArticleContainers(topics);

        try {
            for (let i = 0; i < topics.length; i++) {
                const topic = topics[i];
                
                // Check for pause
                while (this.apiManager.isPaused(topic)) {
                    this.updateProgress((i / topics.length) * 100, `Paused: ${topic}`);
                    await this.delay(1000);
                    continue;
                }

                this.updateProgress((i / topics.length) * 100, `Generating: ${topic}`);
                this.generationStats.completed = i;

                try {
                    const content = await this.apiManager.generateWithGemini(topic, config);
                    
                    // Apply internal linking if enabled
                    let finalContent = content;
                    if (config.enable_linking) {
                        finalContent = this.applyInternalLinks(content, topic, topics);
                    }

                    // Save to database
                    await this.saveGeneratedContent(config.project_id, topic, finalContent, config);

                    this.markArticleComplete(topic, finalContent);
                    this.generationStats.success++;
                    this.apiManager.showInfo(`Generated: ${topic}`);
                    
                } catch (error) {
                    console.error(`Failed to generate content for "${topic}":`, error);
                    this.markArticleError(topic, error);
                    this.generationStats.failed++;
                    this.apiManager.showError(`Failed: ${topic} - ${error.message}`);
                }

                this.updateStats();
                this.currentProgress = ((i + 1) / topics.length) * 100;
                
                // Small delay between generations to avoid rate limiting
                await this.delay(1000);
            }

            this.updateProgress(100, 'Bulk generation complete!');
            this.apiManager.showSuccess(`Successfully generated ${this.generationStats.success} out of ${topics.length} articles`);
            this.enableExportButton();

        } catch (error) {
            this.apiManager.showError('Bulk generation failed: ' + error.message);
        } finally {
            this.isBulkGenerating = false;
            this.generationQueue = [];
        }
    }

    async regenerateArticle(articleId, topic, config) {
        try {
            const articleElement = document.querySelector(`[data-article-id="${articleId}"]`);
            if (articleElement) {
                this.showArticleGenerating(articleElement);
            }

            this.apiManager.showInfo(`Regenerating: ${topic}`);
            const content = await this.apiManager.generateWithGemini(topic, config);
            
            // Apply internal linking if enabled
            let finalContent = content;
            if (config.enable_linking) {
                finalContent = this.applyInternalLinks(content, topic, [topic]);
            }

            // Update in database
            await this.updateGeneratedContent(articleId, finalContent, config);

            if (articleElement) {
                this.markArticleComplete(topic, finalContent, articleElement);
            }

            this.apiManager.showSuccess('Article regenerated successfully!');
        } catch (error) {
            this.apiManager.showError('Failed to regenerate article: ' + error.message);
        }
    }

    showArticleGenerating(articleElement) {
        const contentElement = articleElement.querySelector('.enhanced-article-content');
        const actionsElement = articleElement.querySelector('.enhanced-article-actions');
        
        if (contentElement) {
            contentElement.innerHTML = `
                <div class="loading-spinner medium">
                    <div class="spinner"></div>
                </div>
                <p style="text-align: center; margin-top: 1rem; color: var(--text-secondary);">Regenerating article...</p>
            `;
        }
        
        if (actionsElement) {
            const buttons = actionsElement.querySelectorAll('button');
            buttons.forEach(btn => btn.disabled = true);
            
            // Show pause button during regeneration
            const pauseBtn = actionsElement.querySelector('.pause-article');
            if (pauseBtn) {
                pauseBtn.style.display = 'inline-flex';
                pauseBtn.disabled = false;
            }
        }
    }

    togglePauseGeneration(topic, button) {
        if (this.apiManager.isPaused(topic)) {
            this.apiManager.resumeGeneration(topic);
            button.innerHTML = '<i class="mdi mdi-pause"></i> Pause';
            button.title = 'Pause Generation';
            this.apiManager.showInfo(`Resumed generation for: ${topic}`);
        } else {
            this.apiManager.pauseGeneration(topic);
            button.innerHTML = '<i class="mdi mdi-play"></i> Resume';
            button.title = 'Resume Generation';
            this.apiManager.showInfo(`Paused generation for: ${topic}`);
        }
    }

    createArticleContainers(topics) {
        const outputContainer = document.getElementById('content-output');
        outputContainer.innerHTML = '';

        topics.forEach(topic => {
            const articleElement = document.createElement('div');
            articleElement.className = 'enhanced-article fade-in';
            articleElement.setAttribute('data-topic', topic);
            articleElement.innerHTML = `
                <div class="enhanced-article-header">
                    <h3 class="article-title">${this.escapeHtml(topic)}</h3>
                    <div class="article-meta-enhanced">
                        <span><i class="mdi mdi-counter"></i> 0 words</span>
                        <span><i class="mdi mdi-progress-clock pulse-loading"></i> Generating...</span>
                        <span class="badge badge-primary">In Progress</span>
                    </div>
                </div>
                <div class="enhanced-article-content">
                    <div class="loading-spinner medium">
                        <div class="spinner"></div>
                    </div>
                    <p style="text-align: center; margin-top: 1rem; color: var(--text-secondary);">Generating content using AI...</p>
                </div>
                <div class="enhanced-article-actions">
                    <button class="article-action-btn action-primary copy-article" disabled>
                        <i class="mdi mdi-content-copy"></i> Copy
                    </button>
                    <button class="article-action-btn action-secondary export-article" disabled>
                        <i class="mdi mdi-export"></i> Export
                    </button>
                    <button class="article-action-btn action-warning regenerate-article" disabled>
                        <i class="mdi mdi-refresh"></i> Regenerate
                    </button>
                    <button class="article-action-btn action-secondary pause-article">
                        <i class="mdi mdi-pause"></i> Pause
                    </button>
                </div>
            `;
            outputContainer.appendChild(articleElement);

            // Add event listeners for controls
            this.setupArticleControls(articleElement);
        });
    }

    setupArticleControls(articleElement) {
        const pauseBtn = articleElement.querySelector('.pause-article');
        const topic = articleElement.getAttribute('data-topic');

        // Pause button functionality
        pauseBtn.addEventListener('click', () => {
            this.togglePauseGeneration(topic, pauseBtn);
        });
    }

    markArticleComplete(topic, content, articleElement = null) {
        if (!articleElement) {
            articleElement = document.querySelector(`[data-topic="${CSS.escape(topic)}"]`);
        }
        
        if (articleElement) {
            const contentElement = articleElement.querySelector('.enhanced-article-content');
            const metaElement = articleElement.querySelector('.article-meta-enhanced');
            const copyBtn = articleElement.querySelector('.copy-article');
            const exportBtn = articleElement.querySelector('.export-article');
            const regenerateBtn = articleElement.querySelector('.regenerate-article');
            const pauseBtn = articleElement.querySelector('.pause-article');

            if (contentElement) {
                contentElement.innerHTML = content;
            }

            if (metaElement) {
                const wordCount = content.split(/\s+/).length;
                metaElement.innerHTML = `
                    <span><i class="mdi mdi-counter"></i> ${wordCount} words</span>
                    <span><i class="mdi mdi-check-circle"></i> Complete</span>
                    <span><i class="mdi mdi-calendar"></i> ${new Date().toLocaleTimeString()}</span>
                    <span class="badge badge-success">Generated</span>
                `;
            }

            // Enable buttons
            if (copyBtn) {
                copyBtn.disabled = false;
                copyBtn.setAttribute('data-content', content);
                
                // Add click event
                copyBtn.addEventListener('click', () => {
                    navigator.clipboard.writeText(content).then(() => {
                        this.apiManager.showSuccess('Article copied to clipboard!');
                    }).catch(() => {
                        this.apiManager.showError('Failed to copy article');
                    });
                });
            }
            
            if (exportBtn) {
                exportBtn.disabled = false;
                exportBtn.setAttribute('data-content', content);
                exportBtn.setAttribute('data-title', topic);
                
                // Add click event
                exportBtn.addEventListener('click', () => {
                    this.exportSingleArticle(content, topic);
                });
            }
            
            if (regenerateBtn) {
                regenerateBtn.disabled = false;
                regenerateBtn.setAttribute('data-topic', topic);
            }
            
            if (pauseBtn) {
                pauseBtn.style.display = 'none';
            }
        }
    }

    markArticleError(topic, error) {
        const articleElement = document.querySelector(`[data-topic="${CSS.escape(topic)}"]`);
        if (articleElement) {
            const contentElement = articleElement.querySelector('.enhanced-article-content');
            const metaElement = articleElement.querySelector('.article-meta-enhanced');
            const pauseBtn = articleElement.querySelector('.pause-article');

            if (contentElement) {
                contentElement.innerHTML = `
                    <div class="alert alert-error">
                        <i class="mdi mdi-alert-circle"></i>
                        <span>Failed to generate content: ${error.message || error}</span>
                    </div>
                `;
            }

            if (metaElement) {
                metaElement.innerHTML = `
                    <span><i class="mdi mdi-alert"></i> Generation Failed</span>
                    <span class="badge badge-warning">Error</span>
                `;
            }

            if (pauseBtn) {
                pauseBtn.style.display = 'none';
            }
        }
    }

    showProgressContainer() {
        const progressContainer = document.getElementById('generation-progress');
        if (progressContainer) {
            progressContainer.style.display = 'block';
        }
    }

    updateProgress(percentage, text) {
        const progressFill = document.getElementById('progress-fill');
        const progressText = document.getElementById('progress-text');

        if (progressFill) progressFill.style.width = percentage + '%';
        if (progressText) progressText.textContent = text;
    }

    updateStats() {
        const completedEl = document.getElementById('stat-completed');
        const remainingEl = document.getElementById('stat-remaining');
        const successEl = document.getElementById('stat-success');
        const failedEl = document.getElementById('stat-failed');
        
        if (completedEl) completedEl.textContent = this.generationStats.completed;
        if (remainingEl) remainingEl.textContent = this.generationStats.total - this.generationStats.completed;
        if (successEl) successEl.textContent = this.generationStats.success;
        if (failedEl) failedEl.textContent = this.generationStats.failed;
    }

    enableExportButton() {
        const exportBtn = document.getElementById('export-all');
        if (exportBtn) {
            exportBtn.disabled = false;
        }
    }

    async exportAllArticles(projectId) {
        try {
            this.apiManager.showInfo('Preparing export...');
            
            const response = await fetch(`export_manager.php?action=export_all&project_id=${projectId}&format=zip`);
            
            if (response.ok) {
                const blob = await response.blob();
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `content_export_${new Date().getTime()}.zip`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                window.URL.revokeObjectURL(url);
                
                this.apiManager.showSuccess('All articles exported successfully as ZIP file!');
            } else {
                const errorText = await response.text();
                throw new Error(errorText || 'Export failed');
            }
        } catch (error) {
            this.apiManager.showError('Failed to export articles: ' + error.message);
        }
    }

    exportSingleArticle(content, title) {
        try {
            const blob = new Blob([content], { type: 'text/html' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `${this.sanitizeFilename(title)}.html`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            
            this.apiManager.showSuccess('Article exported successfully!');
        } catch (error) {
            this.apiManager.showError('Failed to export article: ' + error.message);
        }
    }

    applyInternalLinks(content, currentTopic, allTopics) {
        // Simple internal linking implementation
        let linkedContent = content;
        
        allTopics.forEach(topic => {
            if (topic !== currentTopic) {
                const regex = new RegExp(`\\b${this.escapeRegExp(topic)}\\b`, 'gi');
                linkedContent = linkedContent.replace(regex, 
                    `<a href="#${this.sanitizeId(topic)}" class="internal-link" title="Learn more about ${topic}">${topic}</a>`
                );
            }
        });

        return linkedContent;
    }

    async saveGeneratedContent(projectId, topic, content, config) {
        try {
            const response = await fetch('api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'save_content',
                    project_id: projectId,
                    topic: topic,
                    content: content,
                    config: config
                })
            });

            if (!response.ok) {
                throw new Error('Failed to save content to database');
            }
            
            const result = await response.json();
            if (!result.success) {
                throw new Error(result.error || 'Failed to save content');
            }
            
            return result.content_id;
        } catch (error) {
            console.error('Error saving content:', error);
            // Don't throw error here - we still want to show the content to user
            return null;
        }
    }

    async updateGeneratedContent(articleId, content, config) {
        try {
            const response = await fetch('api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'update_content',
                    article_id: articleId,
                    content: content,
                    config: config
                })
            });

            if (!response.ok) {
                throw new Error('Failed to update content in database');
            }
            
            const result = await response.json();
            if (!result.success) {
                throw new Error(result.error || 'Failed to update content');
            }
            
            return true;
        } catch (error) {
            console.error('Error updating content:', error);
            // Don't throw error here - we still want to show the updated content to user
            return false;
        }
    }

    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    sanitizeId(string) {
        return string.replace(/[^a-zA-Z0-9-_]/g, '_');
    }

    sanitizeFilename(filename) {
        return filename.replace(/[^a-z0-9]/gi, '_').toLowerCase();
    }
}

// Enhanced UI Utilities
class UIManager {
    static init() {
        this.setupGlobalEventListeners();
        this.initializeComponents();
    }

    static setupGlobalEventListeners() {
        // Handle all button clicks with loading states
        document.addEventListener('click', (e) => {
            if (e.target.matches('.btn-primary, .btn-secondary, .enhanced-btn')) {
                UIManager.setButtonLoading(e.target, true);
                
                // Auto-remove loading state after action completes
                setTimeout(() => {
                    UIManager.setButtonLoading(e.target, false);
                }, 3000);
            }
        });
    }

    static initializeComponents() {
        // Initialize collapsible panels
        UIManager.initializeCollapsibles();
    }

    static setButtonLoading(button, isLoading) {
        if (isLoading) {
            button.setAttribute('data-original-text', button.innerHTML);
            button.innerHTML = '<div class="loading-spinner small"><div class="spinner"></div></div> Loading...';
            button.disabled = true;
        } else {
            const originalText = button.getAttribute('data-original-text');
            if (originalText) {
                button.innerHTML = originalText;
                button.removeAttribute('data-original-text');
            }
            button.disabled = false;
        }
    }

    static initializeCollapsibles() {
        const collapsibles = document.querySelectorAll('.collapsible-section');
        collapsibles.forEach(section => {
            const header = section.querySelector('.collapsible-header');
            if (header) {
                header.addEventListener('click', () => {
                    section.classList.toggle('active');
                });
            }
        });
    }
}

// Initialize everything when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize theme manager
    const themeManager = new ThemeManager();
    
    // Initialize UI components
    UIManager.init();
    
    // Initialize content generation manager
    const contentManager = new ContentGenerationManager();
    
    // Make managers globally available
    window.themeManager = themeManager;
    window.contentManager = contentManager;
    window.apiManager = contentManager.apiManager;

    console.log('Enhanced Bulk AI Content Writer initialized successfully!');
});

// Utility function for making API requests
function makeRequest(url, data = {}, method = 'POST') {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open(method, url);
        xhr.setRequestHeader('Content-Type', 'application/json');
        
        xhr.onload = function() {
            if (xhr.status >= 200 && xhr.status < 300) {
                try {
                    resolve(JSON.parse(xhr.responseText));
                } catch (e) {
                    resolve(xhr.responseText);
                }
            } else {
                reject(new Error('Request failed: ' + xhr.statusText));
            }
        };
        
        xhr.onerror = function() {
            reject(new Error('Network error'));
        };
        
        xhr.send(JSON.stringify(data));
    });
}

// Error boundary for unhandled errors
window.addEventListener('error', function(e) {
    console.error('Application error:', e.error);
    
    if (window.apiManager) {
        window.apiManager.showNotification('An unexpected error occurred. Please refresh the page.', 'error');
    }
});