// Quillfy AI - Enhanced Chat Functionality
class ChatManager {
    constructor() {
        this.isInitialized = false;
        this.init();
    }

    init() {
        document.addEventListener('DOMContentLoaded', () => {
            this.initializeChatFeatures();
            this.isInitialized = true;
        });
    }

    initializeChatFeatures() {
        this.setupChatInput();
        this.setupAutoFocus();
        this.setupMessageInteractions();
        this.setupMathJax();
    }

    setupChatInput() {
        const chatInput = document.querySelector('.chat-input textarea');
        const sendButton = document.querySelector('.chat-input .send-button');

        if (chatInput && sendButton) {
            // Remove any existing event listeners
            const newChatInput = chatInput.cloneNode(true);
            const newSendButton = sendButton.cloneNode(true);
            
            chatInput.parentNode.replaceChild(newChatInput, chatInput);
            sendButton.parentNode.replaceChild(newSendButton, sendButton);

            // Add single event listener for Enter key
            newChatInput.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    if (!newSendButton.disabled && newChatInput.value.trim()) {
                        this.triggerSendMessage();
                    }
                }
            });

            // Auto-resize textarea
            newChatInput.addEventListener('input', (e) => {
                this.autoResizeTextarea(e.target);
                this.toggleSendButton(e.target.value.trim(), newSendButton);
            });

            // Single click handler for send button
            newSendButton.addEventListener('click', () => {
                if (!newSendButton.disabled && newChatInput.value.trim()) {
                    this.triggerSendMessage();
                }
            });

            // Focus management
            newChatInput.addEventListener('focus', () => {
                newChatInput.parentElement.classList.add('focused');
            });

            newChatInput.addEventListener('blur', () => {
                newChatInput.parentElement.classList.remove('focused');
            });
        }
    }

    setupAutoFocus() {
        const chatSection = document.querySelector('.chat-main');
        const chatInput = document.querySelector('.chat-input textarea');

        if (chatSection && chatInput) {
            chatSection.addEventListener('click', (e) => {
                if (!e.target.closest('.chat-messages .message') && 
                    !e.target.closest('.chat-input button') &&
                    !e.target.closest('.file-upload-btn')) {
                    chatInput.focus();
                }
            });
        }
    }

    setupMessageInteractions() {
        document.addEventListener('click', (e) => {
            if (e.target.closest('.code-block')) {
                this.handleCodeBlockClick(e);
            }

            if (e.target.closest('.message-action')) {
                this.handleMessageAction(e);
            }
        });
    }

    setupMathJax() {
        // MathJax will be configured in the separate config file
        if (window.MathJax) {
            window.MathJax.typesetPromise();
        }
    }

    handleCodeBlockClick(e) {
        const codeBlock = e.target.closest('.code-block');
        if (codeBlock) {
            const code = codeBlock.querySelector('code');
            if (code) {
                navigator.clipboard.writeText(code.textContent).then(() => {
                    this.showCopyFeedback('Code copied to neural cache!');
                }).catch(() => {
                    this.showCopyFeedback('Failed to copy code');
                });
            }
        }
    }

    showCopyFeedback(message) {
        const feedback = document.createElement('div');
        feedback.className = 'copy-feedback';
        feedback.textContent = message;
        feedback.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #4CAF50;
            color: white;
            padding: 10px 20px;
            border-radius: 5px;
            z-index: 1000;
            animation: slideIn 0.3s ease;
        `;

        document.body.appendChild(feedback);

        setTimeout(() => {
            feedback.style.animation = 'slideOut 0.3s ease';
            setTimeout(() => {
                document.body.removeChild(feedback);
            }, 300);
        }, 2000);
    }

    handleMessageAction(e) {
        const action = e.target.closest('.message-action');
        if (!action) return;

        const messageElement = action.closest('.message');
        const messageText = messageElement.querySelector('.message-text').textContent;

        if (action.querySelector('.fa-copy')) {
            this.copyMessageText(messageText);
        }
    }

    copyMessageText(text) {
        navigator.clipboard.writeText(text).then(() => {
            this.showCopyFeedback('Consciousness pattern copied!');
        }).catch(() => {
            this.showCopyFeedback('Failed to copy neural pattern');
        });
    }

    autoResizeTextarea(textarea) {
        textarea.style.height = 'auto';
        textarea.style.height = Math.min(textarea.scrollHeight, 120) + 'px';
    }

    toggleSendButton(text, sendButton) {
        if (sendButton) {
            sendButton.disabled = !text.trim();
            sendButton.style.opacity = text.trim() ? '1' : '0.6';
            sendButton.style.cursor = text.trim() ? 'pointer' : 'not-allowed';
        }
    }

    triggerSendMessage() {
        const sendButton = document.querySelector('.send-button');
        
        if (sendButton && !sendButton.disabled) {
            sendButton.click();
        }
    }

    scrollToBottom() {
        const chatMessages = document.querySelector('.chat-messages');
        if (chatMessages) {
            chatMessages.scrollTop = chatMessages.scrollHeight;
        }
    }

    showTypingIndicator() {
        const chatMessages = document.querySelector('.chat-messages');
        if (!chatMessages) return;

        this.removeTypingIndicator();

        const typingDiv = document.createElement('div');
        typingDiv.className = 'message assistant typing-indicator';
        typingDiv.innerHTML = `
            <div class="message-avatar">
                <img src="${window.currentAgent?.avatar || 'https://randomuser.me/api/portraits/lego/5.jpg'}" alt="AI Agent">
            </div>
            <div class="message-content">
                <div class="typing-dots">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
                <div class="message-time">${new Date().toLocaleTimeString()}</div>
            </div>
        `;

        chatMessages.appendChild(typingDiv);
        this.scrollToBottom();
        return typingDiv;
    }

    removeTypingIndicator() {
        const existingIndicator = document.querySelector('.typing-indicator');
        if (existingIndicator) {
            existingIndicator.remove();
        }
    }
}

// Initialize chat manager when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.chatManager = new ChatManager();
});

// Vue integration helpers
if (typeof Vue !== 'undefined') {
    Vue.mixin({
        methods: {
            // Enhanced chat message sending with streaming
            async sendChatMessage() {
                if (!this.chatInput?.trim() || this.isChatLoading) {
                    return;
                }

                const userMessage = this.chatInput;
                this.chatInput = '';
                
                this.chatMessages.push({
                    type: 'user',
                    text: userMessage,
                    time: new Date().toLocaleTimeString()
                });
                
                this.autoSave();
                this.scrollChatToBottom();

                this.isChatLoading = true;
                this.streamingResponse = true;
                
                try {
                    const conversationHistory = this.chatMessages.slice(-10).map(msg => 
                        `${msg.type === 'user' ? 'User' : this.currentAgent.name}: ${msg.text}`
                    ).join('\n');

                    const formData = new FormData();
                    formData.append('action', 'chat');
                    formData.append('message', userMessage);
                    formData.append('history', conversationHistory);
                    formData.append('user_profile', JSON.stringify(this.user));
                    formData.append('agent_persona', this.currentAgent.persona);
                    formData.append('ai_settings', JSON.stringify(this.aiSettings));
                    formData.append('stream', 'true');

                    if (this.uploadedFile) {
                        formData.append('file_content', this.uploadedFile.content);
                    }

                    const response = await fetch('config/config.php', {
                        method: 'POST',
                        body: formData
                    });

                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }

                    const reader = response.body.getReader();
                    const decoder = new TextDecoder();
                    let assistantMessage = '';

                    const streamingMessageIndex = this.chatMessages.length;
                    this.chatMessages.push({
                        type: 'assistant',
                        text: '▊',
                        time: new Date().toLocaleTimeString()
                    });

                    while (true) {
                        const { done, value } = await reader.read();
                        if (done) break;

                        const chunk = decoder.decode(value);
                        const lines = chunk.split('\n');

                        for (const line of lines) {
                            if (line.startsWith('data: ')) {
                                try {
                                    const data = JSON.parse(line.slice(6));
                                    if (data.candidates && data.candidates[0] && data.candidates[0].content) {
                                        const text = data.candidates[0].content.parts[0].text;
                                        if (text) {
                                            assistantMessage += text;
                                            this.chatMessages[streamingMessageIndex].text = assistantMessage + '▊';
                                            this.scrollChatToBottom();
                                        }
                                    }
                                } catch (e) {
                                    // Continue processing
                                }
                            }
                        }
                    }

                    if (assistantMessage) {
                        this.chatMessages[streamingMessageIndex].text = assistantMessage;
                    } else {
                        await this.sendNonStreamingMessage(userMessage, conversationHistory);
                    }

                    this.autoSave();
                    this.scrollChatToBottom();
                    
                } catch (error) {
                    console.error('Chat error:', error);
                    this.chatMessages.push({
                        type: 'assistant',
                        text: `**🌀 Quantum Anomaly**\n\nI experienced a consciousness disruption: ${error.message}\n\n*Recalibrating neural pathways...*`,
                        time: new Date().toLocaleTimeString()
                    });
                } finally {
                    this.isChatLoading = false;
                    this.streamingResponse = false;
                }
            },

            async sendNonStreamingMessage(userMessage, conversationHistory) {
                const formData = new FormData();
                formData.append('action', 'chat');
                formData.append('message', userMessage);
                formData.append('history', conversationHistory);
                formData.append('user_profile', JSON.stringify(this.user));
                formData.append('agent_persona', this.currentAgent.persona);
                formData.append('ai_settings', JSON.stringify(this.aiSettings));

                if (this.uploadedFile) {
                    formData.append('file_content', this.uploadedFile.content);
                }

                const response = await fetch('config/config.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();
                
                if (result.error) {
                    throw new Error(result.error);
                }

                this.chatMessages.push({
                    type: 'assistant',
                    text: result.text,
                    time: new Date().toLocaleTimeString()
                });
            },

            scrollChatToBottom() {
                this.$nextTick(() => {
                    const container = this.$refs.chatMessagesContainer;
                    if (container) {
                        container.scrollTop = container.scrollHeight;
                    }
                });
            }
        }
    });
}