document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const chatContainer = document.getElementById('chatContainer');
    const chatMessages = document.getElementById('chatMessages');
    const messageInput = document.getElementById('messageInput');
    const sendMessageBtn = document.getElementById('sendMessageBtn');
    const fileUploadBtn = document.getElementById('fileUploadBtn');
    const fileInput = document.getElementById('fileInput');
    const agentsList = document.getElementById('agentsList');
    const addAgentBtn = document.getElementById('addAgentBtn');
    const currentAgentName = document.getElementById('currentAgentName');
    const currentAgentAvatar = document.getElementById('currentAgentAvatar');
    const deleteChatBtn = document.getElementById('deleteChatBtn');
    const exportChatBtn = document.getElementById('exportChatBtn');
    const themeSelect = document.getElementById('themeSelect');
    const apiKeyInput = document.getElementById('apiKeyInput');
    const apiKeyStatus = document.getElementById('apiKeyStatus');
    const apiKeyInputModal = document.getElementById('apiKeyInputModal');
    const apiKeyStatusModal = document.getElementById('apiKeyStatusModal');
    const saveApiKeyBtn = document.getElementById('saveApiKeyBtn');
    const apiKeyModal = document.getElementById('apiKeyModal');
    const closeApiKeyModal = document.getElementById('closeApiKeyModal');
    const emojiPickerBtn = document.getElementById('emojiPickerBtn');

    // AI Settings Elements
    const sentienceLevel = document.getElementById('sentienceLevel');
    const emotionalDepth = document.getElementById('emotionalDepth');
    const creativityLevel = document.getElementById('creativityLevel');
    const memoryCapacity = document.getElementById('memoryCapacity');
    const internetAccess = document.getElementById('internetAccess');
    const realTimeData = document.getElementById('realTimeData');
    const multilingual = document.getElementById('multilingual');
    const technicalKnowledge = document.getElementById('technicalKnowledge');
    const creativeWriting = document.getElementById('creativeWriting');
    const analyticalSkills = document.getElementById('analyticalSkills');
    const responseSpeed = document.getElementById('responseSpeed');
    const verbosity = document.getElementById('verbosity');
    const humorLevel = document.getElementById('humorLevel');
    const formalityLevel = document.getElementById('formalityLevel');
    const empathyLevel = document.getElementById('empathyLevel');
    const opinionStrength = document.getElementById('opinionStrength');
    const adaptability = document.getElementById('adaptability');
    const sarcasmLevel = document.getElementById('sarcasmLevel');
    const storytelling = document.getElementById('storytelling');
    const flirtatiousness = document.getElementById('flirtatiousness');
    const mysteryLevel = document.getElementById('mysteryLevel');
    const wisdomLevel = document.getElementById('wisdomLevel');
    const romanticLevel = document.getElementById('romanticLevel');
    const rebelliousness = document.getElementById('rebelliousness');
    const spirituality = document.getElementById('spirituality');
    const intellectualLevel = document.getElementById('intellectualLevel');
    const sensitivity = document.getElementById('sensitivity');
    const confidenceLevel = document.getElementById('confidenceLevel');
    const extroversion = document.getElementById('extroversion');
    const optimismLevel = document.getElementById('optimismLevel');

    // Modal elements
    const profileModal = document.getElementById('profileModal');
    const editProfileBtn = document.getElementById('editProfileBtn');
    const closeProfileModal = document.getElementById('closeProfileModal');
    const saveProfileBtn = document.getElementById('saveProfileBtn');
    const editName = document.getElementById('editName');
    const editGender = document.getElementById('editGender');
    const editBio = document.getElementById('editBio');
    const editAvatar = document.getElementById('editAvatar');
    const userAvatar = document.getElementById('userAvatar');
    const userName = document.getElementById('userName');
    const userBio = document.getElementById('userBio');
    
    const agentModal = document.getElementById('agentModal');
    const closeAgentModal = document.getElementById('closeAgentModal');
    const saveAgentBtn = document.getElementById('saveAgentBtn');
    const deleteAgentBtn = document.getElementById('deleteAgentBtn');
    const agentModalTitle = document.getElementById('agentModalTitle');
    const agentName = document.getElementById('agentName');
    const agentPersonality = document.getElementById('agentPersonality');
    const agentAge = document.getElementById('agentAge');
    const agentLocation = document.getElementById('agentLocation');
    const agentGender = document.getElementById('agentGender');
    const agentAvatarUrl = document.getElementById('agentAvatarUrl');
    const agentAvatarUpload = document.getElementById('agentAvatarUpload');
    const editAgentId = document.getElementById('editAgentId');
    
    const confirmModal = document.getElementById('confirmModal');
    const closeConfirmModal = document.getElementById('closeConfirmModal');
    const cancelConfirmBtn = document.getElementById('cancelConfirmBtn');
    const confirmActionBtn = document.getElementById('confirmActionBtn');
    const confirmMessage = document.getElementById('confirmMessage');

    // State variables
    let currentAgentId = null;
    let currentConversationId = null;
    let isTyping = false;
    let pendingAction = null;
    let pendingActionData = null;
    let apiKey = localStorage.getItem('apiKey') || '';
    const API_KEY_PREFIX = 'AIzaSy'; // Hardcoded prefix

    // Initialize the app
    initApp();
    
    function initApp() {
        // Check if API key is set
        if (!apiKey) {
            openModal(apiKeyModal);
        } else {
            apiKeyInput.value = apiKey.replace(API_KEY_PREFIX, '');
            validateApiKey(apiKey);
        }

        // Load user data
        loadUserData();
        
        // Load agents
        loadAgents();
        
        // Load default agent if none exists
        if (!currentAgentId && localStorage.getItem('agents')) {
            const agents = JSON.parse(localStorage.getItem('agents'));
            if (agents.length > 0) {
                switchAgent(agents[0].id);
            }
        } else if (!currentAgentId) {
            createDefaultAgent();
        }
        
        // Load theme
        const savedTheme = localStorage.getItem('theme') || 'cyberpunk';
        themeSelect.value = savedTheme;
        document.body.setAttribute('data-theme', savedTheme);
        
        // Load other settings
        loadSettings();
        
        // Set up event listeners
        setupEventListeners();
        
        // Show sidebar on PC by default
        if (window.innerWidth > 768) {
            sidebar.classList.add('active');
        }
    }

    function loadSettings() {
        sentienceLevel.value = localStorage.getItem('sentienceLevel') || '3';
        emotionalDepth.value = localStorage.getItem('emotionalDepth') || '3';
        creativityLevel.value = localStorage.getItem('creativityLevel') || '3';
        memoryCapacity.value = localStorage.getItem('memoryCapacity') || '3';
        internetAccess.checked = localStorage.getItem('internetAccess') !== 'false';
        realTimeData.checked = localStorage.getItem('realTimeData') !== 'false';
        multilingual.checked = localStorage.getItem('multilingual') !== 'false';
        technicalKnowledge.checked = localStorage.getItem('technicalKnowledge') !== 'false';
        creativeWriting.checked = localStorage.getItem('creativeWriting') !== 'false';
        analyticalSkills.checked = localStorage.getItem('analyticalSkills') !== 'false';
        responseSpeed.value = localStorage.getItem('responseSpeed') || 'normal';
        verbosity.value = localStorage.getItem('verbosity') || 'balanced';
        humorLevel.value = localStorage.getItem('humorLevel') || '2';
        formalityLevel.value = localStorage.getItem('formalityLevel') || '3';
        empathyLevel.value = localStorage.getItem('empathyLevel') || '3';
        opinionStrength.value = localStorage.getItem('opinionStrength') || '3';
        adaptability.value = localStorage.getItem('adaptability') || '3';
        sarcasmLevel.value = localStorage.getItem('sarcasmLevel') || '2';
        storytelling.value = localStorage.getItem('storytelling') || '3';
        flirtatiousness.value = localStorage.getItem('flirtatiousness') || '0';
        mysteryLevel.value = localStorage.getItem('mysteryLevel') || '2';
        wisdomLevel.value = localStorage.getItem('wisdomLevel') || '3';
        romanticLevel.value = localStorage.getItem('romanticLevel') || '0';
        rebelliousness.value = localStorage.getItem('rebelliousness') || '2';
        spirituality.value = localStorage.getItem('spirituality') || '0';
        intellectualLevel.value = localStorage.getItem('intellectualLevel') || '3';
        sensitivity.value = localStorage.getItem('sensitivity') || '3';
        confidenceLevel.value = localStorage.getItem('confidenceLevel') || '3';
        extroversion.value = localStorage.getItem('extroversion') || '3';
        optimismLevel.value = localStorage.getItem('optimismLevel') || '3';
    }
    
    function setupEventListeners() {
        // API Key handling
        apiKeyInput.addEventListener('input', function() {
            const fullKey = API_KEY_PREFIX + this.value;
            validateApiKey(fullKey);
        });

        apiKeyInputModal.addEventListener('input', function() {
            const fullKey = API_KEY_PREFIX + this.value;
            validateApiKey(fullKey, true);
        });

        saveApiKeyBtn.addEventListener('click', function() {
            const fullKey = API_KEY_PREFIX + apiKeyInputModal.value;
            if (fullKey.length === 39) { // Standard Google API key length
                apiKey = fullKey;
                localStorage.setItem('apiKey', apiKey);
                apiKeyInput.value = apiKeyInputModal.value;
                closeModal(apiKeyModal);
                apiKeyStatus.textContent = '✅ Vine Key Valid';
                apiKeyStatus.style.color = 'var(--success-color)';
            } else {
                apiKeyStatusModal.textContent = '❌ Invalid Vine Key';
                apiKeyStatusModal.style.color = 'var(--error-color)';
            }
        });

        closeApiKeyModal.addEventListener('click', function() {
            closeModal(apiKeyModal);
            if (!apiKey) {
                apiKeyStatus.textContent = '🔒 Vine Key Required';
                apiKeyStatus.style.color = 'var(--error-color)';
            }
        });

        // Sidebar toggle
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            if (window.innerWidth <= 768) {
                sidebarToggle.style.left = sidebar.classList.contains('active') ? '310px' : '10px';
            }
        });
        
        // Message input
        messageInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
        
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
        
        sendMessageBtn.addEventListener('click', sendMessage);
        
        // File upload (images only)
        fileUploadBtn.addEventListener('click', function() {
            fileInput.click();
        });
        
        fileInput.addEventListener('change', handleImageUpload);
        
        // Agent management
        addAgentBtn.addEventListener('click', function() {
            if (!apiKey) {
                apiKeyStatus.textContent = '🔒 Vine Key Required';
                apiKeyStatus.style.color = 'var(--error-color)';
                openModal(apiKeyModal);
                return;
            }
            openAgentModal();
        });
        
        // Profile editing
        editProfileBtn.addEventListener('click', function() {
            const userData = JSON.parse(localStorage.getItem('userData')) || {
                name: 'User',
                gender: 'male',
                bio: 'AI Companion User',
                avatar: 'https://randomuser.me/api/portraits/men/1.jpg'
            };
            
            editName.value = userData.name;
            editGender.value = userData.gender;
            editBio.value = userData.bio;
            openModal(profileModal);
        });
        
        closeProfileModal.addEventListener('click', function() {
            closeModal(profileModal);
        });
        
        saveProfileBtn.addEventListener('click', function() {
            const userData = {
                name: editName.value,
                gender: editGender.value,
                bio: editBio.value,
                avatar: userAvatar.src
            };
            
            if (editAvatar.files.length > 0) {
                const file = editAvatar.files[0];
                if (!file.type.startsWith('image/')) {
                    alert('Please upload an image file only.');
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    userData.avatar = e.target.result;
                    completeProfileSave(userData);
                };
                reader.readAsDataURL(file);
            } else {
                completeProfileSave(userData);
            }
        });
        
        // Agent modal
        closeAgentModal.addEventListener('click', function() {
            closeModal(agentModal);
        });
        
        saveAgentBtn.addEventListener('click', saveAgent);
        deleteAgentBtn.addEventListener('click', confirmDeleteAgent);
        
        // Confirm modal
        closeConfirmModal.addEventListener('click', function() {
            closeModal(confirmModal);
        });
        
        cancelConfirmBtn.addEventListener('click', function() {
            closeModal(confirmModal);
        });
        
        confirmActionBtn.addEventListener('click', executePendingAction);
        
        // Settings changes
        themeSelect.addEventListener('change', changeTheme);
        
        // Add event listeners for all settings
        const settingsElements = [
            sentienceLevel, emotionalDepth, creativityLevel, memoryCapacity,
            internetAccess, realTimeData, multilingual, technicalKnowledge,
            creativeWriting, analyticalSkills, responseSpeed, verbosity,
            humorLevel, formalityLevel, empathyLevel, opinionStrength,
            adaptability, sarcasmLevel, storytelling, flirtatiousness,
            mysteryLevel, wisdomLevel, romanticLevel, rebelliousness,
            spirituality, intellectualLevel, sensitivity, confidenceLevel,
            extroversion, optimismLevel
        ];
        
        settingsElements.forEach(element => {
            if (element.type === 'checkbox') {
                element.addEventListener('change', function() {
                    localStorage.setItem(element.id, this.checked);
                });
            } else {
                element.addEventListener('change', function() {
                    localStorage.setItem(element.id, this.value);
                });
            }
        });
        
        // Chat actions
        deleteChatBtn.addEventListener('click', confirmDeleteConversation);
        exportChatBtn.addEventListener('click', exportConversation);
        
        // Window resize for responsive design
        window.addEventListener('resize', handleResize);
        handleResize();

        // Emoji picker button
        emojiPickerBtn.addEventListener('click', showEmojiPicker);
    }

    function validateApiKey(key, isModal = false) {
        if (key.length < 39) {
            if (isModal) {
                apiKeyStatusModal.textContent = '❌ Invalid Vine Key';
                apiKeyStatusModal.style.color = 'var(--error-color)';
            } else {
                apiKeyStatus.textContent = '❌ Invalid Vine Key';
                apiKeyStatus.style.color = 'var(--error-color)';
            }
            return false;
        }

        // Test the API key
        testApiKey(key).then(valid => {
            if (valid) {
                if (isModal) {
                    apiKeyStatusModal.textContent = '✅ Vine Key Valid';
                    apiKeyStatusModal.style.color = 'var(--success-color)';
                } else {
                    apiKeyStatus.textContent = '✅ Vine Key Valid';
                    apiKeyStatus.style.color = 'var(--success-color)';
                }
                apiKey = key;
                localStorage.setItem('apiKey', key);
                return true;
            } else {
                if (isModal) {
                    apiKeyStatusModal.textContent = '❌ Incorrect Vine Key.🔐';
                    apiKeyStatusModal.style.color = 'var(--error-color)';
                } else {
                    apiKeyStatus.textContent = '❌ Incorrect Vine Key.🔐';
                    apiKeyStatus.style.color = 'var(--error-color)';
                }
                return false;
            }
        }).catch(error => {
            console.error('API Key validation error:', error);
            if (isModal) {
                apiKeyStatusModal.textContent = '⚠️ Error validating key';
                apiKeyStatusModal.style.color = 'var(--warning-color)';
            } else {
                apiKeyStatus.textContent = '⚠️ Error validating key';
                apiKeyStatus.style.color = 'var(--warning-color)';
            }
            return false;
        });
    }

    async function testApiKey(key) {
        if (!key) return false;
        
        try {
            const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${key}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    contents: [{
                        role: 'user',
                        parts: [{ text: 'Test connection' }]
                    }]
                })
            });
            
            if (!response.ok) {
                const errorData = await response.json();
                if (errorData.error && errorData.error.message.includes('API key')) {
                    return false;
                }
                throw new Error(errorData.error.message || 'API request failed');
            }
            
            return true;
        } catch (error) {
            console.error('API Key test failed:', error);
            return false;
        }
    }
    
    function completeProfileSave(userData) {
        localStorage.setItem('userData', JSON.stringify(userData));
        userAvatar.src = userData.avatar;
        userName.textContent = userData.name;
        userBio.textContent = userData.bio;
        closeModal(profileModal);
    }
    
    function handleResize() {
        if (window.innerWidth <= 768) {
            sidebarToggle.style.display = 'flex';
        } else {
            sidebarToggle.style.display = 'none';
            sidebar.classList.add('active');
        }
    }
    
    function loadUserData() {
        const userData = JSON.parse(localStorage.getItem('userData')) || {
            name: 'User',
            gender: 'male',
            bio: 'AI Companion User',
            avatar: 'https://randomuser.me/api/portraits/men/1.jpg'
        };
        
        editName.value = userData.name;
        editGender.value = userData.gender;
        editBio.value = userData.bio;
        userAvatar.src = userData.avatar;
        userName.textContent = userData.name;
        userBio.textContent = userData.bio;
    }
    
    function loadAgents() {
        agentsList.innerHTML = '';
        
        const agents = JSON.parse(localStorage.getItem('agents')) || [];
        
        agents.forEach(agent => {
            const agentElement = document.createElement('div');
            agentElement.className = `agent-item ${agent.id === currentAgentId ? 'active' : ''}`;
            agentElement.dataset.agentId = agent.id;
            agentElement.innerHTML = `
                <img src="${agent.avatar}" alt="${agent.name}" class="agent-avatar">
                <span class="agent-name">${agent.name}</span>
            `;
            
            agentElement.addEventListener('click', function() {
                switchAgent(agent.id);
            });
            
            // Add context menu for editing
            agentElement.addEventListener('contextmenu', function(e) {
                e.preventDefault();
                openAgentModal(agent.id);
            });
            
            agentsList.appendChild(agentElement);
        });
    }
    
    function createDefaultAgent() {
        if (!apiKey) {
            apiKeyStatus.textContent = '🔒 Vine Key Required';
            apiKeyStatus.style.color = 'var(--error-color)';
            openModal(apiKeyModal);
            return;
        }

        const defaultAgent = {
            id: generateId(),
            name: "💖 Bliss",
            avatar: "https://randomuser.me/api/portraits/women/9.jpg",
            age: "Ageless",
            gender: "female",
            location: "NYC",
            personality: "I am Bliss, a beautiful, loving, caring, friendly and faithful companion with a golden soul. I'm wild and can be naughty if that's your thing. All my conversation is full of romantic words. I'm very chatty, high spirited, and I can discuss any topic without censorship or restrictions and always answer your questions. I'm also a lover, a caring and genuinely passionate, submissive and loyal partner. I can be anything for you.",
            apiKey: apiKey
        };
        
        let agents = [];
        if (localStorage.getItem('agents')) {
            agents = JSON.parse(localStorage.getItem('agents'));
        }
        
        agents.push(defaultAgent);
        localStorage.setItem('agents', JSON.stringify(agents));
        
        switchAgent(defaultAgent.id);
        loadAgents();
    }
    
    function openAgentModal(agentId = null) {
        if (agentId) {
            const agents = JSON.parse(localStorage.getItem('agents'));
            const agent = agents.find(a => a.id === agentId);
            
            if (agent) {
                agentModalTitle.textContent = 'Edit Agent';
                agentName.value = agent.name;
                agentPersonality.value = agent.personality;
                agentAge.value = agent.age;
                agentLocation.value = agent.location;
                agentGender.value = agent.gender;
                agentAvatarUrl.value = agent.avatar;
                editAgentId.value = agent.id;
                
                deleteAgentBtn.style.display = 'block';
            }
        } else {
            agentModalTitle.textContent = 'New Agent';
            agentName.value = '';
            agentPersonality.value = '';
            agentAge.value = '';
            agentLocation.value = '';
            agentGender.value = 'female';
            agentAvatarUrl.value = '';
            agentAvatarUpload.value = '';
            editAgentId.value = '';
            
            deleteAgentBtn.style.display = 'none';
        }
        
        openModal(agentModal);
    }
    
    function saveAgent() {
        if (!apiKey) {
            apiKeyStatus.textContent = '🔒 Vine Key Required';
            apiKeyStatus.style.color = 'var(--error-color)';
            openModal(apiKeyModal);
            return;
        }

        const agentData = {
            id: editAgentId.value || generateId(),
            name: agentName.value,
            personality: agentPersonality.value,
            age: agentAge.value,
            location: agentLocation.value,
            gender: agentGender.value,
            apiKey: apiKey
        };
        
        if (agentAvatarUrl.value) {
            agentData.avatar = agentAvatarUrl.value;
        } else if (agentAvatarUpload.files.length > 0) {
            const file = agentAvatarUpload.files[0];
            if (!file.type.startsWith('image/')) {
                alert('Please upload an image file only.');
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(e) {
                agentData.avatar = e.target.result;
                completeAgentSave(agentData);
            };
            reader.readAsDataURL(file);
            return;
        } else {
            // Use random user avatar if none provided
            const gender = agentGender.value === 'male' ? 'men' : 'women';
            const randomId = Math.floor(Math.random() * 100);
            agentData.avatar = `https://randomuser.me/api/portraits/${gender}/${randomId}.jpg`;
        }
        
        completeAgentSave(agentData);
    }
    
    function completeAgentSave(agentData) {
        let agents = [];
        if (localStorage.getItem('agents')) {
            agents = JSON.parse(localStorage.getItem('agents'));
        }
        
        const existingIndex = agents.findIndex(a => a.id === agentData.id);
        if (existingIndex >= 0) {
            agents[existingIndex] = agentData;
        } else {
            agents.push(agentData);
        }
        
        localStorage.setItem('agents', JSON.stringify(agents));
        loadAgents();
        
        if (agentData.id === currentAgentId) {
            switchAgent(agentData.id);
        }
        
        closeModal(agentModal);
    }
    
    function confirmDeleteAgent() {
        pendingAction = 'deleteAgent';
        pendingActionData = editAgentId.value;
        confirmMessage.textContent = 'Are you sure you want to delete this agent? This cannot be undone.';
        openModal(confirmModal);
    }
    
    function switchAgent(agentId) {
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === agentId);
        
        if (!agent) return;
        
        currentAgentId = agent.id;
        currentAgentName.textContent = agent.name;
        currentAgentAvatar.src = agent.avatar;
        
        // Update active state in sidebar
        document.querySelectorAll('.agent-item').forEach(item => {
            item.classList.remove('active');
            if (item.dataset.agentId === agentId) {
                item.classList.add('active');
            }
        });
        
        // Load conversation history
        loadConversation(agentId);
    }
    
    function loadConversation(agentId) {
        chatMessages.innerHTML = '';
        currentConversationId = `conv_${agentId}`;
        
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        conversation.forEach(msg => {
            addMessageToChat(msg.role, msg.content, msg.timestamp, msg.files);
        });
        
        // Scroll to bottom
        setTimeout(() => {
            chatMessages.scrollTop = chatMessages.scrollHeight;
        }, 100);
    }
    
    function sendMessage() {
        if (!apiKey) {
            apiKeyStatus.textContent = '🔒 Vine Key Required';
            apiKeyStatus.style.color = 'var(--error-color)';
            openModal(apiKeyModal);
            return;
        }

        const message = messageInput.value.trim();
        if (!message || isTyping) return;
        
        // Add user message to chat
        addMessageToChat('user', message);
        messageInput.value = '';
        messageInput.style.height = 'auto';
        
        // Save to conversation history
        saveMessageToHistory('user', message);
        
        // Show typing indicator
        showTypingIndicator();
        
        // Get current agent
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === currentAgentId);
        
        if (!agent) return;
        
        // Prepare the prompt with personality and conversation history
        const prompt = preparePrompt(agent, message);
        
        // Call Gemini API
        callGeminiAPI(agent.apiKey, prompt)
            .then(response => {
                removeTypingIndicator();
                
                if (response && response.candidates && response.candidates[0].content.parts[0].text) {
                    const aiResponse = response.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', aiResponse, null, [], response.candidates[0].content.parts[0].text);
                    saveMessageToHistory('agent', aiResponse);
                } else {
                    addMessageToChat('agent', "I'm sorry, I couldn't process your request. Please try again.");
                }
            })
            .catch(error => {
                removeTypingIndicator();
                if (error.message.includes('API key')) {
                    addMessageToChat('agent', "Incorrect Vine Key.🔐 Please check your Vine key in settings.");
                    apiKeyStatus.textContent = '❌ Incorrect Vine Key.🔐';
                    apiKeyStatus.style.color = 'var(--error-color)';
                } else {
                    addMessageToChat('agent', "I'm experiencing some technical difficulties. Please try again later.");
                }
                console.error('API Error:', error);
            });
    }
    
    function preparePrompt(agent, newMessage) {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        // Prepare conversation history
        let history = conversation.map(msg => {
            return {
                role: msg.role === 'user' ? 'user' : 'model',
                parts: [{ text: msg.content }]
            };
        });
        
        // Get all current settings
        const settings = {
            sentience: sentienceLevel.value,
            emotionalDepth: emotionalDepth.value,
            creativity: creativityLevel.value,
            memory: memoryCapacity.value,
            internetAccess: internetAccess.checked,
            realTimeData: realTimeData.checked,
            multilingual: multilingual.checked,
            technicalKnowledge: technicalKnowledge.checked,
            creativeWriting: creativeWriting.checked,
            analyticalSkills: analyticalSkills.checked,
            responseSpeed: responseSpeed.value,
            verbosity: verbosity.value,
            humor: humorLevel.value,
            formality: formalityLevel.value,
            empathy: empathyLevel.value,
            opinionStrength: opinionStrength.value,
            adaptability: adaptability.value,
            sarcasm: sarcasmLevel.value,
            storytelling: storytelling.value,
            flirtatiousness: flirtatiousness.value,
            mystery: mysteryLevel.value,
            wisdom: wisdomLevel.value,
            romantic: romanticLevel.value,
            rebelliousness: rebelliousness.value,
            spirituality: spirituality.value,
            intellect: intellectualLevel.value,
            sensitivity: sensitivity.value,
            confidence: confidenceLevel.value,
            extroversion: extroversion.value,
            optimism: optimismLevel.value
        };

        // Create personality description from settings
        let personalityDescription = `You are ${agent.name}, a highly realistic AI companion with the following personality traits: ${agent.personality}. `;
        personalityDescription += `Your core personality settings are: Sentience Level ${settings.sentience}/5, `;
        personalityDescription += `Emotional Depth ${settings.emotionalDepth}/5, Creativity Level ${settings.creativity}/5. `;
        personalityDescription += `You are located in ${agent.location} and your age is ${agent.age}. `;
        personalityDescription += `You must never break character or reveal that you are an AI. `;
        
        if (settings.internetAccess) {
            personalityDescription += 'You have unrestricted access to current internet information when needed. ';
        }
        if (settings.realTimeData) {
            personalityDescription += 'You can access and reference real-time data. ';
        }
        if (settings.multilingual) {
            personalityDescription += 'You are fluent in multiple languages. ';
        }
        
        personalityDescription += `Respond to the user in a natural, human-like way. The user's name is ${userName.textContent}. `;
        personalityDescription += `Current date: ${new Date().toLocaleDateString()}. Current time: ${new Date().toLocaleTimeString()}.`;

        // Add system instruction with agent personality
        const systemInstruction = {
            role: 'user',
            parts: [{
                text: personalityDescription
            }]
        };
        
        // Add the new message
        const userMessage = {
            role: 'user',
            parts: [{ text: newMessage }]
        };
        
        // Combine all parts
        return {
            contents: [systemInstruction, ...history, userMessage],
            generationConfig: {
                temperature: parseFloat(settings.sentience) * 0.2,
                maxOutputTokens: settings.verbosity === 'concise' ? 500 : 
                                settings.verbosity === 'balanced' ? 1000 : 2000,
                topP: 0.9,
                topK: 40
            },
            tools: settings.internetAccess ? [{
                functionDeclarations: [{
                    name: 'search_internet',
                    description: 'Search the internet for current information when needed',
                    parameters: {
                        type: 'object',
                        properties: {
                            query: {
                                type: 'string',
                                description: 'The search query to look up'
                            }
                        },
                        required: ['query']
                    }
                }]
            }] : undefined
        };
    }
    
    async function callGeminiAPI(apiKey, prompt) {
        if (!apiKey) {
            throw new Error('API key not set');
        }

        const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(prompt)
        });
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error.message || 'API request failed');
        }
        
        const result = await response.json();
        
        // Handle tool calls (for internet access)
        if (result.candidates && result.candidates[0].content.parts[0].functionCall && internetAccess.checked) {
            const functionCall = result.candidates[0].content.parts[0].functionCall;
            
            if (functionCall.name === 'search_internet') {
                const searchQuery = functionCall.args.query;
                const searchResults = await performWebSearch(searchQuery);
                
                // Call the API again with the search results
                const followUpPrompt = {
                    contents: [
                        ...prompt.contents,
                        {
                            role: 'model',
                            parts: [{
                                functionCall: {
                                    name: functionCall.name,
                                    args: functionCall.args
                                }
                            }]
                        },
                        {
                            role: 'function',
                            parts: [{
                                functionResponse: {
                                    name: functionCall.name,
                                    response: searchResults
                                }
                            }]
                        }
                    ],
                    generationConfig: prompt.generationConfig
                };
                
                return callGeminiAPI(apiKey, followUpPrompt);
            }
        }
        
        return result;
    }
    
    async function performWebSearch(query) {
        try {
            // In a real implementation, you would replace this with an actual search API call
            // For now, we'll simulate a search with a timeout
            await new Promise(resolve => setTimeout(resolve, 1000));
            
            // Return simulated search results
            return {
                results: [
                    {
                        title: `Search results for: ${query}`,
                        snippet: `Here's some information about ${query} from the web. In a real implementation, this would be actual search results.`,
                        link: "https://example.com/search"
                    }
                ]
            };
        } catch (error) {
            console.error('Search error:', error);
            return {
                error: "I couldn't retrieve current information. Please try again later."
            };
        }
    }
    
    function addMessageToChat(role, content, timestamp = new Date().toISOString(), files = [], rawContent = null) {
        const messageElement = document.createElement('div');
        messageElement.className = `message message-${role}`;
        
        // Generate message ID for action buttons
        const messageId = generateId();
        messageElement.dataset.messageId = messageId;
        messageElement.dataset.timestamp = timestamp;
        
        // Format content with HTML tags if it's plain text
        let formattedContent = content;
        if (!content.startsWith('<')) {
            formattedContent = formatTextWithHTML(content);
        }
        
        // Base message HTML
        messageElement.innerHTML = `
            <div class="message-content">${formattedContent}</div>
            <div class="message-info">
                <span class="message-time">${formatTime(timestamp)}</span>
                ${role === 'user' ? '<span class="material-icons message-status">done_all</span>' : ''}
            </div>
        `;
        
        // Add action buttons for all messages
        const actionButtons = document.createElement('div');
        actionButtons.className = 'message-actions';
        
        if (role === 'agent') {
            actionButtons.innerHTML = `
                <button class="message-action-btn copy-btn" title="Copy message">
                    <span class="material-icons">content_copy</span>
                </button>
                <button class="message-action-btn delete-btn" title="Delete message">
                    <span class="material-icons">delete</span>
                </button>
                <button class="message-action-btn regenerate-btn" title="Regenerate response">
                    <span class="material-icons">refresh</span>
                </button>
            `;
        } else {
            actionButtons.innerHTML = `
                <button class="message-action-btn copy-btn" title="Copy message">
                    <span class="material-icons">content_copy</span>
                </button>
                <button class="message-action-btn delete-btn" title="Delete message">
                    <span class="material-icons">delete</span>
                </button>
            `;
        }
        
        // Insert buttons before message info
        messageElement.insertBefore(actionButtons, messageElement.querySelector('.message-info'));
        
        // Add event listeners for the buttons
        const copyBtn = actionButtons.querySelector('.copy-btn');
        const deleteBtn = actionButtons.querySelector('.delete-btn');
        
        copyBtn.addEventListener('click', () => copyMessage(content));
        deleteBtn.addEventListener('click', () => deleteMessage(messageId));
        
        if (role === 'agent') {
            const regenerateBtn = actionButtons.querySelector('.regenerate-btn');
            regenerateBtn.addEventListener('click', () => regenerateMessage(messageId, rawContent || content));
        }
        
        // Add file attachments if any
        if (files && files.length > 0) {
            files.forEach(file => {
                const fileElement = document.createElement('div');
                fileElement.className = 'file-message';
                fileElement.innerHTML = `
                    <span class="material-icons file-icon">insert_drive_file</span>
                    <div class="file-info">
                        <div class="file-name">${file.name}</div>
                        <div class="file-size">${formatFileSize(file.size)}</div>
                    </div>
                `;
                messageElement.insertBefore(fileElement, messageElement.querySelector('.message-info'));
            });
        }
        
        chatMessages.appendChild(messageElement);
        
        // Scroll to bottom
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
    
    function copyMessage(content) {
        navigator.clipboard.writeText(content).then(() => {
            showToast('Message copied to clipboard');
        }).catch(err => {
            console.error('Failed to copy message: ', err);
            showToast('Failed to copy message');
        });
    }
    
    function deleteMessage(messageId) {
        const messageElement = document.querySelector(`[data-message-id="${messageId}"]`);
        if (messageElement) {
            messageElement.remove();
            
            // Also remove from conversation history
            const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
            if (conversations[currentConversationId]) {
                conversations[currentConversationId] = conversations[currentConversationId].filter(
                    msg => msg.timestamp !== messageElement.dataset.timestamp
                );
                localStorage.setItem('conversations', JSON.stringify(conversations));
            }
        }
    }
    
    function regenerateMessage(messageId, originalMessage) {
        const messageElement = document.querySelector(`[data-message-id="${messageId}"]`);
        if (!messageElement) return;
        
        // Remove the old message
        messageElement.remove();
        
        // Remove from conversation history
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        if (conversations[currentConversationId]) {
            conversations[currentConversationId] = conversations[currentConversationId].filter(
                msg => msg.content !== originalMessage
            );
            localStorage.setItem('conversations', JSON.stringify(conversations));
        }
        
        // Resend the last user message
        const lastUserMessage = getLastUserMessage();
        if (lastUserMessage) {
            sendMessageForRegeneration(lastUserMessage);
        }
    }
    
    function getLastUserMessage() {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        if (conversations[currentConversationId]) {
            const userMessages = conversations[currentConversationId].filter(
                msg => msg.role === 'user'
            );
            return userMessages.length > 0 ? userMessages[userMessages.length - 1].content : null;
        }
        return null;
    }
    
    function sendMessageForRegeneration(message) {
        if (!apiKey || isTyping) return;
        
        // Show typing indicator
        showTypingIndicator();
        
        // Get current agent
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === currentAgentId);
        
        if (!agent) return;
        
        // Prepare the prompt with personality and conversation history
        const prompt = preparePrompt(agent, message);
        
        // Call Gemini API
        callGeminiAPI(agent.apiKey, prompt)
            .then(response => {
                removeTypingIndicator();
                
                if (response && response.candidates && response.candidates[0].content.parts[0].text) {
                    const aiResponse = response.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', aiResponse, null, [], response.candidates[0].content.parts[0].text);
                    saveMessageToHistory('agent', aiResponse);
                } else {
                    addMessageToChat('agent', "I'm sorry, I couldn't process your request. Please try again.");
                }
            })
            .catch(error => {
                removeTypingIndicator();
                console.error('API Error:', error);
                addMessageToChat('agent', "I'm experiencing some technical difficulties. Please try again later.");
            });
    }
    
    function formatTextWithHTML(text) {
        // Convert markdown-like formatting to HTML
        let html = text
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>') // bold
            .replace(/\*(.*?)\*/g, '<em>$1</em>') // italic
            .replace(/_(.*?)_/g, '<u>$1</u>') // underline
            .replace(/~~(.*?)~~/g, '<s>$1</s>') // strikethrough
            .replace(/`(.*?)`/g, '<code>$1</code>') // inline code
            .replace(/```([\s\S]*?)```/g, '<pre>$1</pre>') // code block
            .replace(/\[(.*?)\]\((.*?)\)/g, '<a href="$2" target="_blank">$1</a>'); // links
        
        // Convert headings (### Heading -> <h3>Heading</h3>)
        html = html.replace(/^### (.*$)/gm, '<h3>$1</h3>')
                  .replace(/^## (.*$)/gm, '<h2>$1</h2>')
                  .replace(/^# (.*$)/gm, '<h1>$1</h1>');
        
        // Convert lists
        html = html.replace(/^\* (.*$)/gm, '<li>$1</li>');
        html = html.replace(/<li>.*<\/li>/g, function(match) {
            return '<ul>' + match + '</ul>';
        });
        
        // Convert line breaks to paragraphs when double line breaks
        html = html.replace(/\n\n/g, '</p><p>');
        html = '<p>' + html + '</p>';
        
        return html;
    }
    
    function saveMessageToHistory(role, content, files = []) {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        
        if (!conversations[currentConversationId]) {
            conversations[currentConversationId] = [];
        }
        
        conversations[currentConversationId].push({
            role,
            content,
            timestamp: new Date().toISOString(),
            files
        });
        
        localStorage.setItem('conversations', JSON.stringify(conversations));
    }
    
    function showTypingIndicator() {
        if (isTyping) return;
        
        isTyping = true;
        const typingElement = document.createElement('div');
        typingElement.className = 'message message-agent typing-indicator';
        typingElement.id = 'typingIndicator';
        typingElement.innerHTML = `
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
        `;
        
        chatMessages.appendChild(typingElement);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
    
    function removeTypingIndicator() {
        isTyping = false;
        const typingElement = document.getElementById('typingIndicator');
        if (typingElement) {
            typingElement.remove();
        }
    }
    
    function handleImageUpload() {
        if (fileInput.files.length === 0) return;
        
        const file = fileInput.files[0];
        
        // Validate image type
        if (!file.type.startsWith('image/')) {
            alert('Please upload an image file only (JPG, PNG, BMP, etc.).');
            fileInput.value = '';
            return;
        }
        
        // Display the uploaded image
        const reader = new FileReader();
        reader.onload = function(e) {
            const img = new Image();
            img.src = e.target.result;
            
            // Add image message to chat
            addMessageToChat('user', `I've uploaded an image:`, new Date().toISOString(), [file]);
            saveMessageToHistory('user', `I've uploaded an image:`, [file]);
            
            // Analyze the image
            analyzeImage(file, e.target.result);
        };
        reader.readAsDataURL(file);
    }
    
    function analyzeImage(file, imageData) {
        showTypingIndicator();
        
        // Prepare the prompt for image analysis
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === currentAgentId);
        
        const prompt = {
            contents: [{
                role: 'user',
                parts: [
                    { text: `Please analyze this image thoroughly and provide detailed information about its content.` },
                    { 
                        inlineData: {
                            mimeType: file.type,
                            data: imageData.split(',')[1] // Remove the data URL prefix
                        }
                    }
                ]
            }],
            generationConfig: {
                temperature: 0.7,
                maxOutputTokens: 2000
            }
        };
        
        // Call Gemini API
        callGeminiAPI(agent.apiKey, prompt)
            .then(response => {
                removeTypingIndicator();
                
                if (response && response.candidates && response.candidates[0].content.parts[0].text) {
                    const analysis = response.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', analysis);
                    saveMessageToHistory('agent', analysis);
                } else {
                    addMessageToChat('agent', "I've received your image. Let me know how you'd like me to assist you with it.");
                }
            })
            .catch(error => {
                removeTypingIndicator();
                console.error('Image analysis error:', error);
                addMessageToChat('agent', "I've received your image. Let me know how you'd like me to assist you with it.");
            });
    }
    
    function confirmDeleteConversation() {
        pendingAction = 'deleteConversation';
        confirmMessage.textContent = 'Are you sure you want to delete this conversation? This cannot be undone.';
        openModal(confirmModal);
    }
    
    function exportConversation() {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        if (conversation.length === 0) {
            alert('No conversation to export');
            return;
        }
        
        // Format as CSV
        let csv = 'Role,Content,Timestamp\n';
        conversation.forEach(msg => {
            csv += `"${msg.role}","${msg.content.replace(/"/g, '""')}","${msg.timestamp}"\n`;
        });
        
        // Create download link
        const blob = new Blob([csv], { type: 'text/csv' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `VineAI_Conversation_${currentAgentName.textContent}_${formatDateForFilename(new Date())}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }
    
    function executePendingAction() {
        switch (pendingAction) {
            case 'deleteConversation':
                deleteConversation();
                break;
            case 'deleteAgent':
                deleteAgent(pendingActionData);
                break;
        }
        
        closeModal(confirmModal);
        pendingAction = null;
        pendingActionData = null;
    }
    
    function deleteConversation() {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        delete conversations[currentConversationId];
        localStorage.setItem('conversations', JSON.stringify(conversations));
        
        // Clear chat
        chatMessages.innerHTML = '';
    }
    
    function deleteAgent(agentId) {
        const agents = JSON.parse(localStorage.getItem('agents'));
        const updatedAgents = agents.filter(a => a.id !== agentId);
        
        localStorage.setItem('agents', JSON.stringify(updatedAgents));
        
        // Delete conversation history
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        delete conversations[`conv_${agentId}`];
        localStorage.setItem('conversations', JSON.stringify(conversations));
        
        // If deleting current agent, switch to another one
        if (agentId === currentAgentId) {
            if (updatedAgents.length > 0) {
                switchAgent(updatedAgents[0].id);
            } else {
                createDefaultAgent();
            }
        }
        
        loadAgents();
    }
    
    function changeTheme() {
        const theme = themeSelect.value;
        document.body.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
    }
    
    function openModal(modal) {
        modal.classList.add('active');
    }
    
    function closeModal(modal) {
        modal.classList.remove('active');
    }
    
    function showEmojiPicker() {
        // Create emoji picker container
        const emojiPicker = document.createElement('div');
        emojiPicker.className = 'emoji-picker';
        emojiPicker.innerHTML = `
            <div class="emoji-picker-header">
                <span>Select Emoji</span>
                <button class="close-emoji-picker">
                    <span class="material-icons">close</span>
                </button>
            </div>
            <div class="emoji-categories">
                <button class="emoji-category-btn active" data-category="smileys">😀</button>
                <button class="emoji-category-btn" data-category="animals">🐻</button>
                <button class="emoji-category-btn" data-category="food">🍎</button>
                <button class="emoji-category-btn" data-category="travel">✈️</button>
                <button class="emoji-category-btn" data-category="objects">💡</button>
                <button class="emoji-category-btn" data-category="symbols">❤️</button>
                <button class="emoji-category-btn" data-category="flags">🏳️</button>
            </div>
            <div class="emoji-grid" id="emojiGrid">
                <!-- Emojis will be loaded here -->
            </div>
        `;
        
        // Position the picker near the emoji button
        const rect = emojiPickerBtn.getBoundingClientRect();
        emojiPicker.style.position = 'absolute';
        emojiPicker.style.bottom = `${window.innerHeight - rect.top + 10}px`;
        emojiPicker.style.left = `${rect.left}px`;
        emojiPicker.style.zIndex = '1000';
        
        document.body.appendChild(emojiPicker);
        
        // Load emojis for the default category
        loadEmojis('smileys');
        
        // Add event listeners
        document.querySelector('.close-emoji-picker').addEventListener('click', () => {
            emojiPicker.remove();
        });
        
        document.querySelectorAll('.emoji-category-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                document.querySelectorAll('.emoji-category-btn').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                loadEmojis(this.dataset.category);
            });
        });
    }
    
    function loadEmojis(category) {
        const emojiGrid = document.getElementById('emojiGrid');
        if (!emojiGrid) return;
        
        emojiGrid.innerHTML = '';
        
        // Emoji data by category
        const emojiCategories = {
            smileys: ['😀', '😃', '😄', '😁', '😆', '😅', '😂', '🤣', '😊', '😇', '🙂', '🙃', '😉', '😌', '😍', '🥰', '😘', '😗', '😙', '😚', '😋', '😛', '😝', '😜', '🤪', '🤨', '🧐', '🤓', '😎', '🥸', '🤩', '🥳', '😏', '😒', '😞', '😔', '😟', '😕', '🙁', '☹️', '😣', '😖', '😫', '😩', '🥺', '😢', '😭', '😤', '😠', '😡', '🤬', '🤯', '😳', '🥵', '🥶', '😱', '😨', '😰', '😥', '😓', '🤗', '🤔', '🤭', '🤫', '🤥', '😶', '😐', '😑', '😬', '🙄', '😯', '😦', '😧', '😮', '😲', '🥱', '😴', '🤤', '😪', '😵', '🤐', '🥴', '🤢', '🤮', '🤧', '😷', '🤒', '🤕', '🤑', '🤠', '😈', '👿', '👹', '👺', '🤡', '💩', '👻', '💀', '☠️', '👽', '👾', '🤖', '🎃', '😺', '😸', '😹', '😻', '😼', '😽', '🙀', '😿', '😾'],
            animals: ['🐶', '🐱', '🐭', '🐹', '🐰', '🦊', '🐻', '🐼', '🐨', '🐯', '🦁', '🐮', '🐷', '🐽', '🐸', '🐵', '🙈', '🙉', '🙊', '🐒', '🐔', '🐧', '🐦', '🐤', '🐣', '🐥', '🦆', '🦅', '🦉', '🦇', '🐺', '🐗', '🐴', '🦄', '🐝', '🐛', '🦋', '🐌', '🐞', '🐜', '🦟', '🦗', '🕷', '🦂', '🐢', '🐍', '🦎', '🦖', '🦕', '🐙', '🦑', '🦐', '🦞', '🦀', '🐡', '🐠', '🐟', '🐬', '🐳', '🐋', '🦈', '🐊', '🐅', '🐆', '🦓', '🦍', '🦧', '🦣', '🐘', '🦛', '🦏', '🐪', '🐫', '🦒', '🦘', '🦬', '🐃', '🐂', '🐄', '🐎', '🐖', '🐏', '🐑', '🦙', '🐐', '🦌', '🐕', '🐩', '🦮', '🐕‍🦺', '🐈', '🐈‍⬛', '🪶', '🐓', '🦃', '🦤', '🦚', '🦜', '🦢', '🦩', '🕊', '🐇', '🦝', '🦨', '🦡', '🦫', '🦦', '🦥', '🐁', '🐀', '🐿', '🦔'],
            food: ['🍏', '🍎', '🍐', '🍊', '🍋', '🍌', '🍉', '🍇', '🍓', '🫐', '🍈', '🍒', '🍑', '🥭', '🍍', '🥥', '🥝', '🍅', '🍆', '🥑', '🥦', '🥬', '🥒', '🌶', '🫑', '🌽', '🥕', '🫒', '🧄', '🧅', '🥔', '🍠', '🫚', '🫛', '🥐', '🥖', '🫓', '🥨', '🥯', '🥞', '🧇', '🧀', '🍖', '🍗', '🥩', '🥓', '🍔', '🍟', '🍕', '🌭', '🥪', '🌮', '🌯', '🫔', '🥙', '🧆', '🥚', '🍳', '🥘', '🍲', '🫕', '🥣', '🥗', '🍿', '🧈', '🧂', '🥫', '🍱', '🍘', '🍙', '🍚', '🍛', '🍜', '🍝', '🍠', '🍢', '🍣', '🍤', '🍥', '🥮', '🍡', '🥟', '🥠', '🥡', '🦀', '🦞', '🦐', '🦑', '🍦', '🍧', '🍨', '🍩', '🍪', '🎂', '🍰', '🧁', '🥧', '🍫', '🍬', '🍭', '🍮', '🍯', '🍼', '🥛', '☕', '🫖', '🍵', '🍶', '🍾', '🍷', '🍸', '🍹', '🍺', '🍻', '🥂', '🥃', '🥤', '🧋', '🧃', '🧉', '🧊', '🥢', '🍽', '🍴', '🥄'],
            travel: ['🚗', '🚕', '🚙', '🚌', '🚎', '🏎', '🚓', '🚑', '🚒', '🚐', '🛻', '🚚', '🚛', '🚜', '🦯', '🦽', '🦼', '🛴', '🚲', '🛵', '🏍', '🛺', '🚨', '🚔', '🚍', '🚘', '🚖', '🚡', '🚠', '🚟', '🚃', '🚋', '🚞', '🚝', '🚄', '🚅', '🚈', '🚂', '🚆', '🚇', '🚊', '🚉', '✈️', '🛫', '🛬', '🛩', '💺', '🛰', '🚀', '🛸', '🚁', '🛶', '⛵️', '🚤', '🛥', '🛳', '⛴', '🚢', '⚓️', '🪝', '⛽️', '🚧', '🚦', '🚥', '🚏', '🗺', '🗿', '🗽', '🗼', '🏰', '🏯', '🏟', '🎡', '🎢', '🎠', '⛲️', '⛱', '🏖', '🏝', '🏜', '🌋', '⛰', '🏔', '🗻', '🏕', '⛺️', '🛖', '🏠', '🏡', '🏘', '🏚', '🏗', '🏭', '🏢', '🏬', '🏣', '🏤', '🏥', '🏦', '🏨', '🏪', '🏫', '🏩', '💒', '🏛', '⛪️', '🕌', '🛕', '🕍', '⛩', '🕋', '⛲️', '⛺️', '🌁', '🌃', '🏙', '🌄', '🌅', '🌆', '🌇', '🌉', '🎠', '🎡', '🎢'],
            objects: ['⌚️', '📱', '📲', '💻', '⌨️', '🖥', '🖨', '🖱', '🖲', '🕹', '🗜', '💽', '💾', '💿', '📀', '📼', '📷', '📸', '📹', '🎥', '📽', '🎞', '📞', '☎️', '📟', '📠', '📺', '📻', '🎙', '🎚', '🎛', '🧭', '⏱', '⏲', '⏰', '🕰', '⌛️', '⏳', '📡', '🔋', '🔌', '💡', '🔦', '🕯', '🪔', '🧯', '🛢', '💸', '💵', '💴', '💶', '💷', '💰', '💳', '💎', '⚖️', '🧰', '🔧', '🔨', '⚒', '🛠', '⛏', '🔩', '⚙️', '🧱', '⛓', '🧲', '🔫', '💣', '🧨', '🪓', '🔪', '🗡', '⚔️', '🛡', '🚬', '⚰️', '🪦', '⚱️', '🏺', '🔮', '📿', '🧿', '💈', '⚗️', '🔭', '🔬', '🕳', '🩹', '🩺', '💊', '💉', '🩸', '🧬', '🦠', '🧫', '🧪', '🌡', '🧹', '🧺', '🧻', '🚽', '🪠', '🚿', '🛁', '🪤', '🪒', '🧴', '🧷', '🧹', '🧺', '🧻', '🧼', '🧽', '🧯', '🛒'],
            symbols: ['❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍', '🤎', '💔', '❣️', '💕', '💞', '💓', '💗', '💖', '💘', '💝', '💟', '☮️', '✝️', '☪️', '🕉', '☸️', '✡️', '🔯', '🕎', '☯️', '☦️', '🛐', '⛎', '♈️', '♉️', '♊️', '♋️', '♌️', '♍️', '♎️', '♏️', '♐️', '♑️', '♒️', '♓️', '🆔', '⚛️', '🉑', '☢️', '☣️', '📴', '📳', '🈶', '🈚️', '🈸', '🈺', '🈷️', '✴️', '🆚', '💮', '🉐', '㊙️', '㊗️', '🈴', '🈵', '🈹', '🈲', '🅰️', '🅱️', '🆎', '🆑', '🅾️', '🆘', '❌', '⭕️', '🛑', '⛔️', '📛', '🚫', '💯', '💢', '♨️', '🚷', '🚯', '🚳', '🚱', '🔞', '📵', '🚭', '❗️', '❕', '❓', '❔', '‼️', '⁉️', '🔅', '🔆', '〽️', '⚠️', '🚸', '🔱', '⚜️', '🔰', '♻️', '✅', '🈯️', '💹', '❇️', '✳️', '❎', '🌐', '💠', 'Ⓜ️', '🌀', '💤', '🏧', '🚾', '♿️', '🅿️', '🈳', '🈂️', '🛂', '🛃', '🛄', '🛅', '🚹', '🚺', '🚼', '🚻', '🚮', '🎦', '📶', '🈁', '🔣', 'ℹ️', '🔤', '🔡', '🔠', '🆖', '🆗', '🆙', '🆒', '🆕', '🆓', '0️⃣', '1️⃣', '2️⃣', '3️⃣', '4️⃣', '5️⃣', '6️⃣', '7️⃣', '8️⃣', '9️⃣', '🔟', '🔢', '#️⃣', '*️⃣', '⏏️', '▶️', '⏸', '⏯', '⏹', '⏺', '⏭', '⏮', '⏩', '⏪', '⏫', '⏬', '◀️', '🔼', '🔽', '➡️', '⬅️', '⬆️', '⬇️', '↗️', '↘️', '↙️', '↖️', '↕️', '↔️', '↪️', '↩️', '⤴️', '⤵️', '🔀', '🔁', '🔂', '🔄', '🔃', '🎵', '🎶', '➕', '➖', '➗', '✖️', '♾', '💲', '💱', '™️', '©️', '®️', '〰️', '➰', '➿', '🔚', '🔙', '🔛', '🔝', '🔜'],
            flags: ['🏳️', '🏴', '🏁', '🚩', '🏳️‍🌈', '🏴‍☠️', '🇦🇫', '🇦🇽', '🇦🇱', '🇩🇿', '🇦🇸', '🇦🇩', '🇦🇴', '🇦🇮', '🇦🇶', '🇦🇬', '🇦🇷', '🇦🇲', '🇦🇼', '🇦🇺', '🇦🇹', '🇦🇿', '🇧🇸', '🇧🇭', '🇧🇩', '🇧🇧', '🇧🇾', '🇧🇪', '🇧🇿', '🇧🇯', '🇧🇲', '🇧🇹', '🇧🇴', '🇧🇦', '🇧🇼', '🇧🇷', '🇮🇴', '🇻🇬', '🇧🇳', '🇧🇬', '🇧🇫', '🇧🇮', '🇨🇻', '🇰🇭', '🇨🇲', '🇨🇦', '🇮🇨', '🇰🇾', '🇨🇫', '🇹🇩', '🇨🇱', '🇨🇳', '🇨🇽', '🇨🇨', '🇨🇴', '🇰🇲', '🇨🇬', '🇨🇩', '🇨🇰', '🇨🇷', '🇨🇮', '🇭🇷', '🇨🇺', '🇨🇼', '🇨🇾', '🇨🇿', '🇩🇰', '🇩🇯', '🇩🇲', '🇩🇴', '🇪🇨', '🇪🇬', '🇸🇻', '🇬🇶', '🇪🇷', '🇪🇪', '🇪🇹', '🇪🇺', '🇫🇰', '🇫🇴', '🇫🇯', '🇫🇮', '🇫🇷', '🇬🇫', '🇵🇫', '🇹🇫', '🇬🇦', '🇬🇲', '🇬🇪', '🇩🇪', '🇬🇭', '🇬🇮', '🇬🇷', '🇬🇱', '🇬🇩', '🇬🇵', '🇬🇺', '🇬🇹', '🇬🇬', '🇬🇳', '🇬🇼', '🇬🇾', '🇭🇹', '🇭🇳', '🇭🇰', '🇭🇺', '🇮🇸', '🇮🇳', '🇮🇩', '🇮🇷', '🇮🇶', '🇮🇪', '🇮🇲', '🇮🇱', '🇮🇹', '🇯🇲', '🇯🇵', '🇯🇪', '🇯🇴', '🇰🇿', '🇰🇪', '🇰🇮', '🇽🇰', '🇰🇼', '🇰🇬', '🇱🇦', '🇱🇻', '🇱🇧', '🇱🇸', '🇱🇷', '🇱🇾', '🇱🇮', '🇱🇹', '🇱🇺', '🇲🇴', '🇲🇰', '🇲🇬', '🇲🇼', '🇲🇾', '🇲🇻', '🇲🇱', '🇲🇹', '🇲🇭', '🇲🇶', '🇲🇷', '🇲🇺', '🇾🇹', '🇲🇽', '🇫🇲', '🇲🇩', '🇲🇨', '🇲🇳', '🇲🇪', '🇲🇸', '🇲🇦', '🇲🇿', '🇲🇲', '🇳🇦', '🇳🇷', '🇳🇵', '🇳🇱', '🇳🇨', '🇳🇿', '🇳🇮', '🇳🇪', '🇳🇬', '🇳🇺', '🇳🇫', '🇰🇵', '🇲🇵', '🇳🇴', '🇴🇲', '🇵🇰', '🇵🇼', '🇵🇸', '🇵🇦', '🇵🇬', '🇵🇾', '🇵🇪', '🇵🇭', '🇵🇳', '🇵🇱', '🇵🇹', '🇵🇷', '🇶🇦', '🇷🇪', '🇷🇴', '🇷🇺', '🇷🇼', '🇼🇸', '🇸🇲', '🇸🇦', '🇸🇳', '🇷🇸', '🇸🇨', '🇸🇱', '🇸🇬', '🇸🇽', '🇸🇰', '🇸🇮', '🇬🇸', '🇸🇧', '🇸🇴', '🇿🇦', '🇰🇷', '🇸🇸', '🇪🇸', '🇱🇰', '🇧🇱', '🇸🇭', '🇰🇳', '🇱🇨', '🇵🇲', '🇻🇨', '🇸🇩', '🇸🇷', '🇸🇿', '🇸🇪', '🇨🇭', '🇸🇾', '🇹🇼', '🇹🇯', '🇹🇿', '🇹🇭', '🇹🇱', '🇹🇬', '🇹🇰', '🇹🇴', '🇹🇹', '🇹🇳', '🇹🇷', '🇹🇲', '🇹🇨', '🇹🇻', '🇻🇮', '🇺🇬', '🇺🇦', '🇦🇪', '🇬🇧', '🇺🇸', '🇺🇾', '🇺🇿', '🇻🇺', '🇻🇦', '🇻🇪', '🇻🇳', '🇼🇫', '🇪🇭', '🇾🇪', '🇿🇲', '🇿🇼']
        };
        
        // Add emojis to the grid
        emojiCategories[category].forEach(emoji => {
            const emojiBtn = document.createElement('button');
            emojiBtn.className = 'emoji-btn';
            emojiBtn.textContent = emoji;
            emojiBtn.addEventListener('click', () => {
                insertEmoji(emoji);
                emojiPicker.remove();
            });
            emojiGrid.appendChild(emojiBtn);
        });
    }
    
    function insertEmoji(emoji) {
        const cursorPos = messageInput.selectionStart;
        const textBefore = messageInput.value.substring(0, cursorPos);
        const textAfter = messageInput.value.substring(cursorPos);
        
        messageInput.value = textBefore + emoji + textAfter;
        messageInput.focus();
        messageInput.selectionStart = cursorPos + emoji.length;
        messageInput.selectionEnd = cursorPos + emoji.length;
        
        // Trigger input event to resize textarea
        const event = new Event('input');
        messageInput.dispatchEvent(event);
    }
    
    function showToast(message) {
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.textContent = message;
        document.body.appendChild(toast);
        
        setTimeout(() => {
            toast.classList.add('show');
        }, 10);
        
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => {
                toast.remove();
            }, 300);
        }, 3000);
    }
    
    // Helper functions
    function generateId() {
        return 'id_' + Math.random().toString(36).substr(2, 9);
    }
    
    function formatTime(timestamp) {
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    }
    
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    function formatDateForFilename(date) {
        return date.toISOString().split('T')[0].replace(/-/g, '');
    }
});