// Chart.js initialization and configuration for the Student Fees Management System

// Chart colors
const CHART_COLORS = {
    primary: '#667eea',
    secondary: '#764ba2',
    success: '#28a745',
    danger: '#dc3545',
    warning: '#ffc107',
    info: '#17a2b8',
    light: '#f8f9fa',
    dark: '#343a40'
};

// Chart default configuration
Chart.defaults.font.family = "'Roboto', sans-serif";
Chart.defaults.font.size = 14;
Chart.defaults.color = '#6c757d';
Chart.defaults.plugins.tooltip.backgroundColor = 'rgba(0, 0, 0, 0.7)';
Chart.defaults.plugins.legend.labels.usePointStyle = true;

// Initialize all charts on the page
function initCharts() {
    // Collection by month chart
    initCollectionChart();
    
    // Fee distribution chart
    initFeeDistributionChart();
    
    // Payment methods chart
    initPaymentMethodsChart();
    
    // Student status chart
    initStudentStatusChart();
}

// Collection by month chart
function initCollectionChart() {
    const ctx = document.getElementById('collectionChart');
    if (!ctx) return;
    
    // Sample data - in a real application, this would come from an API
    const data = {
        labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
        datasets: [{
            label: 'Collection (₦)',
            data: [125000, 150000, 175000, 140000, 160000, 195000, 210000, 185000, 220000, 240000, 230000, 260000],
            backgroundColor: CHART_COLORS.primary,
            borderColor: CHART_COLORS.primary,
            borderWidth: 2,
            borderRadius: 5,
            borderSkipped: false,
        }]
    };
    
    const config = {
        type: 'bar',
        data: data,
        options: {
            responsive: true,
            plugins: {
                title: {
                    display: true,
                    text: 'Monthly Fee Collection',
                    font: {
                        size: 16,
                        weight: 'bold'
                    }
                },
                legend: {
                    position: 'top',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `₦${context.raw.toLocaleString()}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return '₦' + value.toLocaleString();
                        }
                    }
                }
            }
        }
    };
    
    new Chart(ctx, config);
}

// Fee distribution chart
function initFeeDistributionChart() {
    const ctx = document.getElementById('feeDistributionChart');
    if (!ctx) return;
    
    const data = {
        labels: ['Tuition', 'Examination', 'Sports', 'Library', 'Laboratory', 'Other'],
        datasets: [{
            data: [65, 15, 8, 5, 4, 3],
            backgroundColor: [
                CHART_COLORS.primary,
                CHART_COLORS.secondary,
                CHART_COLORS.success,
                CHART_COLORS.info,
                CHART_COLORS.warning,
                CHART_COLORS.danger
            ],
            borderWidth: 1
        }]
    };
    
    const config = {
        type: 'doughnut',
        data: data,
        options: {
            responsive: true,
            plugins: {
                title: {
                    display: true,
                    text: 'Fee Distribution',
                    font: {
                        size: 16,
                        weight: 'bold'
                    }
                },
                legend: {
                    position: 'right',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `${context.label}: ${context.raw}%`;
                        }
                    }
                }
            }
        }
    };
    
    new Chart(ctx, config);
}

// Payment methods chart
function initPaymentMethodsChart() {
    const ctx = document.getElementById('paymentMethodsChart');
    if (!ctx) return;
    
    const data = {
        labels: ['Bank Transfer', 'Cash', 'Card', 'Online'],
        datasets: [{
            data: [45, 30, 15, 10],
            backgroundColor: [
                CHART_COLORS.primary,
                CHART_COLORS.secondary,
                CHART_COLORS.success,
                CHART_COLORS.info
            ],
            borderWidth: 1
        }]
    };
    
    const config = {
        type: 'pie',
        data: data,
        options: {
            responsive: true,
            plugins: {
                title: {
                    display: true,
                    text: 'Payment Methods',
                    font: {
                        size: 16,
                        weight: 'bold'
                    }
                },
                legend: {
                    position: 'right',
                },
                tooltip: {
                    callbacks: {
                        label: function(context) {
                            return `${context.label}: ${context.raw}%`;
                        }
                    }
                }
            }
        }
    };
    
    new Chart(ctx, config);
}

// Student status chart
function initStudentStatusChart() {
    const ctx = document.getElementById('studentStatusChart');
    if (!ctx) return;
    
    const data = {
        labels: ['Active', 'Inactive', 'Graduated', 'Transferred'],
        datasets: [{
            label: 'Students',
            data: [320, 25, 180, 15],
            backgroundColor: [
                CHART_COLORS.success,
                CHART_COLORS.danger,
                CHART_COLORS.primary,
                CHART_COLORS.warning
            ],
            borderWidth: 1
        }]
    };
    
    const config = {
        type: 'bar',
        data: data,
        options: {
            indexAxis: 'y',
            responsive: true,
            plugins: {
                title: {
                    display: true,
                    text: 'Student Status',
                    font: {
                        size: 16,
                        weight: 'bold'
                    }
                },
                legend: {
                    display: false
                }
            }
        }
    };
    
    new Chart(ctx, config);
}

// Dynamic chart for reports
function createDynamicChart(ctx, type, data, options = {}) {
    const defaultOptions = {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'top',
            }
        }
    };
    
    const mergedOptions = {...defaultOptions, ...options};
    
    return new Chart(ctx, {
        type: type,
        data: data,
        options: mergedOptions
    });
}

// Update chart with new data
function updateChart(chart, newData) {
    chart.data = newData;
    chart.update();
}

// Export chart as image
function exportChartAsImage(chart, filename = 'chart') {
    const imageLink = document.createElement('a');
    imageLink.download = `${filename}.png`;
    imageLink.href = chart.toBase64Image();
    imageLink.click();
}

// Generate random colors for dynamic datasets
function generateColors(count) {
    const colors = [];
    for (let i = 0; i < count; i++) {
        const hue = (i * 137.5) % 360; // Golden angle approximation
        colors.push(`hsl(${hue}, 70%, 65%)`);
    }
    return colors;
}

// Initialize charts when the page loads
document.addEventListener('DOMContentLoaded', function() {
    initCharts();
    
    // Add resize event listener to redraw charts on window resize
    window.addEventListener('resize', function() {
        document.querySelectorAll('canvas').forEach(canvas => {
            const chart = Chart.getChart(canvas);
            if (chart) {
                chart.resize();
            }
        });
    });
});

// Make functions available globally
window.initCharts = initCharts;
window.createDynamicChart = createDynamicChart;
window.updateChart = updateChart;
window.exportChartAsImage = exportChartAsImage;