<?php
require_once "config.php";
check_auth();
check_role(['admin', 'accountant']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_fee_category'])) {
        // Add new fee category
        $name = trim($_POST['name']);
        $amount = trim($_POST['amount']);
        $description = trim($_POST['description']);
        $frequency = trim($_POST['frequency']);
        
        $sql = "INSERT INTO fee_categories (name, amount, description, frequency) 
                VALUES (:name, :amount, :description, :frequency)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':frequency', $frequency);
        
        if ($stmt->execute()) {
            $success = "Fee category added successfully!";
        } else {
            $error = "Error adding fee category. Please try again.";
        }
    }
    
    if (isset($_POST['allocate_fee'])) {
        // Allocate fee to student
        $student_id = $_POST['student_id'];
        $fee_category_id = $_POST['fee_category_id'];
        $amount = trim($_POST['amount']);
        $due_date = trim($_POST['due_date']);
        
        $sql = "INSERT INTO fee_allocations (student_id, fee_category_id, amount, due_date) 
                VALUES (:student_id, :fee_category_id, :amount, :due_date)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':student_id', $student_id);
        $stmt->bindParam(':fee_category_id', $fee_category_id);
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':due_date', $due_date);
        
        if ($stmt->execute()) {
            $success = "Fee allocated to student successfully!";
        } else {
            $error = "Error allocating fee. Please try again.";
        }
    }
}

// Handle delete request
if (isset($_GET['delete_category'])) {
    $id = $_GET['delete_category'];
    
    // Check if category has allocations before deleting
    $check_sql = "SELECT COUNT(*) FROM fee_allocations WHERE fee_category_id = :id";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->bindParam(':id', $id);
    $check_stmt->execute();
    $allocation_count = $check_stmt->fetchColumn();
    
    if ($allocation_count > 0) {
        $error = "Cannot delete fee category. There are allocations associated with this category.";
    } else {
        $sql = "DELETE FROM fee_categories WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            $success = "Fee category deleted successfully!";
        } else {
            $error = "Error deleting fee category. Please try again.";
        }
    }
}

// Get all fee categories
$fee_categories = $pdo->query("SELECT * FROM fee_categories ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);

// Get all students for dropdown
$students = $pdo->query("SELECT id, first_name, last_name, student_id FROM students WHERE status = 'active' ORDER BY first_name")->fetchAll(PDO::FETCH_ASSOC);

// Get fee allocations with student and category details
$allocations = $pdo->query("
    SELECT fa.*, s.first_name, s.last_name, s.student_id, fc.name as fee_name 
    FROM fee_allocations fa 
    JOIN students s ON fa.student_id = s.id 
    JOIN fee_categories fc ON fa.fee_category_id = fc.id
    ORDER BY fa.due_date DESC
")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fees Management - Student Fees Management System</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="assets/icons/material-icons.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="main-content">
        <div class="page-header">
            <h1>Fees Management</h1>
            <div class="breadcrumb">
                <a href="#"><i class="material-icons">home</i> Home</a>
                <span class="divider">/</span>
                <a class="active">Fees</a>
            </div>
        </div>
        
        <?php if (isset($success)): ?>
        <div class="alert alert-success">
            <i class="material-icons">check_circle</i>
            <?php echo $success; ?>
        </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
        <div class="alert alert-error">
            <i class="material-icons">error</i>
            <?php echo $error; ?>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-header">
                <h2>Add Fee Category</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="form-grid">
                    <div class="form-group">
                        <label for="name">Fee Name</label>
                        <input type="text" id="name" name="name" required>
                    </div>
                    <div class="form-group">
                        <label for="amount">Amount (<?php echo CURRENCY; ?>)</label>
                        <input type="number" id="amount" name="amount" step="0.01" required>
                    </div>
                    <div class="form-group">
                        <label for="frequency">Frequency</label>
                        <select id="frequency" name="frequency" required>
                            <option value="one-time">One Time</option>
                            <option value="monthly">Monthly</option>
                            <option value="quarterly">Quarterly</option>
                            <option value="yearly">Yearly</option>
                        </select>
                    </div>
                    <div class="form-group full-width">
                        <label for="description">Description</label>
                        <textarea id="description" name="description" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="add_fee_category" class="btn btn-primary">Add Fee Category</button>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Allocate Fee to Student</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="form-grid">
                    <div class="form-group">
                        <label for="student_id">Student</label>
                        <select id="student_id" name="student_id" required>
                            <option value="">Select Student</option>
                            <?php foreach($students as $student): ?>
                            <option value="<?php echo $student['id']; ?>">
                                <?php echo $student['first_name'] . ' ' . $student['last_name'] . ' (' . $student['student_id'] . ')'; ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="fee_category_id">Fee Category</label>
                        <select id="fee_category_id" name="fee_category_id" required>
                            <option value="">Select Fee Category</option>
                            <?php foreach($fee_categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>" data-amount="<?php echo $category['amount']; ?>">
                                <?php echo $category['name'] . ' (' . CURRENCY . number_format($category['amount'], 2) . ')'; ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="amount">Amount (<?php echo CURRENCY; ?>)</label>
                        <input type="number" id="amount" name="amount" step="0.01" required>
                    </div>
                    <div class="form-group">
                        <label for="due_date">Due Date</label>
                        <input type="date" id="due_date" name="due_date" required>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="allocate_fee" class="btn btn-primary">Allocate Fee</button>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Fee Categories</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Amount</th>
                                <th>Frequency</th>
                                <th>Description</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach($fee_categories as $category): ?>
                            <tr>
                                <td><?php echo $category['name']; ?></td>
                                <td><?php echo CURRENCY . number_format($category['amount'], 2); ?></td>
                                <td><?php echo ucfirst($category['frequency']); ?></td>
                                <td><?php echo $category['description']; ?></td>
                                <td><span class="status-badge status-<?php echo $category['status']; ?>"><?php echo ucfirst($category['status']); ?></span></td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="fees.php?delete_category=<?php echo $category['id']; ?>" class="btn-icon" onclick="return confirm('Are you sure you want to delete this fee category?')">
                                            <i class="material-icons">delete</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Fee Allocations</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Fee Type</th>
                                <th>Amount</th>
                                <th>Due Date</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach($allocations as $allocation): ?>
                            <tr>
                                <td><?php echo $allocation['first_name'] . ' ' . $allocation['last_name']; ?></td>
                                <td><?php echo $allocation['fee_name']; ?></td>
                                <td><?php echo CURRENCY . number_format($allocation['amount'], 2); ?></td>
                                <td><?php echo date('M j, Y', strtotime($allocation['due_date'])); ?></td>
                                <td><span class="status-badge status-<?php echo $allocation['status']; ?>"><?php echo ucfirst($allocation['status']); ?></span></td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="collect_fee.php?allocation_id=<?php echo $allocation['id']; ?>" class="btn-icon">
                                            <i class="material-icons">payment</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <script src="assets/js/script.js"></script>
    <script>
    // Auto-fill amount when fee category is selected
    document.getElementById('fee_category_id').addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const amount = selectedOption.getAttribute('data-amount');
        if (amount) {
            document.getElementById('amount').value = amount;
        }
    });
    
    // Set due date to 30 days from now by default
    const today = new Date();
    const dueDate = new Date();
    dueDate.setDate(today.getDate() + 30);
    document.getElementById('due_date').value = dueDate.toISOString().split('T')[0];
    </script>
</body>
</html>