<?php
// Define access constant
define('ACCESS_ALLOWED', true);

// Start session and include configuration
session_start();
require_once "config.php";

// Redirect to dashboard if already logged in
if (is_logged_in()) {
    header("Location: dashboard.php");
    exit;
}

// Handle login if form is submitted
$error = '';
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $username = trim($_POST['username']);
    $password = trim($_POST['password']);
    
    if (!empty($username) && !empty($password)) {
        // Prepare a select statement
        $sql = "SELECT id, username, password, role FROM users WHERE username = :username AND status = 'active'";
        
        if ($stmt = $pdo->prepare($sql)) {
            $stmt->bindParam(":username", $username, PDO::PARAM_STR);
            
            if ($stmt->execute()) {
                if ($stmt->rowCount() == 1) {
                    if ($row = $stmt->fetch()) {
                        $id = $row['id'];
                        $username = $row['username'];
                        $hashed_password = $row['password'];
                        $role = $row['role'];
                        
                        if (password_verify($password, $hashed_password)) {
                            // Password is correct, start a new session
                            $_SESSION['user_id'] = $id;
                            $_SESSION['username'] = $username;
                            $_SESSION['role'] = $role;
                            
                            // Log the login activity
                            logActivity($id, 'login', 'User logged in successfully');
                            
                            // Redirect user to dashboard
                            header("Location: dashboard.php");
                            exit;
                        } else {
                            $error = "Invalid password.";
                        }
                    }
                } else {
                    $error = "No account found with that username.";
                }
            } else {
                $error = "Oops! Something went wrong. Please try again later.";
            }
            
            unset($stmt);
        }
    } else {
        $error = "Please enter username and password.";
    }
}

// Get school settings for display
$school_name = getSetting('school_name', 'Excel Academy');
$school_address = getSetting('school_address', '123 Education Street, Knowledge City');
$school_phone = getSetting('school_phone', '+234 801 234 5678');
$school_email = getSetting('school_email', 'info@excelacademy.edu.ng');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Fees Management System - <?php echo $school_name; ?></title>
    <meta name="description" content="Comprehensive student fees management system for educational institutions">
    <meta name="keywords" content="student fees, management system, school administration, payment processing">
    <meta name="author" content="Excel Academy">
    
    <!-- Stylesheets -->
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="assets/icons/material-icons.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <link href="assets/css/responsive.css" rel="stylesheet">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="assets/img/favicon.ico">
    
    <style>
        .landing-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            display: flex;
            flex-direction: column;
        }
        
        .landing-header {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            padding: 20px 40px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 20px rgba(0,0,0,0.1);
        }
        
        .logo {
            display: flex;
            align-items: center;
            color: white;
            text-decoration: none;
        }
        
        .logo i {
            font-size: 36px;
            margin-right: 10px;
        }
        
        .logo h1 {
            font-size: 24px;
            font-weight: 700;
            margin: 0;
        }
        
        .nav-links {
            display: flex;
            gap: 20px;
        }
        
        .nav-links a {
            color: white;
            text-decoration: none;
            padding: 10px 15px;
            border-radius: 5px;
            transition: background 0.3s;
        }
        
        .nav-links a:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .hero-section {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 40px 20px;
            color: white;
        }
        
        .hero-content {
            max-width: 1200px;
            width: 100%;
            display: grid;
            grid-template-columns: 1fr 400px;
            gap: 60px;
            align-items: center;
        }
        
        .hero-text h2 {
            font-size: 3.5rem;
            font-weight: 700;
            margin-bottom: 20px;
            line-height: 1.2;
        }
        
        .hero-text p {
            font-size: 1.2rem;
            margin-bottom: 30px;
            opacity: 0.9;
            max-width: 600px;
        }
        
        .features-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .feature-item {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .feature-item i {
            color: #4cd964;
            font-size: 20px;
        }
        
        .hero-buttons {
            display: flex;
            gap: 15px;
            flex-wrap: wrap;
        }
        
        .btn-hero {
            padding: 15px 30px;
            border-radius: 50px;
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background: white;
            color: #667eea;
        }
        
        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
        }
        
        .btn-secondary {
            background: rgba(255, 255, 255, 0.2);
            color: white;
            border: 2px solid white;
        }
        
        .btn-secondary:hover {
            background: white;
            color: #667eea;
        }
        
        .login-form-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
        }
        
        .login-form-header {
            text-align: center;
            margin-bottom: 25px;
        }
        
        .login-form-header h3 {
            color: #333;
            margin-bottom: 10px;
            font-size: 1.5rem;
        }
        
        .login-form-header p {
            color: #666;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 500;
        }
        
        .form-control {
            width: 100%;
            padding: 15px;
            border: 2px solid #e1e5eb;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        .form-control:focus {
            border-color: #667eea;
            outline: none;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .btn-login {
            width: 100%;
            padding: 15px;
            background: linear-gradient(90deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 10px;
        }
        
        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }
        
        .alert-error {
            background: #ffebee;
            color: #c62828;
            padding: 12px 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: <?php echo $error ? 'flex' : 'none'; ?>;
            align-items: center;
            gap: 10px;
        }
        
        .demo-accounts {
            margin-top: 25px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .demo-accounts h4 {
            margin-top: 0;
            color: #333;
            margin-bottom: 10px;
        }
        
        .demo-accounts ul {
            padding-left: 20px;
            margin-bottom: 0;
            color: #666;
        }
        
        .demo-accounts li {
            margin-bottom: 5px;
        }
        
        .landing-footer {
            background: rgba(0, 0, 0, 0.2);
            color: white;
            padding: 30px 40px;
            text-align: center;
        }
        
        .footer-content {
            max-width: 1200px;
            margin: 0 auto;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 30px;
            text-align: left;
        }
        
        .footer-section h4 {
            margin-bottom: 15px;
            font-size: 18px;
        }
        
        .footer-section p, .footer-section a {
            color: rgba(255, 255, 255, 0.8);
            margin-bottom: 10px;
            display: block;
            text-decoration: none;
        }
        
        .footer-section a:hover {
            color: white;
        }
        
        .footer-bottom {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
            color: rgba(255, 255, 255, 0.7);
        }
        
        /* Responsive styles */
        @media (max-width: 992px) {
            .hero-content {
                grid-template-columns: 1fr;
                gap: 40px;
            }
            
            .hero-text h2 {
                font-size: 2.5rem;
            }
            
            .features-grid {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            .landing-header {
                flex-direction: column;
                gap: 15px;
                padding: 15px 20px;
            }
            
            .nav-links {
                flex-wrap: wrap;
                justify-content: center;
            }
            
            .hero-text h2 {
                font-size: 2rem;
            }
            
            .hero-text p {
                font-size: 1rem;
            }
            
            .hero-buttons {
                flex-direction: column;
            }
            
            .footer-content {
                grid-template-columns: 1fr;
                text-align: center;
            }
        }
        
        @media (max-width: 480px) {
            .login-form-container {
                padding: 20px;
            }
            
            .hero-text h2 {
                font-size: 1.8rem;
            }
            
            .nav-links {
                flex-direction: column;
                align-items: center;
                gap: 10px;
            }
            
            .nav-links a {
                width: 100%;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <div class="landing-container">
        <header class="landing-header">
            <a href="index.php" class="logo">
                <i class="material-icons">school</i>
                <h1><?php echo $school_name; ?></h1>
            </a>
            
            <nav class="nav-links">
                <a href="#features">Features</a>
                <a href="#about">About</a>
                <a href="#contact">Contact</a>
                <a href="login.php">Login</a>
            </nav>
        </header>
        
        <section class="hero-section">
            <div class="hero-content">
                <div class="hero-text">
                    <h2>Streamline Your School Fee Management</h2>
                    <p>Comprehensive, easy-to-use system for managing student fees, payments, and financial records. Save time and reduce errors with our automated solution.</p>
                    
                    <div class="features-grid">
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Easy Payment Processing</span>
                        </div>
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Automated Receipt Generation</span>
                        </div>
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Comprehensive Reporting</span>
                        </div>
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Multi-user Access Control</span>
                        </div>
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Student Database Management</span>
                        </div>
                        <div class="feature-item">
                            <i class="material-icons">check_circle</i>
                            <span>Mobile Responsive Design</span>
                        </div>
                    </div>
                    
                    <div class="hero-buttons">
                        <a href="#login" class="btn-hero btn-primary">
                            <i class="material-icons">lock_open</i>
                            Get Started Now
                        </a>
                        <a href="#features" class="btn-hero btn-secondary">
                            <i class="material-icons">info</i>
                            Learn More
                        </a>
                    </div>
                </div>
                
                <div class="login-form-container" id="login">
                    <div class="login-form-header">
                        <i class="material-icons" style="font-size: 48px; color: #667eea;">lock</i>
                        <h3>Login to Your Account</h3>
                        <p>Enter your credentials to access the system</p>
                    </div>
                    
                    <?php if ($error): ?>
                    <div class="alert-error">
                        <i class="material-icons">error</i>
                        <?php echo $error; ?>
                    </div>
                    <?php endif; ?>
                    
                    <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
                        <div class="form-group">
                            <label for="username">Username</label>
                            <input type="text" id="username" name="username" class="form-control" required autofocus>
                        </div>
                        
                        <div class="form-group">
                            <label for="password">Password</label>
                            <input type="password" id="password" name="password" class="form-control" required>
                        </div>
                        
                        <button type="submit" class="btn-login">
                            <i class="material-icons">login</i>
                            Login to Dashboard
                        </button>
                    </form>
                    
                    <div class="demo-accounts">
                        <h4>Demo Accounts:</h4>
                        <ul>
                            <li><strong>Admin:</strong> admin / password</li>
                            <li><strong>Accountant:</strong> accountant / password</li>
                            <li><strong>Member:</strong> johndoe / password</li>
                        </ul>
                    </div>
                </div>
            </div>
        </section>
        
        <footer class="landing-footer">
            <div class="footer-content">
                <div class="footer-section">
                    <h4><?php echo $school_name; ?></h4>
                    <p><?php echo $school_address; ?></p>
                    <p><?php echo $school_phone; ?></p>
                    <p><?php echo $school_email; ?></p>
                </div>
                
                <div class="footer-section">
                    <h4>Quick Links</h4>
                    <a href="index.php">Home</a>
                    <a href="#features">Features</a>
                    <a href="#about">About Us</a>
                    <a href="#contact">Contact</a>
                </div>
                
                <div class="footer-section">
                    <h4>Support</h4>
                    <a href="#">Help Center</a>
                    <a href="#">Documentation</a>
                    <a href="#">FAQs</a>
                    <a href="#">System Status</a>
                </div>
                
                <div class="footer-section">
                    <h4>Connect With Us</h4>
                    <a href="#"><i class="material-icons" style="vertical-align: middle; margin-right: 5px;">facebook</i> Facebook</a>
                    <a href="#"><i class="material-icons" style="vertical-align: middle; margin-right: 5px;">twitter</i> Twitter</a>
                    <a href="#"><i class="material-icons" style="vertical-align: middle; margin-right: 5px;">linkedin</i> LinkedIn</a>
                    <a href="#"><i class="material-icons" style="vertical-align: middle; margin-right: 5px;">instagram</i> Instagram</a>
                </div>
            </div>
            
            <div class="footer-bottom">
                <p>&copy; <?php echo date('Y'); ?> <?php echo $school_name; ?>. All rights reserved.</p>
                <p>Student Fees Management System v2.0</p>
            </div>
        </footer>
    </div>
    
    <script src="assets/js/script.js"></script>
    <script>
        // Smooth scrolling for anchor links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                
                const targetId = this.getAttribute('href');
                if (targetId === '#') return;
                
                const targetElement = document.querySelector(targetId);
                if (targetElement) {
                    targetElement.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });
        
        // Add animation to hero text
        document.addEventListener('DOMContentLoaded', function() {
            const heroText = document.querySelector('.hero-text');
            if (heroText) {
                heroText.style.opacity = '0';
                heroText.style.transform = 'translateY(20px)';
                
                setTimeout(() => {
                    heroText.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
                    heroText.style.opacity = '1';
                    heroText.style.transform = 'translateY(0)';
                }, 300);
            }
        });
    </script>
</body>
</html>