<?php
require_once "config.php";
check_auth();
check_role(['admin', 'accountant']);

// Get filter parameters
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');
$student_id = isset($_GET['student_id']) ? $_GET['student_id'] : '';
$fee_category_id = isset($_GET['fee_category_id']) ? $_GET['fee_category_id'] : '';

// Build WHERE clause for filters
$where_conditions = [];
$params = [];

if (!empty($start_date) && !empty($end_date)) {
    $where_conditions[] = "p.payment_date BETWEEN :start_date AND :end_date";
    $params[':start_date'] = $start_date;
    $params[':end_date'] = $end_date;
}

if (!empty($student_id)) {
    $where_conditions[] = "p.student_id = :student_id";
    $params[':student_id'] = $student_id;
}

if (!empty($fee_category_id)) {
    $where_conditions[] = "fa.fee_category_id = :fee_category_id";
    $params[':fee_category_id'] = $fee_category_id;
}

$where_clause = '';
if (!empty($where_conditions)) {
    $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
}

// Get payment reports
$sql = "
    SELECT p.*, s.first_name, s.last_name, s.student_id, fc.name as fee_name, u.username as received_by 
    FROM payments p 
    JOIN students s ON p.student_id = s.id 
    JOIN fee_allocations fa ON p.fee_allocation_id = fa.id
    JOIN fee_categories fc ON fa.fee_category_id = fc.id
    JOIN users u ON p.received_by = u.id
    $where_clause
    ORDER BY p.payment_date DESC
";

$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$payments = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate total collection
$total_collection = 0;
foreach ($payments as $payment) {
    $total_collection += $payment['amount'];
}

// Get all students for filter dropdown
$students = $pdo->query("SELECT id, first_name, last_name, student_id FROM students ORDER BY first_name")->fetchAll(PDO::FETCH_ASSOC);

// Get all fee categories for filter dropdown
$fee_categories = $pdo->query("SELECT id, name FROM fee_categories ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - Student Fees Management System</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="assets/icons/material-icons.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="main-content">
        <div class="page-header">
            <h1>Reports</h1>
            <div class="breadcrumb">
                <a href="#"><i class="material-icons">home</i> Home</a>
                <span class="divider">/</span>
                <a class="active">Reports</a>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Filter Reports</h2>
            </div>
            <div class="card-body">
                <form method="GET" class="form-grid">
                    <div class="form-group">
                        <label for="start_date">Start Date</label>
                        <input type="date" id="start_date" name="start_date" value="<?php echo $start_date; ?>">
                    </div>
                    <div class="form-group">
                        <label for="end_date">End Date</label>
                        <input type="date" id="end_date" name="end_date" value="<?php echo $end_date; ?>">
                    </div>
                    <div class="form-group">
                        <label for="student_id">Student</label>
                        <select id="student_id" name="student_id">
                            <option value="">All Students</option>
                            <?php foreach($students as $student): ?>
                            <option value="<?php echo $student['id']; ?>" <?php echo $student_id == $student['id'] ? 'selected' : ''; ?>>
                                <?php echo $student['first_name'] . ' ' . $student['last_name'] . ' (' . $student['student_id'] . ')'; ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="fee_category_id">Fee Category</label>
                        <select id="fee_category_id" name="fee_category_id">
                            <option value="">All Categories</option>
                            <?php foreach($fee_categories as $category): ?>
                            <option value="<?php echo $category['id']; ?>" <?php echo $fee_category_id == $category['id'] ? 'selected' : ''; ?>>
                                <?php echo $category['name']; ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <button type="submit" class="btn btn-primary">Apply Filters</button>
                        <a href="reports.php" class="btn btn-secondary">Reset</a>
                        <a href="print_report.php?<?php echo http_build_query($_GET); ?>" class="btn btn-success" target="_blank">
                            <i class="material-icons">print</i> Print Report
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Payment Reports</h2>
                <div class="total-collection">
                    Total Collection: <strong><?php echo CURRENCY . number_format($total_collection, 2); ?></strong>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Receipt No</th>
                                <th>Date</th>
                                <th>Student</th>
                                <th>Fee Type</th>
                                <th>Amount</th>
                                <th>Payment Method</th>
                                <th>Received By</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($payments)): ?>
                            <tr>
                                <td colspan="8" class="text-center">No payments found for the selected filters.</td>
                            </tr>
                            <?php else: ?>
                            <?php foreach($payments as $payment): ?>
                            <tr>
                                <td>#<?php echo $payment['id']; ?></td>
                                <td><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></td>
                                <td><?php echo $payment['first_name'] . ' ' . $payment['last_name']; ?></td>
                                <td><?php echo $payment['fee_name']; ?></td>
                                <td><?php echo CURRENCY . number_format($payment['amount'], 2); ?></td>
                                <td><?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?></td>
                                <td><?php echo $payment['received_by']; ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="print_receipt.php?payment_id=<?php echo $payment['id']; ?>" class="btn-icon" target="_blank">
                                            <i class="material-icons">receipt</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Collection Summary</h2>
            </div>
            <div class="card-body">
                <div class="chart-container">
                    <canvas id="collectionChart" width="400" height="200"></canvas>
                </div>
            </div>
        </div>
    </main>
    
    <script src="assets/js/chart.js"></script>
    <script src="assets/js/script.js"></script>
    <script>
    // Prepare data for the chart
    const paymentMethods = {};
    const feeCategories = {};
    
    <?php foreach($payments as $payment): ?>
    // Count by payment method
    const method = '<?php echo $payment['payment_method']; ?>';
    paymentMethods[method] = (paymentMethods[method] || 0) + <?php echo $payment['amount']; ?>;
    
    // Count by fee category
    const category = '<?php echo $payment['fee_name']; ?>';
    feeCategories[category] = (feeCategories[category] || 0) + <?php echo $payment['amount']; ?>;
    <?php endforeach; ?>
    
    // Create chart when page loads
    document.addEventListener('DOMContentLoaded', function() {
        const ctx = document.getElementById('collectionChart').getContext('2d');
        
        // Prepare data for the chart
        const labels = Object.keys(paymentMethods);
        const data = Object.values(paymentMethods);
        const backgroundColors = [
            'rgba(255, 99, 132, 0.7)',
            'rgba(54, 162, 235, 0.7)',
            'rgba(255, 206, 86, 0.7)',
            'rgba(75, 192, 192, 0.7)',
            'rgba(153, 102, 255, 0.7)'
        ];
        
        // Create the chart
        new Chart(ctx, {
            type: 'pie',
            data: {
                labels: labels,
                datasets: [{
                    data: data,
                    backgroundColor: backgroundColors,
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    title: {
                        display: true,
                        text: 'Collection by Payment Method'
                    },
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    });
    </script>
</body>
</html>