<?php
require_once "config.php";
check_auth();
check_role(['admin']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['update_settings'])) {
        // Update general settings
        $settings = [
            'school_name' => trim($_POST['school_name']),
            'school_address' => trim($_POST['school_address']),
            'school_phone' => trim($_POST['school_phone']),
            'school_email' => trim($_POST['school_email']),
            'currency_symbol' => trim($_POST['currency_symbol']),
            'currency_code' => trim($_POST['currency_code']),
            'receipt_prefix' => trim($_POST['receipt_prefix']),
            'academic_year' => trim($_POST['academic_year']),
            'low_balance_alert' => trim($_POST['low_balance_alert'])
        ];
        
        foreach ($settings as $key => $value) {
            $sql = "REPLACE INTO settings (setting_key, setting_value) VALUES (:key, :value)";
            $stmt = $pdo->prepare($sql);
            $stmt->bindParam(':key', $key);
            $stmt->bindParam(':value', $value);
            $stmt->execute();
        }
        
        $success = "Settings updated successfully!";
    }
    
    if (isset($_POST['add_custom_field'])) {
        // Add custom field
        $field_name = trim($_POST['field_name']);
        $field_type = trim($_POST['field_type']);
        $field_options = trim($_POST['field_options']);
        $model = trim($_POST['model']);
        $is_required = isset($_POST['is_required']) ? 1 : 0;
        
        $sql = "INSERT INTO custom_fields (field_name, field_type, field_options, model, is_required) 
                VALUES (:field_name, :field_type, :field_options, :model, :is_required)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':field_name', $field_name);
        $stmt->bindParam(':field_type', $field_type);
        $stmt->bindParam(':field_options', $field_options);
        $stmt->bindParam(':model', $model);
        $stmt->bindParam(':is_required', $is_required);
        
        if ($stmt->execute()) {
            $success = "Custom field added successfully!";
        } else {
            $error = "Error adding custom field. Please try again.";
        }
    }
}

// Handle delete custom field
if (isset($_GET['delete_field'])) {
    $id = $_GET['delete_field'];
    
    $sql = "DELETE FROM custom_fields WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindParam(':id', $id);
    
    if ($stmt->execute()) {
        $success = "Custom field deleted successfully!";
    } else {
        $error = "Error deleting custom field. Please try again.";
    }
}

// Get all settings
$settings_result = $pdo->query("SELECT * FROM settings")->fetchAll(PDO::FETCH_ASSOC);
$settings = [];
foreach ($settings_result as $setting) {
    $settings[$setting['setting_key']] = $setting['setting_value'];
}

// Get all custom fields
$custom_fields = $pdo->query("SELECT * FROM custom_fields ORDER BY model, field_name")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - Student Fees Management System</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="assets/icons/material-icons.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="main-content">
        <div class="page-header">
            <h1>System Settings</h1>
            <div class="breadcrumb">
                <a href="#"><i class="material-icons">home</i> Home</a>
                <span class="divider">/</span>
                <a class="active">Settings</a>
            </div>
        </div>
        
        <?php if (isset($success)): ?>
        <div class="alert alert-success">
            <i class="material-icons">check_circle</i>
            <?php echo $success; ?>
        </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
        <div class="alert alert-error">
            <i class="material-icons">error</i>
            <?php echo $error; ?>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-header">
                <h2>General Settings</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="form-grid">
                    <div class="form-group">
                        <label for="school_name">School Name</label>
                        <input type="text" id="school_name" name="school_name" value="<?php echo $settings['school_name'] ?? ''; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="school_email">School Email</label>
                        <input type="email" id="school_email" name="school_email" value="<?php echo $settings['school_email'] ?? ''; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="school_phone">School Phone</label>
                        <input type="text" id="school_phone" name="school_phone" value="<?php echo $settings['school_phone'] ?? ''; ?>" required>
                    </div>
                    <div class="form-group full-width">
                        <label for="school_address">School Address</label>
                        <textarea id="school_address" name="school_address" rows="3" required><?php echo $settings['school_address'] ?? ''; ?></textarea>
                    </div>
                    <div class="form-group">
                        <label for="currency_symbol">Currency Symbol</label>
                        <input type="text" id="currency_symbol" name="currency_symbol" value="<?php echo $settings['currency_symbol'] ?? '₦'; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="currency_code">Currency Code</label>
                        <input type="text" id="currency_code" name="currency_code" value="<?php echo $settings['currency_code'] ?? 'NGN'; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="receipt_prefix">Receipt Prefix</label>
                        <input type="text" id="receipt_prefix" name="receipt_prefix" value="<?php echo $settings['receipt_prefix'] ?? 'EXCEL'; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="academic_year">Academic Year</label>
                        <input type="text" id="academic_year" name="academic_year" value="<?php echo $settings['academic_year'] ?? '2023/2024'; ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="low_balance_alert">Low Balance Alert (<?php echo $settings['currency_symbol'] ?? '₦'; ?>)</label>
                        <input type="number" id="low_balance_alert" name="low_balance_alert" value="<?php echo $settings['low_balance_alert'] ?? '5000'; ?>" step="0.01" required>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="update_settings" class="btn btn-primary">Update Settings</button>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>Custom Fields</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="form-grid">
                    <div class="form-group">
                        <label for="field_name">Field Name</label>
                        <input type="text" id="field_name" name="field_name" required>
                    </div>
                    <div class="form-group">
                        <label for="field_type">Field Type</label>
                        <select id="field_type" name="field_type" required>
                            <option value="text">Text</option>
                            <option value="number">Number</option>
                            <option value="date">Date</option>
                            <option value="email">Email</option>
                            <option value="select">Dropdown</option>
                        </select>
                    </div>
                    <div class="form-group" id="field_options_container" style="display: none;">
                        <label for="field_options">Options (comma separated)</label>
                        <input type="text" id="field_options" name="field_options" placeholder="Option 1, Option 2, Option 3">
                    </div>
                    <div class="form-group">
                        <label for="model">Model</label>
                        <select id="model" name="model" required>
                            <option value="students">Students</option>
                            <option value="payments">Payments</option>
                            <option value="fee_categories">Fee Categories</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label class="checkbox-container">
                            <input type="checkbox" id="is_required" name="is_required">
                            <span class="checkmark"></span>
                            Required Field
                        </label>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="add_custom_field" class="btn btn-primary">Add Field</button>
                    </div>
                </form>
                
                <div class="table-responsive" style="margin-top: 30px;">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Field Name</th>
                                <th>Type</th>
                                <th>Model</th>
                                <th>Required</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach($custom_fields as $field): ?>
                            <tr>
                                <td><?php echo $field['field_name']; ?></td>
                                <td><?php echo ucfirst($field['field_type']); ?></td>
                                <td><?php echo ucfirst($field['model']); ?></td>
                                <td><?php echo $field['is_required'] ? 'Yes' : 'No'; ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="settings.php?delete_field=<?php echo $field['id']; ?>" class="btn-icon" onclick="return confirm('Are you sure you want to delete this custom field?')">
                                            <i class="material-icons">delete</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <script src="assets/js/script.js"></script>
    <script>
    // Show/hide field options based on field type
    document.getElementById('field_type').addEventListener('change', function() {
        const fieldOptionsContainer = document.getElementById('field_options_container');
        if (this.value === 'select') {
            fieldOptionsContainer.style.display = 'block';
        } else {
            fieldOptionsContainer.style.display = 'none';
        }
    });
    </script>
</body>
</html>