<?php
require_once "config.php";
check_auth();
check_role(['admin', 'accountant']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_student'])) {
        // Add new student
        $student_id = trim($_POST['student_id']);
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $address = trim($_POST['address']);
        $class = trim($_POST['class']);
        $section = trim($_POST['section']);
        $guardian_name = trim($_POST['guardian_name']);
        $guardian_phone = trim($_POST['guardian_phone']);
        
        $sql = "INSERT INTO students (student_id, first_name, last_name, email, phone, address, class, section, guardian_name, guardian_phone) 
                VALUES (:student_id, :first_name, :last_name, :email, :phone, :address, :class, :section, :guardian_name, :guardian_phone)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':student_id', $student_id);
        $stmt->bindParam(':first_name', $first_name);
        $stmt->bindParam(':last_name', $last_name);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':address', $address);
        $stmt->bindParam(':class', $class);
        $stmt->bindParam(':section', $section);
        $stmt->bindParam(':guardian_name', $guardian_name);
        $stmt->bindParam(':guardian_phone', $guardian_phone);
        
        if ($stmt->execute()) {
            $success = "Student added successfully!";
        } else {
            $error = "Error adding student. Please try again.";
        }
    }
    
    if (isset($_POST['update_student'])) {
        // Update student
        $id = $_POST['student_id'];
        $first_name = trim($_POST['first_name']);
        $last_name = trim($_POST['last_name']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $address = trim($_POST['address']);
        $class = trim($_POST['class']);
        $section = trim($_POST['section']);
        $guardian_name = trim($_POST['guardian_name']);
        $guardian_phone = trim($_POST['guardian_phone']);
        $status = trim($_POST['status']);
        
        $sql = "UPDATE students SET first_name = :first_name, last_name = :last_name, email = :email, 
                phone = :phone, address = :address, class = :class, section = :section, 
                guardian_name = :guardian_name, guardian_phone = :guardian_phone, status = :status 
                WHERE id = :id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':first_name', $first_name);
        $stmt->bindParam(':last_name', $last_name);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':address', $address);
        $stmt->bindParam(':class', $class);
        $stmt->bindParam(':section', $section);
        $stmt->bindParam(':guardian_name', $guardian_name);
        $stmt->bindParam(':guardian_phone', $guardian_phone);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            $success = "Student updated successfully!";
        } else {
            $error = "Error updating student. Please try again.";
        }
    }
}

// Handle delete request
if (isset($_GET['delete'])) {
    $id = $_GET['delete'];
    
    // Check if student has any payments before deleting
    $check_sql = "SELECT COUNT(*) FROM payments WHERE student_id = :id";
    $check_stmt = $pdo->prepare($check_sql);
    $check_stmt->bindParam(':id', $id);
    $check_stmt->execute();
    $payment_count = $check_stmt->fetchColumn();
    
    if ($payment_count > 0) {
        $error = "Cannot delete student. There are payments associated with this student.";
    } else {
        $sql = "DELETE FROM students WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            $success = "Student deleted successfully!";
        } else {
            $error = "Error deleting student. Please try again.";
        }
    }
}

// Get all students
$students = $pdo->query("SELECT * FROM students ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Students - Student Fees Management System</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <link href="assets/icons/material-icons.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    <?php include 'includes/sidebar.php'; ?>
    
    <main class="main-content">
        <div class="page-header">
            <h1>Student Management</h1>
            <div class="breadcrumb">
                <a href="#"><i class="material-icons">home</i> Home</a>
                <span class="divider">/</span>
                <a class="active">Students</a>
            </div>
        </div>
        
        <?php if (isset($success)): ?>
        <div class="alert alert-success">
            <i class="material-icons">check_circle</i>
            <?php echo $success; ?>
        </div>
        <?php endif; ?>
        
        <?php if (isset($error)): ?>
        <div class="alert alert-error">
            <i class="material-icons">error</i>
            <?php echo $error; ?>
        </div>
        <?php endif; ?>
        
        <div class="card">
            <div class="card-header">
                <h2>Add New Student</h2>
            </div>
            <div class="card-body">
                <form method="POST" class="form-grid">
                    <div class="form-group">
                        <label for="student_id">Student ID</label>
                        <input type="text" id="student_id" name="student_id" required>
                    </div>
                    <div class="form-group">
                        <label for="first_name">First Name</label>
                        <input type="text" id="first_name" name="first_name" required>
                    </div>
                    <div class="form-group">
                        <label for="last_name">Last Name</label>
                        <input type="text" id="last_name" name="last_name" required>
                    </div>
                    <div class="form-group">
                        <label for="email">Email</label>
                        <input type="email" id="email" name="email" required>
                    </div>
                    <div class="form-group">
                        <label for="phone">Phone</label>
                        <input type="text" id="phone" name="phone">
                    </div>
                    <div class="form-group">
                        <label for="class">Class</label>
                        <input type="text" id="class" name="class">
                    </div>
                    <div class="form-group">
                        <label for="section">Section</label>
                        <input type="text" id="section" name="section">
                    </div>
                    <div class="form-group">
                        <label for="guardian_name">Guardian Name</label>
                        <input type="text" id="guardian_name" name="guardian_name">
                    </div>
                    <div class="form-group">
                        <label for="guardian_phone">Guardian Phone</label>
                        <input type="text" id="guardian_phone" name="guardian_phone">
                    </div>
                    <div class="form-group full-width">
                        <label for="address">Address</label>
                        <textarea id="address" name="address" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="add_student" class="btn btn-primary">Add Student</button>
                    </div>
                </form>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h2>All Students</h2>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Class</th>
                                <th>Guardian</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach($students as $student): ?>
                            <tr>
                                <td><?php echo $student['student_id']; ?></td>
                                <td><?php echo $student['first_name'] . ' ' . $student['last_name']; ?></td>
                                <td><?php echo $student['email']; ?></td>
                                <td><?php echo $student['class'] . ' - ' . $student['section']; ?></td>
                                <td><?php echo $student['guardian_name']; ?></td>
                                <td><span class="status-badge status-<?php echo $student['status']; ?>"><?php echo ucfirst($student['status']); ?></span></td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn-icon" onclick="editStudent(<?php echo $student['id']; ?>)">
                                            <i class="material-icons">edit</i>
                                        </button>
                                        <a href="students.php?delete=<?php echo $student['id']; ?>" class="btn-icon" onclick="return confirm('Are you sure you want to delete this student?')">
                                            <i class="material-icons">delete</i>
                                        </a>
                                        <a href="student_details.php?id=<?php echo $student['id']; ?>" class="btn-icon">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </main>
    
    <!-- Edit Student Modal -->
    <div id="editModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2>Edit Student</h2>
                <span class="close">&times;</span>
            </div>
            <div class="modal-body">
                <form id="editForm" method="POST" class="form-grid">
                    <input type="hidden" name="student_id" id="edit_id">
                    <div class="form-group">
                        <label for="edit_student_id">Student ID</label>
                        <input type="text" id="edit_student_id" name="student_id" readonly>
                    </div>
                    <div class="form-group">
                        <label for="edit_first_name">First Name</label>
                        <input type="text" id="edit_first_name" name="first_name" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_last_name">Last Name</label>
                        <input type="text" id="edit_last_name" name="last_name" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_email">Email</label>
                        <input type="email" id="edit_email" name="email" required>
                    </div>
                    <div class="form-group">
                        <label for="edit_phone">Phone</label>
                        <input type="text" id="edit_phone" name="phone">
                    </div>
                    <div class="form-group">
                        <label for="edit_class">Class</label>
                        <input type="text" id="edit_class" name="class">
                    </div>
                    <div class="form-group">
                        <label for="edit_section">Section</label>
                        <input type="text" id="edit_section" name="section">
                    </div>
                    <div class="form-group">
                        <label for="edit_guardian_name">Guardian Name</label>
                        <input type="text" id="edit_guardian_name" name="guardian_name">
                    </div>
                    <div class="form-group">
                        <label for="edit_guardian_phone">Guardian Phone</label>
                        <input type="text" id="edit_guardian_phone" name="guardian_phone">
                    </div>
                    <div class="form-group">
                        <label for="edit_status">Status</label>
                        <select id="edit_status" name="status">
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                    <div class="form-group full-width">
                        <label for="edit_address">Address</label>
                        <textarea id="edit_address" name="address" rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <button type="submit" name="update_student" class="btn btn-primary">Update Student</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script src="assets/js/script.js"></script>
    <script>
    function editStudent(id) {
        // Fetch student data via AJAX
        fetch('ajax/get_student.php?id=' + id)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const student = data.student;
                    document.getElementById('edit_id').value = student.id;
                    document.getElementById('edit_student_id').value = student.student_id;
                    document.getElementById('edit_first_name').value = student.first_name;
                    document.getElementById('edit_last_name').value = student.last_name;
                    document.getElementById('edit_email').value = student.email;
                    document.getElementById('edit_phone').value = student.phone;
                    document.getElementById('edit_class').value = student.class;
                    document.getElementById('edit_section').value = student.section;
                    document.getElementById('edit_guardian_name').value = student.guardian_name;
                    document.getElementById('edit_guardian_phone').value = student.guardian_phone;
                    document.getElementById('edit_status').value = student.status;
                    document.getElementById('edit_address').value = student.address;
                    
                    // Show the modal
                    document.getElementById('editModal').style.display = 'block';
                }
            });
    }
    
    // Close modal when clicking on X
    document.querySelector('.close').addEventListener('click', function() {
        document.getElementById('editModal').style.display = 'none';
    });
    
    // Close modal when clicking outside
    window.addEventListener('click', function(event) {
        if (event.target == document.getElementById('editModal')) {
            document.getElementById('editModal').style.display = 'none';
        }
    });
    </script>
</body>
</html>