<?php
require_once 'config.php';
require_login();

// Fetch stats
$students_count = $conn->query("SELECT COUNT(*) as count FROM students")->fetch_assoc()['count'];
$fees_collected = $conn->query("SELECT SUM(amount_paid) as total FROM fees WHERE status = 'completed'")->fetch_assoc()['total'];

// Calculate pending fees
$fee_structure = $conn->query("SELECT class, SUM(amount) as total FROM fee_structure GROUP BY class");
$class_totals = array();
while ($row = $fee_structure->fetch_assoc()) {
    $class_totals[$row['class']] = $row['total'];
}

$students = $conn->query("SELECT id, class FROM students");
$total_fees_due = 0;
while ($student = $students->fetch_assoc()) {
    if (isset($class_totals[$student['class']])) {
        $total_fees_due += $class_totals[$student['class']];
    }
}

$pending_fees = $total_fees_due - $fees_collected;

// Get payment status counts
$paid_count = 0;
$partial_count = 0;
$unpaid_count = 0;

$students = $conn->query("SELECT s.id, s.class, COALESCE(SUM(f.amount_paid), 0) as paid FROM students s LEFT JOIN fees f ON s.id = f.student_id AND f.status = 'completed' GROUP BY s.id, s.class");
while ($student = $students->fetch_assoc()) {
    $total_due = isset($class_totals[$student['class']]) ? $class_totals[$student['class']] : 0;
    $status = get_payment_status($student['paid'], $total_due);
    
    if ($status['status'] === 'paid') $paid_count++;
    elseif ($status['status'] === 'partial') $partial_count++;
    else $unpaid_count++;
}

// Recent payments
$recent_payments = $conn->query("
    SELECT f.*, s.name as student_name, s.student_id, fs.amount as total_amount 
    FROM fees f 
    JOIN students s ON f.student_id = s.id 
    JOIN fee_structure fs ON f.fee_structure_id = fs.id 
    ORDER BY f.payment_date DESC LIMIT 5
");

// Currency change handling
if (isset($_POST['change_currency'])) {
    $currency_code = $_POST['currency_code'];
    $rate = get_exchange_rate($currency_code);
    
    $symbols = [
        'NGN' => '₦',
        'USD' => '$',
        'EUR' => '€',
        'GBP' => '£',
        'JPY' => '¥'
    ];
    
    $_SESSION['currency'] = [
        'symbol' => $symbols[$currency_code] ?? '₦',
        'code' => $currency_code,
        'rate' => $rate
    ];
}

// Monthly fees data for chart
$monthly_data = $conn->query("
    SELECT YEAR(payment_date) as year, MONTH(payment_date) as month, SUM(amount_paid) as total 
    FROM fees 
    WHERE status = 'completed' AND payment_date IS NOT NULL 
    GROUP BY YEAR(payment_date), MONTH(payment_date) 
    ORDER BY year, month
");

$monthly_labels = [];
$monthly_totals = [];

while ($row = $monthly_data->fetch_assoc()) {
    $monthly_labels[] = date('M Y', mktime(0, 0, 0, $row['month'], 1, $row['year']));
    $monthly_totals[] = $row['total'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@mdi/font@7.2.96/css/materialdesignicons.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
            color: white;
            transition: all 0.3s;
        }
        .sidebar .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 0.8rem 1rem;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
        }
        .stat-card {
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        .watermark {
            position: fixed;
            bottom: 20px;
            right: 20px;
            opacity: 0.1;
            font-size: 24px;
            font-weight: bold;
            color: #000;
            pointer-events: none;
            z-index: 9999;
        }
        .chart-container {
            position: relative;
            height: 300px;
        }
        @media print {
            .watermark {
                opacity: 0.3;
            }
        }
        .mdi {
            font-size: 1.5rem;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-center mb-4"><?php echo SITE_NAME; ?></h4>
                    <hr>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link active" href="dashboard.php">
                                <i class="mdi mdi-view-dashboard me-2"></i> Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="students.php">
                                <i class="mdi mdi-account-group me-2"></i> Students
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="fees.php">
                                <i class="mdi mdi-cash-multiple me-2"></i> Fees Management
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="reports.php">
                                <i class="mdi mdi-chart-bar me-2"></i> Reports
                            </a>
                        </li>
                        <?php if (is_admin()): ?>
                        <li class="nav-item">
                            <a class="nav-link" href="settings.php">
                                <i class="mdi mdi-cog me-2"></i> Settings
                            </a>
                        </li>
                        <?php endif; ?>
                        <li class="nav-item mt-4">
                            <a class="nav-link text-danger" href="index.php?logout=1">
                                <i class="mdi mdi-logout me-2"></i> Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main content -->
            <div class="col-md-9 col-lg-10 ms-sm-auto px-4 py-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Dashboard</h2>
                    <form method="POST" class="d-flex align-items-center">
                        <select name="currency_code" class="form-select me-2">
                            <option value="NGN" <?php echo $_SESSION['currency']['code'] === 'NGN' ? 'selected' : ''; ?>>Naira (₦)</option>
                            <option value="USD" <?php echo $_SESSION['currency']['code'] === 'USD' ? 'selected' : ''; ?>>US Dollar ($)</option>
                            <option value="EUR" <?php echo $_SESSION['currency']['code'] === 'EUR' ? 'selected' : ''; ?>>Euro (€)</option>
                            <option value="GBP" <?php echo $_SESSION['currency']['code'] === 'GBP' ? 'selected' : ''; ?>>British Pound (£)</option>
                            <option value="JPY" <?php echo $_SESSION['currency']['code'] === 'JPY' ? 'selected' : ''; ?>>Japanese Yen (¥)</option>
                        </select>
                        <button type="submit" name="change_currency" class="btn btn-sm btn-outline-primary">Change</button>
                    </form>
                </div>

                <!-- Stats cards -->
                <div class="row g-4 mb-4">
                    <div class="col-md-3">
                        <div class="stat-card p-4 bg-primary text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Total Students</h5>
                                    <h2 class="mb-0"><?php echo $students_count; ?></h2>
                                </div>
                                <i class="mdi mdi-account-group"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card p-4 bg-success text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Fees Collected</h5>
                                    <h2 class="mb-0"><?php echo format_currency($fees_collected ?? 0); ?></h2>
                                </div>
                                <i class="mdi mdi-cash-multiple"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card p-4 bg-warning text-dark">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Pending Fees</h5>
                                    <h2 class="mb-0"><?php echo format_currency($pending_fees ?? 0); ?></h2>
                                </div>
                                <i class="mdi mdi-clock-alert"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stat-card p-4 bg-info text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Total Fees Due</h5>
                                    <h2 class="mb-0"><?php echo format_currency($total_fees_due ?? 0); ?></h2>
                                </div>
                                <i class="mdi mdi-cash-clock"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Payment status cards -->
                <div class="row g-4 mb-4">
                    <div class="col-md-4">
                        <div class="stat-card p-4 bg-success text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Fully Paid</h5>
                                    <h2 class="mb-0"><?php echo $paid_count; ?></h2>
                                    <p class="mb-0">Students</p>
                                </div>
                                <i class="mdi mdi-check-circle"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card p-4 bg-warning text-dark">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Partial Payment</h5>
                                    <h2 class="mb-0"><?php echo $partial_count; ?></h2>
                                    <p class="mb-0">Students</p>
                                </div>
                                <i class="mdi mdi-alert-circle"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="stat-card p-4 bg-danger text-white">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <h5 class="card-title">Not Paid</h5>
                                    <h2 class="mb-0"><?php echo $unpaid_count; ?></h2>
                                    <p class="mb-0">Students</p>
                                </div>
                                <i class="mdi mdi-close-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Monthly Fees Collection</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="monthlyChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Payment Status Distribution</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="statusChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent payments -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Recent Payments</h5>
                        <a href="fees.php" class="btn btn-sm btn-primary">View All</a>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Student ID</th>
                                        <th>Student Name</th>
                                        <th>Amount Paid</th>
                                        <th>Payment Date</th>
                                        <th>Payment Method</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($payment = $recent_payments->fetch_assoc()): ?>
                                    <tr>
                                        <td><?php echo $payment['student_id']; ?></td>
                                        <td><?php echo $payment['student_name']; ?></td>
                                        <td><?php echo format_currency($payment['amount_paid']); ?></td>
                                        <td><?php echo date('M j, Y', strtotime($payment['payment_date'])); ?></td>
                                        <td><?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $payment['status'] === 'completed' ? 'success' : 'warning'; ?>">
                                                <?php echo ucfirst($payment['status']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php apply_watermark(); ?>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Monthly fees chart
        var monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
        var monthlyChart = new Chart(monthlyCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($monthly_labels); ?>,
                datasets: [{
                    label: 'Fees Collected (<?php echo $_SESSION['currency']['code']; ?>)',
                    data: <?php echo json_encode($monthly_totals); ?>,
                    backgroundColor: 'rgba(54, 162, 235, 0.2)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 2,
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Payment status chart
        var statusCtx = document.getElementById('statusChart').getContext('2d');
        var statusChart = new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: ['Fully Paid', 'Partial Payment', 'Not Paid'],
                datasets: [{
                    data: [<?php echo $paid_count; ?>, <?php echo $partial_count; ?>, <?php echo $unpaid_count; ?>],
                    backgroundColor: [
                        'rgba(40, 167, 69, 0.8)',
                        'rgba(255, 193, 7, 0.8)',
                        'rgba(220, 53, 69, 0.8)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    </script>
</body>
</html>