-- Create database
CREATE DATABASE IF NOT EXISTS fuzesale_mem3;
USE fuzesale_mem3;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) NOT NULL,
    role ENUM('admin', 'accountant') DEFAULT 'accountant',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Students table
CREATE TABLE IF NOT EXISTS students (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id VARCHAR(20) NOT NULL UNIQUE,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    class ENUM('KG', 'Nursery', 'Primary 1', 'Primary 2', 'Primary 3', 'Primary 4', 'Primary 5', 'Primary 6', 'JSS 1', 'JSS 2', 'JSS 3', 'SS 1', 'SS 2', 'SS 3') NOT NULL,
    guardian_name VARCHAR(100),
    guardian_phone VARCHAR(20),
    address TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Fees structure table
CREATE TABLE IF NOT EXISTS fee_structure (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class ENUM('KG', 'Nursery', 'Primary 1', 'Primary 2', 'Primary 3', 'Primary 4', 'Primary 5', 'Primary 6', 'JSS 1', 'JSS 2', 'JSS 3', 'SS 1', 'SS 2', 'SS 3') NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    academic_year VARCHAR(9) NOT NULL,
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Fees payments table
CREATE TABLE IF NOT EXISTS fees (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    fee_structure_id INT NOT NULL,
    amount_paid DECIMAL(10, 2) NOT NULL,
    payment_date DATE NOT NULL,
    payment_method ENUM('cash', 'bank transfer', 'card', 'online') DEFAULT 'cash',
    transaction_id VARCHAR(100),
    status ENUM('pending', 'completed', 'failed') DEFAULT 'completed',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
    FOREIGN KEY (fee_structure_id) REFERENCES fee_structure(id) ON DELETE CASCADE
);

-- Custom fields table
CREATE TABLE IF NOT EXISTS custom_fields (
    id INT AUTO_INCREMENT PRIMARY KEY,
    field_name VARCHAR(50) NOT NULL,
    field_type ENUM('text', 'number', 'date', 'select') NOT NULL,
    field_options TEXT,
    model ENUM('students', 'fees') NOT NULL,
    is_required BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Custom field values table
CREATE TABLE IF NOT EXISTS custom_field_values (
    id INT AUTO_INCREMENT PRIMARY KEY,
    field_id INT NOT NULL,
    record_id INT NOT NULL,
    value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (field_id) REFERENCES custom_fields(id) ON DELETE CASCADE
);

-- System settings table
CREATE TABLE IF NOT EXISTS system_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(50) NOT NULL UNIQUE,
    setting_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Insert default admin and accountant accounts
-- Password for admin: password (hashed)
-- Password for accountant: password123 (hashed)
INSERT INTO users (username, password, email, role) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@school.com', 'admin'),
('accountant', '$2y$10$r.2Iu2UJ/4qQZqZqZqZqZuZqZqZqZqZqZqZqZqZqZqZqZqZqZqZq', 'accountant@school.com', 'accountant');

-- Insert default fee structure
INSERT INTO fee_structure (class, amount, academic_year, description) VALUES
('KG', 20000.00, '2023-2024', 'Kindergarten Fees'),
('Nursery', 25000.00, '2023-2024', 'Nursery School Fees'),
('Primary 1', 30000.00, '2023-2024', 'Primary 1 Fees'),
('Primary 2', 30000.00, '2023-2024', 'Primary 2 Fees'),
('Primary 3', 32000.00, '2023-2024', 'Primary 3 Fees'),
('Primary 4', 32000.00, '2023-2024', 'Primary 4 Fees'),
('Primary 5', 35000.00, '2023-2024', 'Primary 5 Fees'),
('Primary 6', 35000.00, '2023-2024', 'Primary 6 Fees'),
('JSS 1', 40000.00, '2023-2024', 'Junior Secondary 1 Fees'),
('JSS 2', 40000.00, '2023-2024', 'Junior Secondary 2 Fees'),
('JSS 3', 45000.00, '2023-2024', 'Junior Secondary 3 Fees'),
('SS 1', 50000.00, '2023-2024', 'Senior Secondary 1 Fees'),
('SS 2', 50000.00, '2023-2024', 'Senior Secondary 2 Fees'),
('SS 3', 55000.00, '2023-2024', 'Senior Secondary 3 Fees');

-- Insert sample students
INSERT INTO students (student_id, name, email, phone, class, guardian_name, guardian_phone, address) VALUES
('STU001', 'John Doe', 'john.doe@example.com', '08012345678', 'Primary 5', 'Michael Doe', '08087654321', '123 Main Street, Lagos'),
('STU002', 'Jane Smith', 'jane.smith@example.com', '08023456789', 'JSS 2', 'Robert Smith', '08098765432', '456 Oak Avenue, Abuja'),
('STU003', 'Michael Johnson', 'michael.j@example.com', '08034567890', 'SS 1', 'Sarah Johnson', '08076543210', '789 Pine Road, Port Harcourt'),
('STU004', 'Sarah Williams', 'sarah.w@example.com', '08045678901', 'Primary 4', 'James Williams', '08065432109', '321 Elm Street, Ibadan'),
('STU005', 'Robert Brown', 'robert.b@example.com', '08056789012', 'JSS 3', 'Elizabeth Brown', '08054321098', '654 Cedar Lane, Kano');

-- Insert sample fees records
INSERT INTO fees (student_id, fee_structure_id, amount_paid, payment_date, payment_method, transaction_id, status) VALUES
(1, 7, 35000.00, '2023-10-10', 'bank transfer', 'TRX001', 'completed'),
(2, 9, 20000.00, '2023-10-12', 'cash', 'TRX002', 'completed'),
(3, 12, 25000.00, '2023-10-15', 'online', 'TRX003', 'completed'),
(4, 4, 32000.00, '2023-10-18', 'bank transfer', 'TRX004', 'completed'),
(5, 10, 40000.00, '2023-10-20', 'card', 'TRX005', 'completed'),
(2, 9, 20000.00, '2023-10-25', 'cash', 'TRX006', 'completed');

-- Insert system settings
INSERT INTO system_settings (setting_key, setting_value) VALUES
('school_name', 'Excel Academy'),
('school_address', '123 Education Street, Knowledge City'),
('school_phone', '+234 801 234 5678'),
('school_email', 'info@excelacademy.edu.ng'),
('currency', 'NGN'),
('currency_rate', '1'),
('academic_year', '2023-2024');