<?php
require_once 'config.php';
require_login();

// Handle export requests
if (isset($_GET['export'])) {
    $export_type = $_GET['export'];
    
    if ($export_type === 'students') {
        $students = $conn->query("
            SELECT s.student_id, s.name, s.email, s.phone, s.class, s.guardian_name, s.guardian_phone, s.address,
            COALESCE(SUM(f.amount_paid), 0) as paid_amount,
            (SELECT amount FROM fee_structure WHERE class = s.class LIMIT 1) as total_amount
            FROM students s 
            LEFT JOIN fees f ON s.id = f.student_id AND f.status = 'completed'
            GROUP BY s.id
            ORDER BY s.id DESC
        ");
        
        $data = array();
        while ($row = $students->fetch_assoc()) {
            $status = get_payment_status($row['paid_amount'], $row['total_amount']);
            $row['payment_status'] = $status['status'];
            $row['balance'] = $row['total_amount'] - $row['paid_amount'];
            $data[] = $row;
        }
        
        if ($_GET['format'] === 'csv') {
            export_to_csv($data, 'students_report_' . date('Y-m-d'));
        } elseif ($_GET['format'] === 'json') {
            export_to_json($data, 'students_report_' . date('Y-m-d'));
        }
    } elseif ($export_type === 'fees') {
        $fees = $conn->query("
            SELECT f.*, s.name as student_name, s.student_id as student_code, s.class, fs.amount as total_amount 
            FROM fees f 
            JOIN students s ON f.student_id = s.id 
            JOIN fee_structure fs ON f.fee_structure_id = fs.id 
            ORDER BY f.payment_date DESC
        ");
        
        $data = array();
        while ($row = $fees->fetch_assoc()) {
            $data[] = $row;
        }
        
        if ($_GET['format'] === 'csv') {
            export_to_csv($data, 'fees_report_' . date('Y-m-d'));
        } elseif ($_GET['format'] === 'json') {
            export_to_json($data, 'fees_report_' . date('Y-m-d'));
        }
    }
}

// Handle backup requests
if (isset($_GET['backup'])) {
    $backup = backup_database($conn);
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="database_backup_' . date('Y-m-d') . '.json"');
    echo json_encode($backup, JSON_PRETTY_PRINT);
    exit();
}

// Handle restore requests
if (isset($_POST['restore'])) {
    if ($_FILES['backup_file']['error'] === UPLOAD_ERR_OK) {
        $backup = import_from_json($_FILES['backup_file']);
        restore_database($conn, $backup);
        $success = "Database restored successfully!";
    } else {
        $error = "Error uploading backup file.";
    }
}

// Fetch data for reports
$total_students = $conn->query("SELECT COUNT(*) as count FROM students")->fetch_assoc()['count'];
$total_fees = $conn->query("SELECT SUM(amount_paid) as total FROM fees WHERE status = 'completed'")->fetch_assoc()['total'];

// Calculate pending fees
$fee_structure = $conn->query("SELECT class, SUM(amount) as total FROM fee_structure GROUP BY class");
$class_totals = array();
while ($row = $fee_structure->fetch_assoc()) {
    $class_totals[$row['class']] = $row['total'];
}

$students = $conn->query("SELECT id, class FROM students");
$total_fees_due = 0;
while ($student = $students->fetch_assoc()) {
    if (isset($class_totals[$student['class']])) {
        $total_fees_due += $class_totals[$student['class']];
    }
}

$pending_fees = $total_fees_due - $total_fees;

// Get payment status counts
$paid_count = 0;
$partial_count = 0;
$unpaid_count = 0;

$students = $conn->query("SELECT s.id, s.class, COALESCE(SUM(f.amount_paid), 0) as paid FROM students s LEFT JOIN fees f ON s.id = f.student_id AND f.status = 'completed' GROUP BY s.id, s.class");
while ($student = $students->fetch_assoc()) {
    $total_due = isset($class_totals[$student['class']]) ? $class_totals[$student['class']] : 0;
    $status = get_payment_status($student['paid'], $total_due);
    
    if ($status['status'] === 'paid') $paid_count++;
    elseif ($status['status'] === 'partial') $partial_count++;
    else $unpaid_count++;
}

// Monthly fees data for chart
$monthly_data = $conn->query("
    SELECT YEAR(payment_date) as year, MONTH(payment_date) as month, SUM(amount_paid) as total 
    FROM fees 
    WHERE status = 'completed' AND payment_date IS NOT NULL 
    GROUP BY YEAR(payment_date), MONTH(payment_date) 
    ORDER BY year, month
");

$monthly_labels = [];
$monthly_totals = [];

while ($row = $monthly_data->fetch_assoc()) {
    $monthly_labels[] = date('M Y', mktime(0, 0, 0, $row['month'], 1, $row['year']));
    $monthly_totals[] = $row['total'];
}

// Class-wise data
$class_data = $conn->query("
    SELECT s.class, SUM(f.amount_paid) as total 
    FROM fees f 
    JOIN students s ON f.student_id = s.id 
    WHERE f.status = 'completed' 
    GROUP BY s.class
");

$class_labels = [];
$class_totals = [];

while ($row = $class_data->fetch_assoc()) {
    $class_labels[] = $row['class'];
    $class_totals[] = $row['total'];
}

// Payment method data
$method_data = $conn->query("
    SELECT payment_method, SUM(amount_paid) as total 
    FROM fees 
    WHERE status = 'completed' 
    GROUP BY payment_method
");

$method_labels = [];
$method_totals = [];

while ($row = $method_data->fetch_assoc()) {
    $method_labels[] = ucfirst(str_replace('_', ' ', $row['payment_method']));
    $method_totals[] = $row['total'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports - <?php echo SITE_NAME; ?></title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@mdi/font@7.2.96/css/materialdesignicons.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .sidebar {
            min-height: 100vh;
            background: #2c3e50;
            color: white;
        }
        .sidebar .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 0.8rem 1rem;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            color: white;
            background: rgba(255, 255, 255, 0.1);
        }
        .watermark {
            position: fixed;
            bottom: 20px;
            right: 20px;
            opacity: 0.1;
            font-size: 24px;
            font-weight: bold;
            color: #000;
            pointer-events: none;
            z-index: 9999;
        }
        @media print {
            .sidebar, .btn, .no-print {
                display: none !important;
            }
            .main-content {
                margin-left: 0 !important;
                width: 100% !important;
            }
            .watermark {
                opacity: 0.3;
            }
        }
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 30px;
        }
        .export-buttons {
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 sidebar p-0">
                <div class="p-4">
                    <h4 class="text-center mb-4"><?php echo SITE_NAME; ?></h4>
                    <hr>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">
                                <i class="mdi mdi-view-dashboard me-2"></i> Dashboard
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="students.php">
                                <i class="mdi mdi-account-group me-2"></i> Students
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="fees.php">
                                <i class="mdi mdi-cash-multiple me-2"></i> Fees Management
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link active" href="reports.php">
                                <i class="mdi mdi-chart-bar me-2"></i> Reports
                            </a>
                        </li>
                        <?php if (is_admin()): ?>
                        <li class="nav-item">
                            <a class="nav-link" href="settings.php">
                                <i class="mdi mdi-cog me-2"></i> Settings
                            </a>
                        </li>
                        <?php endif; ?>
                        <li class="nav-item mt-4">
                            <a class="nav-link text-danger" href="index.php?logout=1">
                                <i class="mdi mdi-logout me-2"></i> Logout
                            </a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Main content -->
            <div class="col-md-9 col-lg-10 ms-sm-auto px-4 py-4 main-content">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Reports & Analytics</h2>
                    <div>
                        <button class="btn btn-primary no-print" onclick="window.print()">
                            <i class="mdi mdi-printer me-1"></i> Print Report
                        </button>
                    </div>
                </div>

                <!-- Export buttons -->
                <div class="row export-buttons no-print">
                    <div class="col-md-12">
                        <div class="card">
                            <div class="card-body">
                                <h5 class="card-title">Export Data</h5>
                                <div class="btn-group me-2">
                                    <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                        <i class="mdi mdi-file-export me-1"></i> Export Students
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="reports.php?export=students&format=csv">CSV Format</a></li>
                                        <li><a class="dropdown-item" href="reports.php?export=students&format=json">JSON Format</a></li>
                                    </ul>
                                </div>
                                <div class="btn-group me-2">
                                    <button type="button" class="btn btn-outline-primary dropdown-toggle" data-bs-toggle="dropdown">
                                        <i class="mdi mdi-file-export me-1"></i> Export Fees
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="reports.php?export=fees&format=csv">CSV Format</a></li>
                                        <li><a class="dropdown-item" href="reports.php?export=fees&format=json">JSON Format</a></li>
                                    </ul>
                                </div>
                                <div class="btn-group">
                                    <a href="reports.php?backup=1" class="btn btn-outline-success">
                                        <i class="mdi mdi-database-export me-1"></i> Backup Database
                                    </a>
                                </div>
                                
                                <?php if (is_admin()): ?>
                                <div class="mt-3">
                                    <h6>Restore Database</h6>
                                    <form method="POST" enctype="multipart/form-data" class="row g-3">
                                        <div class="col-md-6">
                                            <input type="file" class="form-control" name="backup_file" accept=".json" required>
                                        </div>
                                        <div class="col-md-6">
                                            <button type="submit" name="restore" class="btn btn-outline-warning">
                                                <i class="mdi mdi-database-import me-1"></i> Restore Database
                                            </button>
                                        </div>
                                    </form>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Alerts -->
                <?php if (isset($success)): ?>
                <div class="alert alert-success"><?php echo $success; ?></div>
                <?php endif; ?>
                
                <?php if (isset($error)): ?>
                <div class="alert alert-danger"><?php echo $error; ?></div>
                <?php endif; ?>

                <!-- Summary cards -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card bg-primary text-white p-3 text-center">
                            <h5>Total Students</h5>
                            <h2><?php echo $total_students; ?></h2>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-success text-white p-3 text-center">
                            <h5>Total Fees Collected</h5>
                            <h2><?php echo format_currency($total_fees ?? 0); ?></h2>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-warning text-dark p-3 text-center">
                            <h5>Pending Fees</h5>
                            <h2><?php echo format_currency($pending_fees ?? 0); ?></h2>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card bg-info text-white p-3 text-center">
                            <h5>Total Fees Due</h5>
                            <h2><?php echo format_currency($total_fees_due ?? 0); ?></h2>
                        </div>
                    </div>
                </div>

                <!-- Payment status cards -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="card bg-success text-white p-3 text-center">
                            <h5>Fully Paid</h5>
                            <h2><?php echo $paid_count; ?></h2>
                            <p class="mb-0">Students</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card bg-warning text-dark p-3 text-center">
                            <h5>Partial Payment</h5>
                            <h2><?php echo $partial_count; ?></h2>
                            <p class="mb-0">Students</p>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="card bg-danger text-white p-3 text-center">
                            <h5>Not Paid</h5>
                            <h2><?php echo $unpaid_count; ?></h2>
                            <p class="mb-0">Students</p>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Monthly Fees Collection</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="monthlyChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Fees by Class</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="classChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Payment Status Distribution</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="statusChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <h5>Payment Methods</h5>
                            </div>
                            <div class="card-body">
                                <div class="chart-container">
                                    <canvas id="methodChart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Detailed report -->
                <div class="card">
                    <div class="card-header">
                        <h5>Detailed Fees Report</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Student ID</th>
                                        <th>Student Name</th>
                                        <th>Class</th>
                                        <th>Amount Paid</th>
                                        <th>Total Amount</th>
                                        <th>Balance</th>
                                        <th>Payment Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    $detailed_report = $conn->query("
                                        SELECT s.student_id, s.name, s.class,
                                        COALESCE(SUM(f.amount_paid), 0) as paid_amount,
                                        (SELECT amount FROM fee_structure WHERE class = s.class LIMIT 1) as total_amount
                                        FROM students s 
                                        LEFT JOIN fees f ON s.id = f.student_id AND f.status = 'completed'
                                        GROUP BY s.id
                                        ORDER BY s.class, s.name
                                    ");
                                    
                                    while($row = $detailed_report->fetch_assoc()): 
                                        $status = get_payment_status($row['paid_amount'], $row['total_amount']);
                                        $balance = $row['total_amount'] - $row['paid_amount'];
                                    ?>
                                    <tr>
                                        <td><?php echo $row['student_id']; ?></td>
                                        <td><?php echo $row['name']; ?></td>
                                        <td><?php echo $row['class']; ?></td>
                                        <td><?php echo format_currency($row['paid_amount']); ?></td>
                                        <td><?php echo format_currency($row['total_amount']); ?></td>
                                        <td><?php echo format_currency($balance); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $status['class']; ?>">
                                                <?php echo ucfirst($status['status']); ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Footer -->
                <footer class="mt-5 pt-4 border-top text-center no-print">
                    <p>&copy; <?php echo date('Y'); ?> <?php echo SITE_NAME; ?>. All rights reserved.</p>
                </footer>
            </div>
        </div>
    </div>
    
    <?php apply_watermark(); ?>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Monthly fees chart
        var monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
        var monthlyChart = new Chart(monthlyCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode($monthly_labels); ?>,
                datasets: [{
                    label: 'Fees Collected (<?php echo $_SESSION['currency']['code']; ?>)',
                    data: <?php echo json_encode($monthly_totals); ?>,
                    backgroundColor: 'rgba(54, 162, 235, 0.2)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 2,
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Class-wise chart
        var classCtx = document.getElementById('classChart').getContext('2d');
        var classChart = new Chart(classCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($class_labels); ?>,
                datasets: [{
                    label: 'Fees by Class (<?php echo $_SESSION['currency']['code']; ?>)',
                    data: <?php echo json_encode($class_totals); ?>,
                    backgroundColor: 'rgba(40, 167, 69, 0.7)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Payment status chart
        var statusCtx = document.getElementById('statusChart').getContext('2d');
        var statusChart = new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: ['Fully Paid', 'Partial Payment', 'Not Paid'],
                datasets: [{
                    data: [<?php echo $paid_count; ?>, <?php echo $partial_count; ?>, <?php echo $unpaid_count; ?>],
                    backgroundColor: [
                        'rgba(40, 167, 69, 0.8)',
                        'rgba(255, 193, 7, 0.8)',
                        'rgba(220, 53, 69, 0.8)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Payment method chart
        var methodCtx = document.getElementById('methodChart').getContext('2d');
        var methodChart = new Chart(methodCtx, {
            type: 'pie',
            data: {
                labels: <?php echo json_encode($method_labels); ?>,
                datasets: [{
                    data: <?php echo json_encode($method_totals); ?>,
                    backgroundColor: [
                        'rgba(54, 162, 235, 0.8)',
                        'rgba(255, 99, 132, 0.8)',
                        'rgba(255, 206, 86, 0.8)',
                        'rgba(75, 192, 192, 0.8)'
                    ],
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    </script>
</body>
</html>