<?php
require_once 'includes/auth.php';
requireAdmin();

$users = getAllUsers();
$notes = getAllNotes();

// Handle admin actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add_user') {
        $username = trim($_POST['username']);
        $password = $_POST['password'];
        $email = trim($_POST['email']);
        $full_name = trim($_POST['full_name']);
        $role = $_POST['role'];
        
        // Check if username already exists
        $existingUser = getUserByUsername($username);
        
        if ($existingUser) {
            $error = "Username $username already exists!";
        } else {
            $success = createUser($username, $password, $email, $full_name, $role);
            
            if ($success) {
                $success = "User $username added successfully!";
                // Refresh users list
                $users = getAllUsers();
            } else {
                $error = "Error creating user. Please try again.";
            }
        }
    } elseif ($action === 'delete_user') {
        $user_id = $_POST['user_id'];
        
        // Prevent admin from deleting themselves
        if ($user_id == $_SESSION['user']['id']) {
            $error = "You cannot delete your own account!";
        } else {
            $success = deleteUser($user_id);
            
            if ($success) {
                $success = "User deleted successfully!";
                // Refresh users list
                $users = getAllUsers();
            } else {
                $error = "Error deleting user. Please try again.";
            }
        }
    } elseif ($action === 'delete_note') {
        $note_id = $_POST['note_id'];
        $success = deleteNote($note_id, null, true);
        
        if ($success) {
            $success = "Note deleted successfully!";
            // Refresh notes list
            $notes = getAllNotes();
        } else {
            $error = "Error deleting note. Please try again.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Vine Note Maker</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <header>
        <div class="container">
            <div class="header-content">
                <div class="logo-container">
                    <img src="assets/images/machform-logo.jpg" alt="Prime Gateway Academy Logo" class="logo">
                    <div class="app-name">Vine Note Maker - Admin Panel</div>
                </div>
                
                <button class="menu-toggle">
                    <span></span>
                    <span></span>
                    <span></span>
                </button>
                
                <nav>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="notewriter.php">Note Writer</a></li>
                        <li><a href="logout.php">Logout</a></li>
                    </ul>
                </nav>
            </div>
        </div>
    </header>

    <div class="container" style="padding: 30px 15px;">
        <?php if (isset($error)): ?>
            <div class="alert alert-error"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <?php if (isset($success)): ?>
            <div class="alert alert-success"><?php echo $success; ?></div>
        <?php endif; ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <i class="fas fa-users"></i>
                <div class="stat-number"><?php echo count($users); ?></div>
                <div class="stat-label">Total Users</div>
            </div>
            
            <div class="stat-card">
                <i class="fas fa-sticky-note"></i>
                <div class="stat-number"><?php echo count($notes); ?></div>
                <div class="stat-label">Total Notes</div>
            </div>
            
            <div class="stat-card">
                <i class="fas fa-user-shield"></i>
                <div class="stat-number"><?php echo count(array_filter($users, function($user) { return $user['role'] === 'admin'; })); ?></div>
                <div class="stat-label">Admin Users</div>
            </div>
            
            <div class="stat-card">
                <i class="fas fa-user-graduate"></i>
                <div class="stat-number"><?php echo count(array_filter($users, function($user) { return $user['role'] === 'user'; })); ?></div>
                <div class="stat-label">Regular Users</div>
            </div>
        </div>

        <div class="row">
            <div class="col">
                <div class="card">
                    <h2>Add New User</h2>
                    
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="add_user">
                        
                        <div class="form-group">
                            <label for="full_name">Full Name</label>
                            <input type="text" id="full_name" name="full_name" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="email">Email</label>
                            <input type="email" id="email" name="email" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="username">Username</label>
                            <input type="text" id="username" name="username" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="password">Password</label>
                            <input type="password" id="password" name="password" class="form-control" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="role">Role</label>
                            <select id="role" name="role" class="form-control" required>
                                <option value="user">User</option>
                                <option value="admin">Admin</option>
                            </select>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">Add User</button>
                    </form>
                </div>
            </div>

            <div class="col">
                <div class="card">
                    <h2>User Management</h2>
                    
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Username</th>
                                    <th>Full Name</th>
                                    <th>Email</th>
                                    <th>Role</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($users as $user): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($user['username']); ?></td>
                                        <td><?php echo htmlspecialchars($user['full_name']); ?></td>
                                        <td><?php echo htmlspecialchars($user['email']); ?></td>
                                        <td><?php echo htmlspecialchars($user['role']); ?></td>
                                        <td>
                                            <?php if ($user['id'] != $_SESSION['user']['id']): ?>
                                                <form method="POST" action="" style="display: inline;">
                                                    <input type="hidden" name="action" value="delete_user">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this user?')">Delete</button>
                                                </form>
                                            <?php else: ?>
                                                <em>Current user</em>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <div class="card">
            <h2>All Notes</h2>
            
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Topic</th>
                            <th>Student Age</th>
                            <th>Created By</th>
                            <th>Created At</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($notes as $note): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($note['topic']); ?></td>
                                <td><?php echo htmlspecialchars($note['user_age']); ?></td>
                                <td><?php echo htmlspecialchars($note['username'] . ' (' . $note['full_name'] . ')'); ?></td>
                                <td><?php echo date('M j, Y g:i A', strtotime($note['created_at'])); ?></td>
                                <td>
                                    <form method="POST" action="" style="display: inline;">
                                        <input type="hidden" name="action" value="delete_note">
                                        <input type="hidden" name="note_id" value="<?php echo $note['id']; ?>">
                                        <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this note?')">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <footer>
        <div class="container">
       Copyright © 2025 - PRIME GATEWAY ACADEMY | Powered by <a href="https://vinesubtech.com.ng" target="_blank">VineSub Tech</a>.
        </div>
    </footer>

    <script src="assets/js/hide.js"></script>
</body>
</html>