<?php
session_start();
require_once 'database.php';

// Check if user is logged in
function isLoggedIn() {
    return isset($_SESSION['user']);
}

// Check if user is admin
function isAdmin() {
    return isLoggedIn() && $_SESSION['user']['role'] === 'admin';
}

// Redirect if not logged in
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit();
    }
}

// Redirect if not admin
function requireAdmin() {
    requireLogin();
    
    if (!isAdmin()) {
        header('Location: index.php');
        exit();
    }
}

// Get user by username
function getUserByUsername($username) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM users WHERE username = ?");
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    
    return $user;
}

// Get user by ID
function getUserById($id) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    
    return $user;
}

// Get all users
function getAllUsers() {
    $conn = getDBConnection();
    $result = $conn->query("SELECT id, username, email, full_name, role, created_at FROM users ORDER BY created_at DESC");
    $users = [];
    
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
    
    $conn->close();
    return $users;
}

// Create new user
function createUser($username, $password, $email, $full_name, $role = 'user') {
    $conn = getDBConnection();
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    $stmt = $conn->prepare("INSERT INTO users (username, password, email, full_name, role) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param("sssss", $username, $hashedPassword, $email, $full_name, $role);
    $success = $stmt->execute();
    $stmt->close();
    $conn->close();
    
    return $success;
}

// Delete user
function deleteUser($id) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
    $stmt->bind_param("i", $id);
    $success = $stmt->execute();
    $stmt->close();
    $conn->close();
    
    return $success;
}

// Get user's notes
function getUserNotes($user_id) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM notes WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $notes = [];
    
    while ($row = $result->fetch_assoc()) {
        $notes[] = $row;
    }
    
    $stmt->close();
    $conn->close();
    
    return $notes;
}

// Get all notes (for admin)
function getAllNotes() {
    $conn = getDBConnection();
    $result = $conn->query("SELECT notes.*, users.username, users.full_name 
                           FROM notes 
                           JOIN users ON notes.user_id = users.id 
                           ORDER BY notes.created_at DESC");
    $notes = [];
    
    while ($row = $result->fetch_assoc()) {
        $notes[] = $row;
    }
    
    $conn->close();
    return $notes;
}

// Get note by ID
function getNoteById($id) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT * FROM notes WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $note = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    
    return $note;
}

// Save note
function saveNote($user_id, $topic, $user_age, $content, $note_id = null) {
    $conn = getDBConnection();
    
    if ($note_id) {
        // Update existing note
        $stmt = $conn->prepare("UPDATE notes SET topic = ?, user_age = ?, content = ?, updated_at = NOW() WHERE id = ? AND user_id = ?");
        $stmt->bind_param("sisi", $topic, $user_age, $content, $note_id, $user_id);
    } else {
        // Create new note
        $stmt = $conn->prepare("INSERT INTO notes (user_id, topic, user_age, content) VALUES (?, ?, ?, ?)");
        $stmt->bind_param("isis", $user_id, $topic, $user_age, $content);
    }
    
    $success = $stmt->execute();
    $note_id = $note_id ?: $conn->insert_id;
    $stmt->close();
    $conn->close();
    
    return $success ? $note_id : false;
}

// Delete note
function deleteNote($id, $user_id, $is_admin = false) {
    $conn = getDBConnection();
    
    if ($is_admin) {
        $stmt = $conn->prepare("DELETE FROM notes WHERE id = ?");
        $stmt->bind_param("i", $id);
    } else {
        $stmt = $conn->prepare("DELETE FROM notes WHERE id = ? AND user_id = ?");
        $stmt->bind_param("ii", $id, $user_id);
    }
    
    $success = $stmt->execute();
    $stmt->close();
    $conn->close();
    
    return $success;
}
// Get note with author information (for admin)
function getNoteWithAuthor($id) {
    $conn = getDBConnection();
    $stmt = $conn->prepare("SELECT notes.*, users.username, users.full_name 
                           FROM notes 
                           JOIN users ON notes.user_id = users.id 
                           WHERE notes.id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $note = $result->fetch_assoc();
    $stmt->close();
    $conn->close();
    
    return $note;
}