<?php
require_once 'includes/auth.php';
requireLogin();

// Get user's notes or all notes if admin
if (isAdmin()) {
    $notes = getAllNotes();
} else {
    $notes = getUserNotes($_SESSION['user']['id']);
}

// Check if previewing a specific note
$previewNote = null;
if (isset($_GET['preview'])) {
    $note = getNoteById($_GET['preview']);
    
    // Check if user owns the note or is admin
    if ($note && ($note['user_id'] == $_SESSION['user']['id'] || isAdmin())) {
        $previewNote = $note;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Notes - Vine Note Maker</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .notes-container {
            display: flex;
            gap: 30px;
            margin-top: 20px;
        }
        
        .notes-list-container {
            flex: 0 0 350px;
        }
        
        .note-preview-container {
            flex: 1;
            min-width: 300px;
        }
        
        .note-preview {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            height: 100%;
            overflow-y: auto;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #7f8c8d;
        }
        
        .empty-state i {
            font-size: 3rem;
            margin-bottom: 15px;
            color: #bdc3c7;
        }
        
        .note-author {
            font-size: 0.9rem;
            color: #7f8c8d;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }
        
        @media (max-width: 992px) {
            .notes-container {
                flex-direction: column;
            }
            
            .notes-list-container {
                flex: none;
            }
        }
    </style>
</head>
<body>
    <header>
        <div class="container">
            <div class="header-content">
                <div class="logo-container">
                    <img src="assets/images/machform-logo.jpg" alt="Prime Gateway Academy Logo" class="logo">
                    <div class="app-name">Vine Note Maker</div>
                </div>
                
                <button class="menu-toggle">
                    <span></span>
                    <span></span>
                    <span></span>
                </button>
                
                <nav>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="notewriter.php">Create Note</a></li>
                        <li><a href="logout.php">Logout</a></li>
                        <?php if (isAdmin()): ?>
                            <li><a href="admin.php">Admin Panel</a></li>
                        <?php endif; ?>
                    </ul>
                </nav>
            </div>
        </div>
    </header>

    <div class="container" style="padding: 30px 15px;">
        <div class="card">
            <h1>My Saved Notes</h1>
            <p>Here you can view and manage all your saved lesson notes. Click on any note to preview its content.</p>
        </div>
        
        <div class="notes-container">
            <div class="notes-list-container">
                <div class="card">
                    <h2><?php echo isAdmin() ? 'All Notes' : 'Your Notes'; ?></h2>
                    
                    <?php if (count($notes) > 0): ?>
                        <div class="notes-list">
                            <?php foreach ($notes as $note): 
                                $isActive = $previewNote && $previewNote['id'] == $note['id'];
                            ?>
                                <div class="note-item <?php echo $isActive ? 'active' : ''; ?>">
                                    <div class="note-title"><?php echo htmlspecialchars($note['topic']); ?></div>
                                    <div class="note-meta">
                                        Age: <?php echo htmlspecialchars($note['user_age']); ?> | 
                                        <?php echo date('M j, Y', strtotime($note['created_at'])); ?>
                                        <?php if (isAdmin() && isset($note['username'])): ?>
                                            <br>By: <?php echo htmlspecialchars($note['username']); ?>
                                        <?php endif; ?>
                                    </div>
                                    <div class="note-actions">
                                        <a href="mynotes.php?preview=<?php echo $note['id']; ?>" class="btn btn-primary">Preview</a>
                                        <a href="notewriter.php?edit=<?php echo $note['id']; ?>" class="btn btn-secondary">Edit</a>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-sticky-note"></i>
                            <h3>No notes yet</h3>
                            <p>You haven't created any notes yet. Start by creating your first note!</p>
                            <a href="notewriter.php" class="btn btn-primary">Create Note</a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="note-preview-container">
                <div class="card note-preview">
                    <?php if ($previewNote): ?>
                        <h2><?php echo htmlspecialchars($previewNote['topic']); ?></h2>
                        
                        <?php if (isAdmin()): ?>
                            <?php
                            $author = getUserById($previewNote['user_id']);
                            if ($author):
                            ?>
                                <div class="note-author">
                                    Created by: <?php echo htmlspecialchars($author['full_name'] . ' (' . $author['username'] . ')'); ?><br>
                                    Created on: <?php echo date('F j, Y \a\t g:i A', strtotime($previewNote['created_at'])); ?>
                                </div>
                            <?php endif; ?>
                        <?php endif; ?>
                        
                        <div class="note-content">
                            <?php echo $previewNote['content']; ?>
                        </div>
                        
                        <div style="margin-top: 30px;">
                            <a href="notewriter.php?edit=<?php echo $previewNote['id']; ?>" class="btn btn-primary">Edit Note</a>
                            <a href="#" class="btn btn-success download-note" data-topic="<?php echo htmlspecialchars($previewNote['topic']); ?>" data-content="<?php echo htmlspecialchars($previewNote['content']); ?>">Download</a>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-search"></i>
                            <h3>Select a note to preview</h3>
                            <p>Click on any note from the list to view its content here.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <footer>
        <div class="container">
            Copyright © 2025 - PRIME GATEWAY ACADEMY | Powered by <a href="https://vinesubtech.com.ng" target="_blank">VineSub Tech</a>.
        </div>
    </footer>

    <script src="assets/js/hide.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Download note functionality
            document.querySelectorAll('.download-note').forEach(button => {
                button.addEventListener('click', function(e) {
                    e.preventDefault();
                    const topic = this.getAttribute('data-topic');
                    const content = this.getAttribute('data-content');
                    
                    // Create a simple DOC file using HTML format
                    const htmlContent = `
                        <html xmlns:o="urn:schemas-microsoft-com:office:office" 
                              xmlns:w="urn:schemas-microsoft-com:office:word" 
                              xmlns="http://www.w3.org/TR/REC-html40">
                        <head>
                            <meta charset="utf-8">
                            <title>${topic}</title>
                            <style>
                                body { font-family: Arial, sans-serif; line-height: 1.6; }
                                h1 { color: #2c3e50; }
                                h2 { color: #34495e; }
                            </style>
                        </head>
                        <body>
                            <h1>${topic}</h1>
                            <div>${content}</div>
                        </body>
                        </html>
                    `;
                    
                    const source = 'data:application/vnd.ms-word;charset=utf-8,' + encodeURIComponent(htmlContent);
                    const downloadLink = document.createElement('a');
                    downloadLink.href = source;
                    downloadLink.download = `${topic.replace(/\s+/g, '_')}_note.doc`;
                    document.body.appendChild(downloadLink);
                    downloadLink.click();
                    document.body.removeChild(downloadLink);
                });
            });
            
            // Auto-scroll to active note
            const activeNote = document.querySelector('.note-item.active');
            if (activeNote) {
                activeNote.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
            }
        });
    </script>
</body>
</html>