<?php
require_once 'includes/auth.php';
requireLogin();

// Handle note operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'save') {
        $note_id = $_POST['note_id'] ?? null;
        $topic = $_POST['topic'];
        $user_age = $_POST['user_age'];
        $content = $_POST['content'];
        
        $result = saveNote($_SESSION['user']['id'], $topic, $user_age, $content, $note_id);
        
        if ($result) {
            echo json_encode(['success' => true, 'note_id' => $result]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Error saving note']);
        }
        exit();
    } elseif ($action === 'delete') {
        $note_id = $_POST['note_id'];
        $success = deleteNote($note_id, $_SESSION['user']['id']);
        
        echo json_encode(['success' => $success]);
        exit();
    }
}

// Get user's notes
$userNotes = getUserNotes($_SESSION['user']['id']);

// Check if editing a specific note
$editNote = null;
if (isset($_GET['edit'])) {
    $note = getNoteById($_GET['edit']);
    
    // Check if user owns the note or is admin
    if ($note && ($note['user_id'] == $_SESSION['user']['id'] || isAdmin())) {
        $editNote = $note;
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Note Writer - Vine Note Maker</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <header>
        <div class="container">
            <div class="header-content">
                <div class="logo-container">
                    <img src="assets/images/machform-logo.jpg" alt="Prime Gateway Academy Logo" class="logo">
                    <div class="app-name">Vine Note Maker</div>
                </div>
                
                <button class="menu-toggle">
                    <span></span>
                    <span></span>
                    <span></span>
                </button>
                
                <nav>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="mynotes.php">My Notes</a></li>
                        <li><a href="logout.php">Logout</a></li>
                        <?php if (isAdmin()): ?>
                            <li><a href="admin.php">Admin Panel</a></li>
                        <?php endif; ?>
                    </ul>
                </nav>
            </div>
        </div>
    </header>

    <div class="container" style="padding: 30px 15px;">
        <div class="row">
            <div class="col" style="min-width: 300px;">
                <div class="card">
                    <h2>Your Notes</h2>
                    <?php if (count($userNotes) > 0): ?>
                        <div class="notes-list">
                            <?php foreach ($userNotes as $note): ?>
                                <div class="note-item <?php echo ($editNote && $editNote['id'] == $note['id']) ? 'active' : ''; ?>">
                                    <div class="note-title"><?php echo htmlspecialchars($note['topic']); ?></div>
                                    <div class="note-meta">Age: <?php echo htmlspecialchars($note['user_age']); ?> | <?php echo date('M j, Y', strtotime($note['created_at'])); ?></div>
                                    <div class="note-actions">
                                        <a href="notewriter.php?edit=<?php echo $note['id']; ?>" class="btn btn-primary">Edit</a>
                                        <button class="btn btn-danger delete-note" data-id="<?php echo $note['id']; ?>">Delete</button>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <p>You haven't created any notes yet.</p>
                    <?php endif; ?>
                </div>
            </div>

            <div class="col">
                <div class="card">
                    <h2><?php echo $editNote ? 'Edit Note' : 'Create New Note'; ?></h2>
                    
                    <form id="noteForm">
                        <input type="hidden" id="note_id" name="note_id" value="<?php echo $editNote ? $editNote['id'] : ''; ?>">
                        
                        <div class="form-group">
                            <label for="topic">Topic/Subject</label>
                            <input type="text" id="topic" name="topic" class="form-control" placeholder="e.g., Mathematics, Science, English" required 
                                   value="<?php echo $editNote ? htmlspecialchars($editNote['topic']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="user_age">Student Age</label>
                            <input type="number" id="user_age" name="user_age" class="form-control" min="5" max="18" placeholder="Age between 5 and 18" required
                                   value="<?php echo $editNote ? htmlspecialchars($editNote['user_age']) : ''; ?>">
                        </div>
                        
                        <div class="form-group">
                            <label for="content">Note Content</label>
                            <textarea id="content" name="content" class="form-control" rows="10" placeholder="Enter your note content here..." required><?php echo $editNote ? htmlspecialchars($editNote['content']) : ''; ?></textarea>
                        </div>
                        
                        <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                            <button type="button" id="generateBtn" class="btn btn-primary">Generate Note</button>
                            <button type="button" id="saveBtn" class="btn btn-success">Save Note</button>
                            <button type="button" id="clearBtn" class="btn btn-secondary">Clear</button>
                        </div>
                    </form>
                    
                    <div class="loading" id="loading" style="display: none;">
                        <i class="fas fa-spinner fa-spin"></i> Generating your note...
                    </div>
                    
                    <div id="response" style="display: none; margin-top: 20px;"></div>
                    
                    <div id="exportButtons" style="display: none; margin-top: 20px; gap: 15px; flex-wrap: wrap;">
                        <button id="downloadTxt" class="btn btn-success">Download as TXT</button>
                        <button id="downloadDoc" class="btn btn-success">Download as DOC</button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <footer>
        <div class="container">
Copyright © 2025 - PRIME GATEWAY ACADEMY | Powered by <a href="https://vinesubtech.com.ng" target="_blank">VineSub Tech</a>.
        </div>
    </footer>

    <script src="assets/js/hide.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const noteForm = document.getElementById('noteForm');
            const generateBtn = document.getElementById('generateBtn');
            const saveBtn = document.getElementById('saveBtn');
            const clearBtn = document.getElementById('clearBtn');
            const responseDiv = document.getElementById('response');
            const loadingDiv = document.getElementById('loading');
            const exportButtons = document.getElementById('exportButtons');
            const downloadTxtBtn = document.getElementById('downloadTxt');
            const downloadDocBtn = document.getElementById('downloadDoc');
            
            // Generate note button
            generateBtn.addEventListener('click', function() {
                const topic = document.getElementById('topic').value;
                const userAge = document.getElementById('user_age').value;
                
                if (!topic || !userAge) {
                    showMessage('Please fill in the topic and student age fields.', 'error');
                    return;
                }
                
                loadingDiv.style.display = 'block';
                responseDiv.style.display = 'none';
                exportButtons.style.display = 'none';
                
                const prompt = `Explain "${topic}" for a ${userAge}-year-old student. 
                    Use friendly, simple terms with real-life examples. Start answering the question by generating the following: 
                          
                    Topic: ${topic}<hr>
                    Learning Objective: (list out the learning objectives. Example: At the end of the lesson, students should be able to - {Objectives based on the topic})
                    Instructional Materials: (list out the suitable instructional materials).
                    Previous Knowledge: (State the expected previous knowledge based on the topic).
                    <hr>
                    Step 1: Introduction and definitions.
                    Step 2: break down of the topic based on the input.
                    Step 3: Summary of the topic.
                    Assignment: Ask five relevant questions based on the topic.
                    Include as much key points as possible, tailor entire explanation to suit Nigerian curriculum.
                    
                    Format using HTML tags (h1, h2, p, b, i, ul, li). Avoid markdown.`;
                
                // Call the Gemini API
                fetchExplanation(prompt)
                    .then(explanation => {
                        loadingDiv.style.display = 'none';
                        
                        document.getElementById('content').value = explanation;
                        
                        responseDiv.innerHTML = `
                            <h3>Generated Note Preview:</h3>
                            <div class="note-content">${explanation}</div>
                        `;
                        responseDiv.style.display = 'block';
                        exportButtons.style.display = 'flex';
                        
                        showMessage('Note generated successfully!', 'success');
                    })
                    .catch(error => {
                        loadingDiv.style.display = 'none';
                        showMessage('Error generating note: ' + error.message, 'error');
                        console.error('Error:', error);
                    });
            });
            
            // Save note button
            saveBtn.addEventListener('click', function() {
                const topic = document.getElementById('topic').value;
                const userAge = document.getElementById('user_age').value;
                const content = document.getElementById('content').value;
                const noteId = document.getElementById('note_id').value;
                
                if (!topic || !userAge || !content) {
                    showMessage('Please fill in all fields before saving.', 'error');
                    return;
                }
                
                const formData = new FormData();
                formData.append('action', 'save');
                formData.append('note_id', noteId);
                formData.append('topic', topic);
                formData.append('user_age', userAge);
                formData.append('content', content);
                
                fetch('notewriter.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showMessage('Note saved successfully!', 'success');
                        // Update the note_id field with the returned ID if it's a new note
                        if (!noteId) {
                            document.getElementById('note_id').value = data.note_id;
                        }
                        // Reload the page to update the notes list
                        setTimeout(() => {
                            window.location.reload();
                        }, 1000);
                    } else {
                        showMessage('Error saving note: ' + (data.error || 'Unknown error'), 'error');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    showMessage('Error saving note.', 'error');
                });
            });
            
            // Clear form button
            clearBtn.addEventListener('click', function() {
                if (confirm('Are you sure you want to clear the form?')) {
                    noteForm.reset();
                    responseDiv.style.display = 'none';
                    exportButtons.style.display = 'none';
                    window.location.href = 'notewriter.php';
                }
            });
            
            // Delete note buttons
            document.querySelectorAll('.delete-note').forEach(button => {
                button.addEventListener('click', function() {
                    const noteId = this.getAttribute('data-id');
                    
                    if (confirm('Are you sure you want to delete this note?')) {
                        const formData = new FormData();
                        formData.append('action', 'delete');
                        formData.append('note_id', noteId);
                        
                        fetch('notewriter.php', {
                            method: 'POST',
                            body: formData
                        })
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                showMessage('Note deleted successfully!', 'success');
                                // Reload the page to update the notes list
                                setTimeout(() => {
                                    window.location.reload();
                                }, 1000);
                            } else {
                                showMessage('Error deleting note.', 'error');
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            showMessage('Error deleting note.', 'error');
                        });
                    }
                });
            });
            
            // Download as TXT
            downloadTxtBtn.addEventListener('click', function() {
                const topic = document.getElementById('topic').value;
                const content = document.getElementById('content').value;
                downloadAsTxt(topic, content);
            });
            
            // Download as DOC
            downloadDocBtn.addEventListener('click', function() {
                const topic = document.getElementById('topic').value;
                const content = document.getElementById('content').value;
                downloadAsDoc(topic, content);
            });
            
            // Helper functions
            function showMessage(message, type) {
                // Remove any existing messages
                const existingMessages = document.querySelectorAll('.alert');
                existingMessages.forEach(msg => msg.remove());
                
                const messageDiv = document.createElement('div');
                messageDiv.className = `alert alert-${type === 'error' ? 'error' : 'success'}`;
                messageDiv.textContent = message;
                
                noteForm.parentNode.insertBefore(messageDiv, noteForm);
                
                // Auto remove after 5 seconds
                setTimeout(() => {
                    messageDiv.remove();
                }, 5000);
            }
            
            function downloadAsTxt(topic, content) {
                const element = document.createElement('a');
                const file = new Blob([content], {type: 'text/plain'});
                element.href = URL.createObjectURL(file);
                element.download = `${topic.replace(/\s+/g, '_')}_note.txt`;
                document.body.appendChild(element);
                element.click();
                document.body.removeChild(element);
            }
            
            function downloadAsDoc(topic, content) {
                // Create a simple DOC file using HTML format
                const htmlContent = `
                    <html xmlns:o="urn:schemas-microsoft-com:office:office" 
                          xmlns:w="urn:schemas-microsoft-com:office:word" 
                          xmlns="http://www.w3.org/TR/REC-html40">
                    <head>
                        <meta charset="utf-8">
                        <title>${topic}</title>
                        <style>
                            body { font-family: Arial, sans-serif; line-height: 1.6; }
                            h1 { color: #2c3e50; }
                            h2 { color: #34495e; }
                        </style>
                    </head>
                    <body>
                        <h1>${topic}</h1>
                        <div>${content.replace(/\n/g, '<br>')}</div>
                    </body>
                    </html>
                `;
                
                const source = 'data:application/vnd.ms-word;charset=utf-8,' + encodeURIComponent(htmlContent);
                const downloadLink = document.createElement('a');
                downloadLink.href = source;
                downloadLink.download = `${topic.replace(/\s+/g, '_')}_note.doc`;
                document.body.appendChild(downloadLink);
                downloadLink.click();
                document.body.removeChild(downloadLink);
            }
        });
    </script>
</body>
</html>