<?php
require_once 'config.php';
requireAdmin();

$user_id = $_SESSION['user_id'];

// Get statistics
$total_users = $pdo->query("SELECT COUNT(*) FROM users WHERE role = 'teacher'")->fetchColumn();
$total_notes = $pdo->query("SELECT COUNT(*) FROM notes")->fetchColumn();
$total_transactions = $pdo->query("SELECT COUNT(*) FROM transactions WHERE status = 'completed'")->fetchColumn();
$total_revenue = $pdo->query("SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE status = 'completed'")->fetchColumn();

// Get recent activities
$recent_activities = $pdo->query("
    SELECT l.*, u.username 
    FROM system_logs l 
    LEFT JOIN users u ON l.user_id = u.id 
    ORDER BY l.created_at DESC 
    LIMIT 10
")->fetchAll();

// Get recent transactions
$recent_transactions = $pdo->query("
    SELECT t.*, u.username, u.email, p.name as package_name, p.credits 
    FROM transactions t 
    JOIN users u ON t.user_id = u.id 
    JOIN credit_packages p ON t.package_id = p.id 
    ORDER BY t.created_at DESC 
    LIMIT 10
")->fetchAll();

// Get all users for management
$users = $pdo->query("SELECT * FROM users ORDER BY created_at DESC")->fetchAll();

// Get all credit packages
$credit_packages = $pdo->query("SELECT * FROM credit_packages ORDER BY credits")->fetchAll();

// Get all settings
$settings_result = $pdo->query("SELECT * FROM settings");
$settings = [];
while ($row = $settings_result->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Update settings
    if (isset($_POST['update_settings'])) {
        foreach ($_POST['settings'] as $key => $value) {
            $stmt = $pdo->prepare("UPDATE settings SET setting_value = ? WHERE setting_key = ?");
            $stmt->execute([$value, $key]);
        }
        $_SESSION['success'] = 'Settings updated successfully!';
        header('Location: admin.php');
        exit();
    }
    
    // Add credit package
    if (isset($_POST['add_package'])) {
        $name = trim($_POST['name']);
        $credits = (int)$_POST['credits'];
        $price = (float)$_POST['price'];
        
        if (!empty($name) && $credits > 0 && $price > 0) {
            $stmt = $pdo->prepare("INSERT INTO credit_packages (name, credits, price) VALUES (?, ?, ?)");
            $stmt->execute([$name, $credits, $price]);
            $_SESSION['success'] = 'Credit package added successfully!';
            header('Location: admin.php');
            exit();
        } else {
            $_SESSION['error'] = 'Please fill all fields with valid values.';
        }
    }
    
    // Update user credits
    if (isset($_POST['update_user_credits'])) {
        $user_id_to_update = (int)$_POST['user_id'];
        $credits = (int)$_POST['credits'];
        
        $stmt = $pdo->prepare("UPDATE users SET note_credits = ? WHERE id = ?");
        $stmt->execute([$credits, $user_id_to_update]);
        
        // Log credit transaction
        $stmt = $pdo->prepare("INSERT INTO credit_transactions (user_id, admin_id, credits, type, description) VALUES (?, ?, ?, 'adjustment', 'Admin manual adjustment')");
        $stmt->execute([$user_id_to_update, $user_id, $credits - $user['note_credits']]);
        
        $_SESSION['success'] = 'User credits updated successfully!';
        header('Location: admin.php');
        exit();
    }
    
    // Delete user
    if (isset($_POST['delete_user'])) {
        $user_id_to_delete = (int)$_POST['user_id'];
        
        // Prevent admin from deleting themselves
        if ($user_id_to_delete != $user_id) {
            $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $stmt->execute([$user_id_to_delete]);
            $_SESSION['success'] = 'User deleted successfully!';
        } else {
            $_SESSION['error'] = 'You cannot delete your own account.';
        }
        header('Location: admin.php');
        exit();
    }
    
    // Toggle user status
    if (isset($_POST['toggle_user_status'])) {
        $user_id_to_toggle = (int)$_POST['user_id'];
        $current_status = $_POST['current_status'];
        $new_status = $current_status == 'active' ? 'suspended' : 'active';
        
        $stmt = $pdo->prepare("UPDATE users SET status = ? WHERE id = ?");
        $stmt->execute([$new_status, $user_id_to_toggle]);
        
        $_SESSION['success'] = 'User status updated successfully!';
        header('Location: admin.php');
        exit();
    }
    
    // Delete credit package
    if (isset($_POST['delete_package'])) {
        $package_id = (int)$_POST['package_id'];
        
        $stmt = $pdo->prepare("DELETE FROM credit_packages WHERE id = ?");
        $stmt->execute([$package_id]);
        
        $_SESSION['success'] = 'Credit package deleted successfully!';
        header('Location: admin.php');
        exit();
    }
}

// Display messages
$error = isset($_SESSION['error']) ? $_SESSION['error'] : '';
$success = isset($_SESSION['success']) ? $_SESSION['success'] : '';
unset($_SESSION['error'], $_SESSION['success']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - <?php echo SITE_NAME; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="style.css">
</head>
<body class="admin-dashboard">
    <header>
        <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
            <div class="container">
                <a class="navbar-brand" href="admin.php">
                    <i class="fas fa-crown me-2"></i>
                    <?php echo SITE_NAME; ?> - Admin
                </a>
                <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                    <span class="navbar-toggler-icon"></span>
                </button>
                <div class="collapse navbar-collapse" id="navbarNav">
                    <ul class="navbar-nav me-auto">
                        <li class="nav-item">
                            <a class="nav-link active" href="admin.php">Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">User View</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="support.php">Support</a>
                        </li>
                    </ul>
                    <ul class="navbar-nav ms-auto">
                        <li class="nav-item dropdown">
                            <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                                <i class="fas fa-user-circle me-1"></i> Admin
                            </a>
                            <ul class="dropdown-menu dropdown-menu-end">
                                <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                            </ul>
                        </li>
                    </ul>
                </div>
            </div>
        </nav>
    </header>

    <main class="py-4">
        <div class="container">
            <h2 class="mb-4">Admin Dashboard</h2>

            <!-- Messages -->
            <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="row mb-4">
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-primary text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Users</h5>
                                    <h2 class="card-text"><?php echo $total_users; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-users fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-success text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Notes</h5>
                                    <h2 class="card-text"><?php echo $total_notes; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-sticky-note fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-info text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Transactions</h5>
                                    <h2 class="card-text"><?php echo $total_transactions; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-credit-card fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-warning text-dark">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Revenue</h5>
                                    <h2 class="card-text">₦<?php echo number_format($total_revenue, 0); ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-money-bill-wave fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row">
                <!-- Recent Activities -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">Recent Activities</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>User</th>
                                            <th>Action</th>
                                            <th>Time</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_activities as $activity): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($activity['username'] ?? 'System'); ?></td>
                                            <td><?php echo htmlspecialchars($activity['action']); ?></td>
                                            <td><?php echo date('M j, H:i', strtotime($activity['created_at'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Transactions -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">Recent Transactions</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>User</th>
                                            <th>Package</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_transactions as $transaction): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($transaction['username']); ?></td>
                                            <td><?php echo $transaction['credits']; ?> credits</td>
                                            <td>₦<?php echo number_format($transaction['amount'], 0); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $transaction['status'] === 'completed' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($transaction['status']); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- User Management -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h5 class="card-title mb-0">User Management</h5>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Username</th>
                                            <th>Email</th>
                                            <th>Role</th>
                                            <th>Status</th>
                                            <th>Credits</th>
                                            <th>Joined</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user_item): ?>
                                        <tr>
                                            <td><?php echo $user_item['id']; ?></td>
                                            <td><?php echo htmlspecialchars($user_item['username']); ?></td>
                                            <td><?php echo htmlspecialchars($user_item['email']); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $user_item['role'] === 'admin' ? 'danger' : 'primary'; ?>">
                                                    <?php echo ucfirst($user_item['role']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?php echo $user_item['status'] === 'active' ? 'success' : 'danger'; ?>">
                                                    <?php echo ucfirst($user_item['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo $user_item['note_credits']; ?></td>
                                            <td><?php echo date('M j, Y', strtotime($user_item['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <!-- Edit Credits Form -->
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="user_id" value="<?php echo $user_item['id']; ?>">
                                                        <input type="number" name="credits" value="<?php echo $user_item['note_credits']; ?>" class="form-control form-control-sm d-inline-block" style="width: 70px;">
                                                        <button type="submit" name="update_user_credits" class="btn btn-outline-primary btn-sm">
                                                            <i class="fas fa-save"></i>
                                                        </button>
                                                    </form>
                                                    
                                                    <!-- Toggle Status Form -->
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="user_id" value="<?php echo $user_item['id']; ?>">
                                                        <input type="hidden" name="current_status" value="<?php echo $user_item['status']; ?>">
                                                        <button type="submit" name="toggle_user_status" class="btn btn-<?php echo $user_item['status'] === 'active' ? 'warning' : 'success'; ?> btn-sm">
                                                            <i class="fas fa-<?php echo $user_item['status'] === 'active' ? 'ban' : 'check'; ?>"></i>
                                                        </button>
                                                    </form>
                                                    
                                                    <!-- Delete User Form -->
                                                    <?php if ($user_item['id'] != $user_id): ?>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="user_id" value="<?php echo $user_item['id']; ?>">
                                                        <button type="submit" name="delete_user" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure you want to delete this user?')">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Credit Packages Management -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-success text-white">
                            <h5 class="card-title mb-0">Credit Packages Management</h5>
                        </div>
                        <div class="card-body">
                            <!-- Add New Package Form -->
                            <form method="POST" class="row mb-4">
                                <div class="col-md-4">
                                    <input type="text" name="name" class="form-control" placeholder="Package Name" required>
                                </div>
                                <div class="col-md-3">
                                    <input type="number" name="credits" class="form-control" placeholder="Credits" min="1" required>
                                </div>
                                <div class="col-md-3">
                                    <input type="number" name="price" class="form-control" placeholder="Price (₦)" min="0" step="0.01" required>
                                </div>
                                <div class="col-md-2">
                                    <button type="submit" name="add_package" class="btn btn-primary w-100">Add Package</button>
                                </div>
                            </form>
                            
                            <!-- Packages Table -->
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Name</th>
                                            <th>Credits</th>
                                            <th>Price</th>
                                            <th>Status</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($credit_packages as $package): ?>
                                        <tr>
                                            <td><?php echo $package['id']; ?></td>
                                            <td><?php echo htmlspecialchars($package['name']); ?></td>
                                            <td><?php echo $package['credits']; ?></td>
                                            <td>₦<?php echo number_format($package['price'], 2); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $package['is_active'] ? 'success' : 'secondary'; ?>">
                                                    <?php echo $package['is_active'] ? 'Active' : 'Inactive'; ?>
                                                </span>
                                            </td>
                                            <td>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="package_id" value="<?php echo $package['id']; ?>">
                                                    <button type="submit" name="delete_package" class="btn btn-danger btn-sm" onclick="return confirm('Are you sure you want to delete this package?')">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- System Settings -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">System Settings</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="">
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label class="form-label">Site Name</label>
                                        <input type="text" class="form-control" name="settings[app_name]" value="<?php echo htmlspecialchars($settings['app_name'] ?? SITE_NAME); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Site URL</label>
                                        <input type="url" class="form-control" name="settings[site_url]" value="<?php echo htmlspecialchars($settings['site_url'] ?? SITE_URL); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Default Credits for New Users</label>
                                        <input type="number" class="form-control" name="settings[default_credits]" value="<?php echo $settings['default_credits'] ?? '2'; ?>" min="0">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label class="form-label">Allow New Registrations</label>
                                        <select class="form-select" name="settings[allow_registrations]">
                                            <option value="1" <?php echo ($settings['allow_registrations'] ?? '1') === '1' ? 'selected' : ''; ?>>Yes</option>
                                            <option value="0" <?php echo ($settings['allow_registrations'] ?? '1') === '0' ? 'selected' : ''; ?>>No</option>
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Maintenance Mode</label>
                                        <select class="form-select" name="settings[maintenance_mode]">
                                            <option value="0" <?php echo ($settings['maintenance_mode'] ?? '0') === '0' ? 'selected' : ''; ?>>No</option>
                                            <option value="1" <?php echo ($settings['maintenance_mode'] ?? '0') === '1' ? 'selected' : ''; ?>>Yes</option>
                                        </select>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Currency Symbol</label>
                                        <input type="text" class="form-control" name="settings[currency]" value="<?php echo htmlspecialchars($settings['currency'] ?? '₦'); ?>" maxlength="3">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label class="form-label">Flutterwave Public Key</label>
                                        <input type="text" class="form-control" name="settings[flutterwave_public_key]" value="<?php echo htmlspecialchars($settings['flutterwave_public_key'] ?? FLW_PUBLIC_KEY); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Flutterwave Secret Key</label>
                                        <input type="password" class="form-control" name="settings[flutterwave_secret_key]" value="<?php echo htmlspecialchars($settings['flutterwave_secret_key'] ?? FLW_SECRET_KEY); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Gemini API Key</label>
                                        <input type="password" class="form-control" name="settings[ai_api_key]" value="<?php echo htmlspecialchars($settings['ai_api_key'] ?? GEMINI_API_KEY); ?>">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label class="form-label">SMTP Host</label>
                                        <input type="text" class="form-control" name="settings[smtp_host]" value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">SMTP Port</label>
                                        <input type="number" class="form-control" name="settings[smtp_port]" value="<?php echo htmlspecialchars($settings['smtp_port'] ?? '587'); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">SMTP Encryption</label>
                                        <select class="form-select" name="settings[smtp_encryption]">
                                            <option value="tls" <?php echo ($settings['smtp_encryption'] ?? 'tls') === 'tls' ? 'selected' : ''; ?>>TLS</option>
                                            <option value="ssl" <?php echo ($settings['smtp_encryption'] ?? 'tls') === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">SMTP Username</label>
                                        <input type="text" class="form-control" name="settings[smtp_username]" value="<?php echo htmlspecialchars($settings['smtp_username'] ?? ''); ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">SMTP Password</label>
                                        <input type="password" class="form-control" name="settings[smtp_password]" value="<?php echo htmlspecialchars($settings['smtp_password'] ?? ''); ?>">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">From Email</label>
                                        <input type="email" class="form-control" name="settings[from_email]" value="<?php echo htmlspecialchars($settings['from_email'] ?? ''); ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">From Name</label>
                                        <input type="text" class="form-control" name="settings[from_name]" value="<?php echo htmlspecialchars($settings['from_name'] ?? SITE_NAME); ?>">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-4">
                                        <label class="form-label">Admin Email</label>
                                        <input type="email" class="form-control" name="settings[admin_email]" value="<?php echo htmlspecialchars($settings['admin_email'] ?? ''); ?>">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Max Notes Per Day</label>
                                        <input type="number" class="form-control" name="settings[max_notes_per_day]" value="<?php echo $settings['max_notes_per_day'] ?? '10'; ?>" min="1">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Credit Price</label>
                                        <input type="number" class="form-control" name="settings[credit_price]" value="<?php echo $settings['credit_price'] ?? '100'; ?>" min="0" step="0.01">
                                    </div>
                                </div>
                                
                                <button type="submit" name="update_settings" class="btn btn-primary">Save Settings</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-6">
                    <p class="mb-0">Copyright © 2025 - ORION HUB | Powered by <a href="https://orion.vinesubtech.com.ng" target="_blank" class="text-white">VineSub Tech</a>. All rights reserved.</p>
                </div>
                <div class="col-md-6 text-md-end">
                    <a href="support.php" class="text-white text-decoration-none me-3">Support</a>
                    <a href="#" class="text-white text-decoration-none">Admin</a>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Confirm before deleting
        document.addEventListener('DOMContentLoaded', function() {
            const deleteForms = document.querySelectorAll('form[action*="delete"]');
            deleteForms.forEach(form => {
                form.addEventListener('submit', function(e) {
                    if (!confirm('Are you sure you want to delete this item?')) {
                        e.preventDefault();
                    }
                });
            });
            
            // Toggle SMTP password visibility
            const smtpPasswordInputs = document.querySelectorAll('input[name*="smtp_password"], input[name*="flutterwave_secret_key"], input[name*="ai_api_key"]');
            smtpPasswordInputs.forEach(input => {
                const toggleBtn = document.createElement('button');
                toggleBtn.type = 'button';
                toggleBtn.className = 'btn btn-outline-secondary btn-sm';
                toggleBtn.innerHTML = '<i class="fas fa-eye"></i>';
                toggleBtn.style.position = 'absolute';
                toggleBtn.style.right = '5px';
                toggleBtn.style.top = '50%';
                toggleBtn.style.transform = 'translateY(-50%)';
                toggleBtn.addEventListener('click', function() {
                    const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
                    input.setAttribute('type', type);
                    this.innerHTML = type === 'password' ? '<i class="fas fa-eye"></i>' : '<i class="fas fa-eye-slash"></i>';
                });
                
                input.parentElement.style.position = 'relative';
                input.parentElement.appendChild(toggleBtn);
            });
        });
    </script>
</body>
</html>