<?php
require_once 'config.php';
requireLogin();

$user_id = $_SESSION['user_id'];
$user = getUserData($user_id);

// Get user's notes
$stmt = $pdo->prepare("SELECT * FROM notes WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$notes = $stmt->fetchAll();

// Get credit packages for purchase
$credit_packages = $pdo->query("SELECT * FROM credit_packages WHERE is_active = 1 ORDER BY credits")->fetchAll();

// Generate new note
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['generate_note'])) {
    $topic = trim($_POST['topic']);
    $age_group = trim($_POST['age_group']);
    
    if (empty($topic) || empty($age_group)) {
        $_SESSION['error'] = 'Please provide both topic and age group.';
    } elseif ($user['note_credits'] < 1) {
        $_SESSION['error'] = 'You have insufficient credits. Please purchase more credits.';
    } else {
        try {
            // Generate AI content
            $content = generateAIContent($topic, $age_group);
            
            // Save note to database
            $stmt = $pdo->prepare("INSERT INTO notes (user_id, topic, age_group, content) VALUES (?, ?, ?, ?)");
            $stmt->execute([$user_id, $topic, $age_group, $content]);
            
            // Deduct credit
            $stmt = $pdo->prepare("UPDATE users SET note_credits = note_credits - 1 WHERE id = ?");
            $stmt->execute([$user_id]);
            
            // Update user data
            $user['note_credits']--;
            
            // Log action
            logAction($user_id, 'note_generated', 'Generated note: ' . $topic);
            
            $_SESSION['success'] = 'Note generated successfully!';
            header('Location: dashboard.php');
            exit();
        } catch (Exception $e) {
            $_SESSION['error'] = 'Error generating note: ' . $e->getMessage();
        }
    }
}

// Update note content
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_note'])) {
    $note_id = $_POST['note_id'];
    $content = $_POST['content'];
    
    // Verify note belongs to user
    $stmt = $pdo->prepare("SELECT * FROM notes WHERE id = ? AND user_id = ?");
    $stmt->execute([$note_id, $user_id]);
    
    if ($stmt->rowCount() > 0) {
        $stmt = $pdo->prepare("UPDATE notes SET content = ?, updated_at = NOW() WHERE id = ?");
        $stmt->execute([$content, $note_id]);
        
        // Log action
        logAction($user_id, 'note_updated', 'Updated note ID: ' . $note_id);
        
        $_SESSION['success'] = 'Note updated successfully!';
    } else {
        $_SESSION['error'] = 'Note not found.';
    }
    
    header('Location: dashboard.php');
    exit();
}

// Export note
if (isset($_GET['export'])) {
    $export_type = $_GET['export'];
    $note_id = $_GET['id'];
    
    // Verify note belongs to user
    $stmt = $pdo->prepare("SELECT * FROM notes WHERE id = ? AND user_id = ?");
    $stmt->execute([$note_id, $user_id]);
    $note = $stmt->fetch();
    
    if ($note) {
        if ($export_type === 'pdf') {
            exportNoteAsPDF($note);
        } elseif ($export_type === 'word') {
            exportNoteAsWord($note);
        }
    } else {
        $_SESSION['error'] = 'Note not found.';
        header('Location: dashboard.php');
        exit();
    }
}

// Display messages
$error = isset($_SESSION['error']) ? $_SESSION['error'] : '';
$success = isset($_SESSION['success']) ? $_SESSION['success'] : '';
unset($_SESSION['error'], $_SESSION['success']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo SITE_NAME; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="style.css">
</head>
<body class="dashboard">
    <header>
        <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
            <div class="container">
                <a class="navbar-brand" href="dashboard.php">
                    <i class="fas fa-book-open me-2"></i>
                    <?php echo SITE_NAME; ?>
                </a>
                <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                    <span class="navbar-toggler-icon"></span>
                </button>
                <div class="collapse navbar-collapse" id="navbarNav">
                    <ul class="navbar-nav me-auto">
                        <li class="nav-item">
                            <a class="nav-link active" href="dashboard.php">Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="support.php">Support</a>
                        </li>
                        <?php if (isAdmin()): ?>
                        <li class="nav-item">
                            <a class="nav-link" href="admin.php">Admin</a>
                        </li>
                        <?php endif; ?>
                    </ul>
                    <ul class="navbar-nav ms-auto">
                        <li class="nav-item dropdown">
                            <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                                <i class="fas fa-user-circle me-1"></i> <?php echo htmlspecialchars($user['username']); ?>
                            </a>
                            <ul class="dropdown-menu dropdown-menu-end">
                                <li><span class="dropdown-item-text">Credits: <?php echo $user['note_credits']; ?></span></li>
                                <li><hr class="dropdown-divider"></li>
                                <li><a class="dropdown-item" href="#"><i class="fas fa-user me-2"></i>Profile</a></li>
                                <li><a class="dropdown-item" href="#"><i class="fas fa-cog me-2"></i>Settings</a></li>
                                <li><hr class="dropdown-divider"></li>
                                <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                            </ul>
                        </li>
                    </ul>
                </div>
            </div>
        </nav>
    </header>

    <main class="py-4">
        <div class="container">
            <!-- Welcome Section -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card bg-light border-0">
                        <div class="card-body p-4">
                            <div class="row align-items-center">
                                <div class="col-md-8">
                                    <h2 class="card-title mb-2">Welcome, <?php echo htmlspecialchars($user['full_name']); ?>!</h2>
                                    <p class="card-text text-muted mb-2">You have <span class="fw-bold text-primary"><?php echo $user['note_credits']; ?></span> AI note generation credits remaining.</p>
                                    <p class="card-text">Generate lesson notes, customize them, and download for your classes.</p>
                                </div>
                                <div class="col-md-4 text-md-end">
                                    <a href="#purchaseCredits" class="btn btn-primary me-2">
                                        <i class="fas fa-credit-card me-1"></i> Buy Credits
                                    </a>
                                    <a href="#generateNote" class="btn btn-outline-primary">
                                        <i class="fas fa-plus me-1"></i> New Note
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Messages -->
            <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <div class="row">
                <!-- Generate Note Form -->
                <div class="col-lg-5 mb-4">
                    <div class="card h-100">
                        <div class="card-header bg-primary text-white">
                            <h5 class="card-title mb-0" id="generateNote">
                                <i class="fas fa-magic me-2"></i>Generate New Note
                            </h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="">
                                <div class="mb-3">
                                    <label for="topic" class="form-label">Lesson Topic *</label>
                                    <input type="text" class="form-control" id="topic" name="topic" required placeholder="e.g., Introduction to Photosynthesis">
                                </div>
                                <div class="mb-3">
                                    <label for="age_group" class="form-label">Student Age Group *</label>
                                    <select class="form-select" id="age_group" name="age_group" required>
                                        <option value="">Select Age Group</option>
                                        <option value="5-7">5-7 years</option>
                                        <option value="8-10">8-10 years</option>
                                        <option value="11-13">11-13 years</option>
                                        <option value="14-16">14-16 years</option>
                                        <option value="17-19">17-19 years</option>
                                        <option value="20+">20+ years</option>
                                    </select>
                                </div>
                                <button type="submit" name="generate_note" class="btn btn-primary w-100">
                                    <i class="fas fa-robot me-1"></i> Generate Note (1 Credit)
                                </button>
                            </form>
                        </div>
                    </div>
                </div>

                <!-- Purchase Credits -->
                <div class="col-lg-7 mb-4">
                    <div class="card h-100">
                        <div class="card-header bg-success text-white">
                            <h5 class="card-title mb-0" id="purchaseCredits">
                                <i class="fas fa-credit-card me-2"></i>Purchase Credits
                            </h5>
                        </div>
                        <div class="card-body">
                            <p class="card-text">Buy more credits to generate additional AI lesson notes.</p>
                            <div class="row g-3">
                                <?php foreach ($credit_packages as $package): ?>
                                <div class="col-md-6">
                                    <div class="card credit-package-card">
                                        <div class="card-body text-center">
                                            <h5 class="card-title"><?php echo htmlspecialchars($package['name']); ?></h5>
                                            <h3 class="text-primary fw-bold"><?php echo $package['credits']; ?> Credits</h3>
                                            <p class="card-text">₦<?php echo number_format($package['price'], 0); ?></p>
                                            <form method="POST" action="payment.php">
                                                <input type="hidden" name="package_id" value="<?php echo $package['id']; ?>">
                                                <button type="submit" class="btn btn-outline-primary w-100">Purchase</button>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Notes List -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-sticky-note me-2"></i>Your Lesson Notes
                            </h5>
                        </div>
                        <div class="card-body">
                            <?php if (count($notes) > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Topic</th>
                                            <th>Age Group</th>
                                            <th>Created</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($notes as $note): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($note['topic']); ?></td>
                                            <td><?php echo htmlspecialchars($note['age_group']); ?> years</td>
                                            <td><?php echo date('M j, Y', strtotime($note['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <button type="button" class="btn btn-outline-primary view-note" data-bs-toggle="modal" data-bs-target="#noteModal" data-id="<?php echo $note['id']; ?>" data-topic="<?php echo htmlspecialchars($note['topic']); ?>" data-age="<?php echo htmlspecialchars($note['age_group']); ?>" data-content="<?php echo htmlspecialchars($note['content']); ?>">
                                                        <i class="fas fa-eye"></i>
                                                    </button>
                                                    <a href="dashboard.php?export=pdf&id=<?php echo $note['id']; ?>" class="btn btn-outline-danger">
                                                        <i class="fas fa-file-pdf"></i>
                                                    </a>
                                                    <a href="dashboard.php?export=word&id=<?php echo $note['id']; ?>" class="btn btn-outline-info">
                                                        <i class="fas fa-file-word"></i>
                                                    </a>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php else: ?>
                            <div class="text-center py-4">
                                <i class="fas fa-sticky-note fa-3x text-muted mb-3"></i>
                                <p class="text-muted">You haven't generated any lesson notes yet.</p>
                                <a href="#generateNote" class="btn btn-primary">Generate Your First Note</a>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Note View/Edit Modal -->
    <div class="modal fade" id="noteModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="noteModalTitle">Lesson Note</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <input type="hidden" name="note_id" id="noteId">
                        <div class="mb-3">
                            <label class="form-label">Topic</label>
                            <input type="text" class="form-control" id="noteTopic" readonly>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Age Group</label>
                            <input type="text" class="form-control" id="noteAge" readonly>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Content</label>
                            <textarea class="form-control" id="noteContent" name="content" rows="15" style="white-space: pre-wrap;"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" name="update_note" class="btn btn-primary">Save Changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-6">
                    <p class="mb-0">Copyright © 2025 - ORION HUB | Powered by <a href="https://orion.vinesubtech.com.ng" target="_blank" class="text-white">VineSub Tech</a>. All rights reserved.</p>
                </div>
                <div class="col-md-6 text-md-end">
                    <a href="support.php" class="text-white text-decoration-none me-3">Support</a>
                    <a href="#" class="text-white text-decoration-none me-3">Privacy Policy</a>
                    <a href="#" class="text-white text-decoration-none">Terms of Service</a>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Note modal functionality
        document.addEventListener('DOMContentLoaded', function() {
            const noteModal = document.getElementById('noteModal');
            noteModal.addEventListener('show.bs.modal', function(event) {
                const button = event.relatedTarget;
                const noteId = button.getAttribute('data-id');
                const topic = button.getAttribute('data-topic');
                const age = button.getAttribute('data-age');
                const content = button.getAttribute('data-content');
                
                document.getElementById('noteModalTitle').textContent = topic;
                document.getElementById('noteId').value = noteId;
                document.getElementById('noteTopic').value = topic;
                document.getElementById('noteAge').value = age;
                document.getElementById('noteContent').value = content;
            });
        });
    </script>
</body>
</html>