<?php
require_once 'config.php';

// Check if registrations are allowed
if (getSetting('allow_registrations', '1') === '0') {
    die('New registrations are currently disabled. Please contact the administrator.');
}

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: dashboard.php');
    exit();
}

$error = '';
$success = '';

// Process registration form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $full_name = trim($_POST['full_name']);
    $school = trim($_POST['school']);
    
    // Validate inputs
    if (empty($username) || empty($email) || empty($password) || empty($full_name)) {
        $error = 'Please fill in all required fields.';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters long.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        try {
            // Check if username or email already exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            $stmt->execute([$username, $email]);
            
            if ($stmt->rowCount() > 0) {
                $error = 'Username or email already exists.';
            } else {
                // Get default credits setting
                $default_credits = (int)getSetting('default_credits', '2');
                
                // Hash password
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                // Insert new user
                $stmt = $pdo->prepare("INSERT INTO users (username, password, email, full_name, school, note_credits, role, status) VALUES (?, ?, ?, ?, ?, ?, 'teacher', 'active')");
                $stmt->execute([$username, $hashed_password, $email, $full_name, $school, $default_credits]);
                
                $user_id = $pdo->lastInsertId();
                
                // Log action
                logAction($user_id, 'register', 'New user registration');
                
                // Send welcome email
                $subject = "Welcome to " . SITE_NAME;
                $message = "Hello " . $full_name . ",<br><br>";
                $message .= "Your account has been successfully created.<br><br>";
                $message .= "You can now login using:<br>";
                $message .= "Username: " . $username . "<br><br>";
                $message .= "You have been awarded " . $default_credits . " free credits to get started with AI note generation.<br><br>";
                $message .= "Thank you for joining us!<br><br>";
                $message .= "Best regards,<br>" . SITE_NAME . " Team";
                
                if (sendEmail($email, $subject, $message)) {
                    $success = 'Registration successful! You can now login.';
                } else {
                    $success = 'Registration successful! However, the welcome email could not be sent.';
                }
                
                // Clear form
                $_POST = array();
            }
        } catch (PDOException $e) {
            $error = 'Registration error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - <?php echo SITE_NAME; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="style.css">
</head>
<body class="auth-container">
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8 col-lg-6">
                <div class="card auth-card">
                    <div class="auth-header">
                        <h2 class="text-white mb-0"><i class="fas fa-user-plus me-2"></i>Create Account</h2>
                        <p class="text-white-50 mb-0">Join <?php echo SITE_NAME; ?> today</p>
                    </div>
                    <div class="auth-body">
                        <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <i class="fas fa-exclamation-circle me-2"></i> <?php echo $error; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php endif; ?>
                        
                        <?php if ($success): ?>
                        <div class="alert alert-success alert-dismissible fade show">
                            <i class="fas fa-check-circle me-2"></i> <?php echo $success; ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="" id="registrationForm">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="full_name" class="form-label">Full Name *</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-user"></i></span>
                                        <input type="text" class="form-control" id="full_name" name="full_name" 
                                               value="<?php echo isset($_POST['full_name']) ? htmlspecialchars($_POST['full_name']) : ''; ?>" 
                                               required>
                                    </div>
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="school" class="form-label">School/Institution</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-school"></i></span>
                                        <input type="text" class="form-control" id="school" name="school" 
                                               value="<?php echo isset($_POST['school']) ? htmlspecialchars($_POST['school']) : ''; ?>">
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="username" class="form-label">Username *</label>
                                <div class="input-group">
                                    <span class="input-group-text"><i class="fas fa-at"></i></span>
                                    <input type="text" class="form-control" id="username" name="username" 
                                           value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>" 
                                           required pattern="[a-zA-Z0-9_]+" title="Username can only contain letters, numbers, and underscores">
                                </div>
                                <div class="form-text">Must be unique, no spaces. Only letters, numbers, and underscores.</div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="email" class="form-label">Email Address *</label>
                                <div class="input-group">
                                    <span class="input-group-text"><i class="fas fa-envelope"></i></span>
                                    <input type="email" class="form-control" id="email" name="email" 
                                           value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>" 
                                           required>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="password" class="form-label">Password *</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                        <input type="password" class="form-control" id="password" name="password" required minlength="6">
                                        <button type="button" class="input-group-text password-toggle">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                    <div class="form-text">Minimum 6 characters</div>
                                </div>
                                
                                <div class="col-md-6 mb-4">
                                    <label for="confirm_password" class="form-label">Confirm Password *</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                        <button type="button" class="input-group-text password-toggle">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3 form-check">
                                <input type="checkbox" class="form-check-input" id="terms" name="terms" required>
                                <label class="form-check-label" for="terms">
                                    I agree to the <a href="#" data-bs-toggle="modal" data-bs-target="#termsModal">Terms of Service</a> and <a href="#" data-bs-toggle="modal" data-bs-target="#privacyModal">Privacy Policy</a>
                                </label>
                            </div>
                            
                            <div class="d-grid mb-3">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="fas fa-user-plus me-2"></i>Create Account
                                </button>
                            </div>
                            
                            <div class="alert alert-info">
                                <i class="fas fa-gift me-2"></i>
                                You'll receive <?php echo getSetting('default_credits', '2'); ?> free credits to get started with AI note generation!
                            </div>
                        </form>
                    </div>
                </div>
                
                <div class="text-center mt-4">
                    <p class="text-white">Already have an account? 
                        <a href="login.php" class="text-white fw-bold text-decoration-none">Sign in here</a>
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Terms of Service Modal -->
    <div class="modal fade" id="termsModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Terms of Service</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <h5>1. Acceptance of Terms</h5>
                    <p>By accessing or using Vine AI Note Maker, you agree to be bound by these Terms of Service.</p>
                    
                    <h5>2. Service Description</h5>
                    <p>Vine AI Note Maker provides AI-powered lesson note generation services for educators.</p>
                    
                    <h5>3. User Accounts</h5>
                    <p>You are responsible for maintaining the confidentiality of your account credentials.</p>
                    
                    <h5>4. Payment Terms</h5>
                    <p>All credit purchases are final and non-refundable. Credits do not expire.</p>
                    
                    <h5>5. Intellectual Property</h5>
                    <p>Users retain ownership of their lesson notes. Vine AI Note Maker retains ownership of the AI technology.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Privacy Policy Modal -->
    <div class="modal fade" id="privacyModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Privacy Policy</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <h5>1. Information We Collect</h5>
                    <p>We collect personal information such as name, email, and school information when you register.</p>
                    
                    <h5>2. How We Use Your Information</h5>
                    <p>We use your information to provide and improve our services, process payments, and communicate with you.</p>
                    
                    <h5>3. Data Security</h5>
                    <p>We implement security measures to protect your personal information.</p>
                    
                    <h5>4. Third-Party Services</h5>
                    <p>We use third-party services for payment processing and email delivery.</p>
                    
                    <h5>5. Your Rights</h5>
                    <p>You have the right to access, update, or delete your personal information.</p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Password visibility toggle
        const passwordToggles = document.querySelectorAll('.password-toggle');
        passwordToggles.forEach(function(toggle) {
            toggle.addEventListener('click', function() {
                const input = this.parentElement.querySelector('input');
                const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
                input.setAttribute('type', type);
                this.querySelector('i').classList.toggle('fa-eye');
                this.querySelector('i').classList.toggle('fa-eye-slash');
            });
        });

        // Form validation
        const form = document.getElementById('registrationForm');
        form.addEventListener('submit', function(e) {
            // Check if passwords match
            const password = document.getElementById('password');
            const confirmPassword = document.getElementById('confirm_password');
            
            if (password.value !== confirmPassword.value) {
                e.preventDefault();
                alert('Passwords do not match. Please check and try again.');
                confirmPassword.focus();
                return false;
            }
            
            // Check if terms are accepted
            const terms = document.getElementById('terms');
            if (!terms.checked) {
                e.preventDefault();
                alert('Please accept the Terms of Service and Privacy Policy to continue.');
                return false;
            }
            
            return true;
        });

        // Username validation
        const usernameInput = document.getElementById('username');
        usernameInput.addEventListener('input', function() {
            this.value = this.value.replace(/[^a-zA-Z0-9_]/g, '');
        });

        // Check username availability
        usernameInput.addEventListener('blur', function() {
            const username = this.value.trim();
            if (username.length < 3) return;
            
            // In a real implementation, you would make an AJAX call to check username availability
            console.log('Checking username availability for:', username);
        });
    });
    </script>
</body>
</html>