<?php
require_once 'config.php';

// Check if user is logged in and is admin
if (!isLoggedIn()) {
    header('Location: auth.php?action=login');
    exit();
}

if (!isAdmin()) {
    header('Location: dashboard.php');
    exit();
}

$user_id = $_SESSION['user_id'];

// Initialize variables
$error = '';
$success = '';

// Get statistics
try {
    $total_users = $pdo->query("SELECT COUNT(*) FROM users WHERE role = 'teacher'")->fetchColumn();
    $total_notes = $pdo->query("SELECT COUNT(*) FROM notes")->fetchColumn();
    $total_transactions = $pdo->query("SELECT COUNT(*) FROM transactions WHERE status = 'completed'")->fetchColumn();
    $total_revenue = $pdo->query("SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE status = 'completed'")->fetchColumn();
} catch (PDOException $e) {
    error_log("Database error in admin.php: " . $e->getMessage());
    $total_users = $total_notes = $total_transactions = $total_revenue = 0;
}

// Get recent activities
try {
    $recent_activities = $pdo->query("
        SELECT l.*, u.username 
        FROM system_logs l 
        LEFT JOIN users u ON l.user_id = u.id 
        ORDER BY l.created_at DESC 
        LIMIT 10
    ")->fetchAll();
} catch (PDOException $e) {
    error_log("Database error fetching recent activities: " . $e->getMessage());
    $recent_activities = [];
}

// Get recent transactions
try {
    $recent_transactions = $pdo->query("
        SELECT t.*, u.username, p.name as package_name, p.credits 
        FROM transactions t 
        JOIN users u ON t.user_id = u.id 
        JOIN credit_packages p ON t.package_id = p.id 
        ORDER BY t.created_at DESC 
        LIMIT 10
    ")->fetchAll();
} catch (PDOException $e) {
    error_log("Database error fetching recent transactions: " . $e->getMessage());
    $recent_transactions = [];
}

// Update settings
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    try {
        foreach ($_POST['settings'] as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO settings (setting_key, setting_value) VALUES (?, ?) 
                                  ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute([$key, $value, $value]);
        }
        
        $success = 'Settings updated successfully!';
        logAction($user_id, 'update_settings', 'Updated system settings');
    } catch (PDOException $e) {
        error_log("Error updating settings: " . $e->getMessage());
        $error = 'Error updating settings: ' . $e->getMessage();
    }
}

// Get current settings
try {
    $settings_result = $pdo->query("SELECT * FROM settings")->fetchAll();
    $settings = [];
    foreach ($settings_result as $setting) {
        $settings[$setting['setting_key']] = $setting['setting_value'];
    }
} catch (PDOException $e) {
    error_log("Error fetching settings: " . $e->getMessage());
    $settings = [
        'default_credits' => '2',
        'allow_registrations' => '1'
    ];
}

// Handle user management actions
if (isset($_GET['action']) && isset($_GET['id'])) {
    $action = $_GET['action'];
    $target_id = (int)$_GET['id'];
    
    try {
        if ($action === 'delete_user') {
            // Prevent admin from deleting themselves
            if ($target_id !== $user_id) {
                $stmt = $pdo->prepare("DELETE FROM users WHERE id = ? AND role != 'admin'");
                $stmt->execute([$target_id]);
                
                if ($stmt->rowCount() > 0) {
                    $success = 'User deleted successfully!';
                    logAction($user_id, 'delete_user', 'Deleted user ID: ' . $target_id);
                } else {
                    $error = 'Cannot delete admin users or user not found.';
                }
            } else {
                $error = 'You cannot delete your own account.';
            }
        } elseif ($action === 'toggle_status') {
            $stmt = $pdo->prepare("SELECT status FROM users WHERE id = ?");
            $stmt->execute([$target_id]);
            $current_status = $stmt->fetchColumn();
            
            $new_status = ($current_status === 'active') ? 'suspended' : 'active';
            
            $stmt = $pdo->prepare("UPDATE users SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $target_id]);
            
            $success = 'User status updated successfully!';
            logAction($user_id, 'toggle_user_status', 'Changed user ' . $target_id . ' status to ' . $new_status);
        }
    } catch (PDOException $e) {
        error_log("Error performing user action: " . $e->getMessage());
        $error = 'Error performing action: ' . $e->getMessage();
    }
    
    // Redirect to avoid form resubmission
    header('Location: admin.php');
    exit();
}

// Get all users for management
try {
    $users = $pdo->query("SELECT * FROM users ORDER BY created_at DESC")->fetchAll();
} catch (PDOException $e) {
    error_log("Error fetching users: " . $e->getMessage());
    $users = [];
}

// Handle credit adjustments
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['adjust_credits'])) {
    $target_id = (int)$_POST['user_id'];
    $credits = (int)$_POST['credits'];
    $reason = trim($_POST['reason']);
    
    if ($credits !== 0 && !empty($reason)) {
        try {
            // Update user credits
            $stmt = $pdo->prepare("UPDATE users SET note_credits = note_credits + ? WHERE id = ?");
            $stmt->execute([$credits, $target_id]);
            
            // Log credit transaction
            $type = $credits > 0 ? 'admin_add' : 'admin_remove';
            $stmt = $pdo->prepare("INSERT INTO credit_transactions (user_id, credits, type, description) VALUES (?, ?, ?, ?)");
            $stmt->execute([$target_id, $credits, $type, $reason]);
            
            $success = 'Credits adjusted successfully!';
            logAction($user_id, 'adjust_credits', 'Adjusted ' . $credits . ' credits for user ' . $target_id);
        } catch (PDOException $e) {
            error_log("Error adjusting credits: " . $e->getMessage());
            $error = 'Error adjusting credits: ' . $e->getMessage();
        }
    } else {
        $error = 'Please provide valid credit amount and reason.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - <?php echo SITE_NAME; ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="style.css">
</head>
<body class="admin-dashboard">
    <header>
        <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
            <div class="container">
                <a class="navbar-brand" href="admin.php">
                    <i class="fas fa-book-open me-2"></i>
                    <?php echo SITE_NAME; ?> - Admin
                </a>
                <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                    <span class="navbar-toggler-icon"></span>
                </button>
                <div class="collapse navbar-collapse" id="navbarNav">
                    <ul class="navbar-nav me-auto">
                        <li class="nav-item">
                            <a class="nav-link active" href="admin.php">Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="dashboard.php">User View</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="support.php">Support</a>
                        </li>
                    </ul>
                    <ul class="navbar-nav ms-auto">
                        <li class="nav-item dropdown">
                            <a class="nav-link dropdown-toggle" href="#" role="button" data-bs-toggle="dropdown">
                                <i class="fas fa-user-circle me-1"></i> Admin
                            </a>
                            <ul class="dropdown-menu dropdown-menu-end">
                                <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                            </ul>
                        </li>
                    </ul>
                </div>
            </div>
        </nav>
    </header>

    <main class="py-4">
        <div class="container">
            <h2 class="mb-4">Admin Dashboard</h2>

            <!-- Display Messages -->
            <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
            <?php endif; ?>

            <!-- Statistics Cards -->
            <div class="row mb-4">
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-primary text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Users</h5>
                                    <h2 class="card-text"><?php echo $total_users; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-users fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-success text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Notes</h5>
                                    <h2 class="card-text"><?php echo $total_notes; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-sticky-note fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-info text-white">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Transactions</h5>
                                    <h2 class="card-text"><?php echo $total_transactions; ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-credit-card fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-3 mb-3">
                    <div class="card stat-card bg-warning text-dark">
                        <div class="card-body">
                            <div class="d-flex align-items-center">
                                <div class="flex-grow-1">
                                    <h5 class="card-title">Total Revenue</h5>
                                    <h2 class="card-text">₦<?php echo number_format($total_revenue, 0); ?></h2>
                                </div>
                                <div class="flex-shrink-0">
                                    <i class="fas fa-money-bill-wave fa-2x opacity-50"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row">
                <!-- Recent Activities -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">Recent Activities</h5>
                        </div>
                        <div class="card-body">
                            <?php if (count($recent_activities) > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>User</th>
                                            <th>Action</th>
                                            <th>Time</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_activities as $activity): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($activity['username'] ?? 'System'); ?></td>
                                            <td><?php echo htmlspecialchars($activity['action']); ?></td>
                                            <td><?php echo date('M j, H:i', strtotime($activity['created_at'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php else: ?>
                            <p class="text-muted text-center py-3">No recent activities found.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Recent Transactions -->
                <div class="col-lg-6 mb-4">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">Recent Transactions</h5>
                        </div>
                        <div class="card-body">
                            <?php if (count($recent_transactions) > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>User</th>
                                            <th>Package</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_transactions as $transaction): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($transaction['username']); ?></td>
                                            <td><?php echo $transaction['credits']; ?> credits</td>
                                            <td>₦<?php echo number_format($transaction['amount'], 0); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $transaction['status'] === 'completed' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($transaction['status']); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php else: ?>
                            <p class="text-muted text-center py-3">No recent transactions found.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- User Management -->
            <div class="row mb-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">User Management</h5>
                        </div>
                        <div class="card-body">
                            <?php if (count($users) > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Username</th>
                                            <th>Email</th>
                                            <th>Role</th>
                                            <th>Status</th>
                                            <th>Credits</th>
                                            <th>Joined</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($users as $user_item): ?>
                                        <tr>
                                            <td><?php echo $user_item['id']; ?></td>
                                            <td><?php echo htmlspecialchars($user_item['username']); ?></td>
                                            <td><?php echo htmlspecialchars($user_item['email']); ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $user_item['role'] === 'admin' ? 'danger' : 'primary'; ?>">
                                                    <?php echo ucfirst($user_item['role']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?php echo $user_item['status'] === 'active' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($user_item['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo $user_item['note_credits']; ?></td>
                                            <td><?php echo date('M j, Y', strtotime($user_item['created_at'])); ?></td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <?php if ($user_item['id'] != $user_id && $user_item['role'] != 'admin'): ?>
                                                    <a href="admin.php?action=delete_user&id=<?php echo $user_item['id']; ?>" 
                                                       class="btn btn-outline-danger" 
                                                       onclick="return confirm('Are you sure you want to delete this user?')">
                                                        <i class="fas fa-trash"></i>
                                                    </a>
                                                    <?php endif; ?>
                                                    <a href="admin.php?action=toggle_status&id=<?php echo $user_item['id']; ?>" 
                                                       class="btn btn-outline-<?php echo $user_item['status'] === 'active' ? 'warning' : 'success'; ?>">
                                                        <i class="fas fa-<?php echo $user_item['status'] === 'active' ? 'ban' : 'check'; ?>"></i>
                                                    </a>
                                                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#creditModal" data-user-id="<?php echo $user_item['id']; ?>" data-user-name="<?php echo htmlspecialchars($user_item['username']); ?>">
                                                        <i class="fas fa-coins"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php else: ?>
                            <p class="text-muted text-center py-3">No users found.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- System Settings -->
            <div class="row">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="card-title mb-0">System Settings</h5>
                        </div>
                        <div class="card-body">
                            <form method="POST" action="">
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Default Credits for New Users</label>
                                        <input type="number" class="form-control" name="settings[default_credits]" value="<?php echo $settings['default_credits'] ?? '2'; ?>" min="0">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Allow New Registrations</label>
                                        <select class="form-select" name="settings[allow_registrations]">
                                            <option value="1" <?php echo ($settings['allow_registrations'] ?? '1') === '1' ? 'selected' : ''; ?>>Yes</option>
                                            <option value="0" <?php echo ($settings['allow_registrations'] ?? '1') === '0' ? 'selected' : ''; ?>>No</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Site Name</label>
                                        <input type="text" class="form-control" name="settings[app_name]" value="<?php echo $settings['app_name'] ?? SITE_NAME; ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Site URL</label>
                                        <input type="url" class="form-control" name="settings[site_url]" value="<?php echo $settings['site_url'] ?? SITE_URL; ?>">
                                    </div>
                                </div>
                                <button type="submit" name="update_settings" class="btn btn-primary">Save Settings</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Credit Adjustment Modal -->
    <div class="modal fade" id="creditModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Adjust Credits</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <input type="hidden" name="user_id" id="creditUserId">
                        <div class="mb-3">
                            <label class="form-label">User: <span id="creditUserName"></span></label>
                        </div>
                        <div class="mb-3">
                            <label for="credits" class="form-label">Credit Amount</label>
                            <input type="number" class="form-control" id="credits" name="credits" required>
                            <div class="form-text">Positive number to add credits, negative to remove.</div>
                        </div>
                        <div class="mb-3">
                            <label for="reason" class="form-label">Reason</label>
                            <textarea class="form-control" id="reason" name="reason" rows="3" required></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="adjust_credits" class="btn btn-primary">Save Changes</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-6">
                    <p class="mb-0">Copyright © 2025 - ORION HUB | Powered by <a href="https://orion.vinesubtech.com.ng" target="_blank" class="text-white">VineSub Tech</a>. All rights reserved.</p>
                </div>
                <div class="col-md-6 text-md-end">
                    <a href="support.php" class="text-white text-decoration-none me-3">Support</a>
                    <a href="#" class="text-white text-decoration-none">Admin</a>
                </div>
            </div>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Credit Modal functionality
        document.addEventListener('DOMContentLoaded', function() {
            const creditModal = document.getElementById('creditModal');
            creditModal.addEventListener('show.bs.modal', function(event) {
                const button = event.relatedTarget;
                const userId = button.getAttribute('data-user-id');
                const userName = button.getAttribute('data-user-name');
                
                document.getElementById('creditUserId').value = userId;
                document.getElementById('creditUserName').textContent = userName;
            });
        });
    </script>
</body>
</html>