<?php
require_once 'auth.php';

if (!$auth->isLoggedIn() || !$auth->isAdmin()) {
    header('Location: login.php');
    exit;
}

// Get all users
$users_stmt = $pdo->prepare("SELECT * FROM users ORDER BY created_at DESC");
$users_stmt->execute();
$users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_users,
        COUNT(CASE WHEN vine_access_key IS NOT NULL THEN 1 END) as users_with_key,
        COUNT(CASE WHEN is_admin = 1 THEN 1 END) as admin_users
    FROM users
");
$stats_stmt->execute();
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'assign_key':
                $user_id = $_POST['user_id'];
                $api_key = $_POST['api_key'];
                
                if (empty($api_key)) {
                    $message = "Please enter a valid API key";
                    break;
                }
                
                $update_stmt = $pdo->prepare("UPDATE users SET vine_access_key = ? WHERE id = ?");
                if ($update_stmt->execute([$api_key, $user_id])) {
                    $message = "Vine Access Key assigned successfully!";
                    
                    // Log the assignment
                    error_log("Admin assigned API key to user ID: $user_id");
                } else {
                    $message = "Failed to assign API key";
                }
                break;
                
            case 'remove_key':
                $user_id = $_POST['user_id'];
                
                $update_stmt = $pdo->prepare("UPDATE users SET vine_access_key = NULL WHERE id = ?");
                $update_stmt->execute([$user_id]);
                $message = "Vine Access Key removed successfully!";
                break;
                
            case 'delete_user':
                $user_id = $_POST['user_id'];
                
                $delete_stmt = $pdo->prepare("DELETE FROM users WHERE id = ? AND is_admin = 0");
                $delete_stmt->execute([$user_id]);
                $message = "User deleted successfully!";
                break;
                
            case 'create_agent':
                $name = $_POST['agent_name'];
                $personality = $_POST['agent_personality'];
                $age = $_POST['agent_age'];
                $location = $_POST['agent_location'];
                $gender = $_POST['agent_gender'];
                $avatar = $_POST['agent_avatar'];
                $is_global = isset($_POST['is_global']) ? 1 : 0;
                
                if ($is_global) {
                    // Create global agent for all users
                    $users_stmt = $pdo->prepare("SELECT id FROM users");
                    $users_stmt->execute();
                    $all_users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $agent_stmt = $pdo->prepare("INSERT INTO ai_agents (user_id, name, avatar, age, gender, location, personality, is_default) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
                    
                    foreach ($all_users as $user) {
                        $agent_stmt->execute([$user['id'], $name, $avatar, $age, $gender, $location, $personality, 1]);
                    }
                    $message = "Global agent created for all users!";
                } else {
                    $message = "Agent creation feature for specific users coming soon!";
                }
                break;
        }
        
        // Refresh page to show changes
        header("Location: admin.php?message=" . urlencode($message));
        exit;
    }
}

// Get any message from URL
$success_message = $_GET['message'] ?? '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - Vine-AI Companion</title>
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Roboto', sans-serif;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5a6fd8;
            --secondary: #764ba2;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --dark: #1f2937;
            --light: #f8fafc;
            --gray: #6b7280;
        }

        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: var(--dark);
        }

        .admin-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }

        .admin-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
            text-align: center;
        }

        .admin-header h1 {
            color: var(--primary);
            font-size: 2.5em;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
        }

        .admin-nav {
            display: flex;
            gap: 10px;
            justify-content: center;
            margin-top: 20px;
            flex-wrap: wrap;
        }

        .nav-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 25px;
            cursor: pointer;
            font-weight: 500;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .nav-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255, 255, 255, 0.95);
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.1);
            text-align: center;
        }

        .stat-number {
            font-size: 2.5em;
            font-weight: bold;
            color: var(--primary);
            margin-bottom: 10px;
        }

        .stat-label {
            color: var(--gray);
            font-size: 1.1em;
        }

        .content-section {
            background: rgba(255, 255, 255, 0.95);
            border-radius: 20px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--primary);
        }

        .section-header h2 {
            color: var(--primary);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 500;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }

        .btn:hover {
            background: var(--primary-dark);
            transform: translateY(-1px);
        }

        .btn-success {
            background: var(--success);
        }

        .btn-success:hover {
            background: #0da271;
        }

        .btn-danger {
            background: var(--danger);
        }

        .btn-danger:hover {
            background: #dc2626;
        }

        .btn-warning {
            background: var(--warning);
        }

        .btn-warning:hover {
            background: #e5940b;
        }

        .table-container {
            overflow-x: auto;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }

        th, td {
            padding: 15px;
            text-align: left;
            border-bottom: 1px solid #e5e7eb;
        }

        th {
            background: var(--primary);
            color: white;
            font-weight: 500;
        }

        tr:hover {
            background: #f8fafc;
        }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            object-fit: cover;
        }

        .status-badge {
            padding: 5px 12px;
            border-radius: 20px;
            font-size: 0.85em;
            font-weight: 500;
        }

        .status-active {
            background: #d1fae5;
            color: var(--success);
        }

        .status-inactive {
            background: #fee2e2;
            color: var(--danger);
        }

        .status-pending {
            background: #fef3c7;
            color: var(--warning);
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 500;
            color: var(--dark);
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 16px;
            transition: all 0.3s ease;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .form-group textarea {
            resize: vertical;
            min-height: 100px;
        }

        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .checkbox-group input {
            width: auto;
        }

        .api-key-input-group {
            display: flex;
            gap: 10px;
            align-items: flex-end;
        }

        .api-key-input-group input {
            flex: 1;
        }

        .api-key-input-group .btn {
            white-space: nowrap;
        }

        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            align-items: center;
            justify-content: center;
        }

        .modal.active {
            display: flex;
        }

        .modal-content {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.2);
            max-width: 500px;
            width: 90%;
            max-height: 90vh;
            overflow-y: auto;
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid var(--primary);
        }

        .modal-header h3 {
            color: var(--primary);
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: var(--gray);
        }

        .success-message {
            background: #d1fae5;
            color: var(--success);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid var(--success);
        }

        @media (max-width: 768px) {
            .admin-header h1 {
                font-size: 2em;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }

            .admin-nav {
                flex-direction: column;
            }

            .table-container {
                font-size: 0.9em;
            }

            th, td {
                padding: 10px 5px;
            }

            .api-key-input-group {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Header -->
        <div class="admin-header">
            <h1>
                <span class="material-icons">admin_panel_settings</span>
                Vine-AI Admin Panel
            </h1>
            <p>Manage users, API keys, and AI agents</p>
            
            <div class="admin-nav">
                <button class="nav-btn" onclick="showSection('users')">
                    <span class="material-icons">people</span>
                    Manage Users
                </button>
                <button class="nav-btn" onclick="showSection('agents')">
                    <span class="material-icons">smart_toy</span>
                    Manage Agents
                </button>
                <button class="nav-btn" onclick="showSection('analytics')">
                    <span class="material-icons">analytics</span>
                    Analytics
                </button>
                <a href="index.php" class="nav-btn">
                    <span class="material-icons">arrow_back</span>
                    Back to App
                </a>
            </div>
        </div>

        <?php if ($success_message): ?>
            <div class="success-message">
                <span class="material-icons">check_circle</span>
                <?php echo htmlspecialchars($success_message); ?>
            </div>
        <?php endif; ?>

        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?php echo $stats['total_users']; ?></div>
                <div class="stat-label">Total Users</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $stats['users_with_key']; ?></div>
                <div class="stat-label">Users with API Keys</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $stats['admin_users']; ?></div>
                <div class="stat-label">Administrators</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo count($users); ?></div>
                <div class="stat-label">Active Users</div>
            </div>
        </div>

        <!-- Users Section -->
        <div id="users-section" class="content-section">
            <div class="section-header">
                <h2><span class="material-icons">people</span>User Management</h2>
            </div>

            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Email</th>
                            <th>API Key</th>
                            <th>Status</th>
                            <th>Joined</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td>
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="Avatar" class="user-avatar">
                                        <div>
                                            <strong><?php echo htmlspecialchars($user['name']); ?></strong>
                                            <div style="font-size: 0.9em; color: var(--gray);">@<?php echo htmlspecialchars($user['username']); ?></div>
                                        </div>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($user['email']); ?></td>
                                <td>
                                    <?php if ($user['vine_access_key']): ?>
                                        <span class="status-badge status-active">Key Active</span>
                                    <?php else: ?>
                                        <span class="status-badge status-inactive">No Key</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($user['is_admin']): ?>
                                        <span class="status-badge status-active">Admin</span>
                                    <?php else: ?>
                                        <span class="status-badge status-pending">User</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <div style="display: flex; gap: 5px; flex-wrap: wrap;">
                                        <?php if (!$user['vine_access_key']): ?>
                                            <button class="btn btn-success" onclick="showAssignKeyModal(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['name']); ?>')" style="padding: 8px 12px; font-size: 0.9em;">
                                                <span class="material-icons" style="font-size: 16px;">vpn_key</span>
                                                Assign Key
                                            </button>
                                        <?php else: ?>
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="remove_key">
                                                <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                <button type="submit" class="btn btn-warning" style="padding: 8px 12px; font-size: 0.9em;">
                                                    <span class="material-icons" style="font-size: 16px;">key_off</span>
                                                    Remove Key
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                        
                                        <?php if (!$user['is_admin']): ?>
                                            <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this user? This action cannot be undone.');">
                                                <input type="hidden" name="action" value="delete_user">
                                                <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                <button type="submit" class="btn btn-danger" style="padding: 8px 12px; font-size: 0.9em;">
                                                    <span class="material-icons" style="font-size: 16px;">delete</span>
                                                    Delete
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Agents Section -->
        <div id="agents-section" class="content-section" style="display: none;">
            <div class="section-header">
                <h2><span class="material-icons">smart_toy</span>AI Agents Management</h2>
            </div>

            <div class="form-group">
                <h3>Create New AI Agent</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="create_agent">
                    
                    <div class="form-group">
                        <label for="agent_name">Agent Name</label>
                        <input type="text" id="agent_name" name="agent_name" required placeholder="Enter agent name">
                    </div>
                    
                    <div class="form-group">
                        <label for="agent_personality">Personality Description</label>
                        <textarea id="agent_personality" name="agent_personality" required placeholder="Describe the agent's personality, behavior, and characteristics"></textarea>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                        <div class="form-group">
                            <label for="agent_age">Age</label>
                            <input type="text" id="agent_age" name="agent_age" placeholder="e.g., Ageless, 25, Unknown">
                        </div>
                        
                        <div class="form-group">
                            <label for="agent_gender">Gender</label>
                            <select id="agent_gender" name="agent_gender">
                                <option value="female">Female</option>
                                <option value="male">Male</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label for="agent_location">Location</label>
                        <input type="text" id="agent_location" name="agent_location" placeholder="e.g., NYC, Virtual Space, Internet">
                    </div>
                    
                    <div class="form-group">
                        <label for="agent_avatar">Avatar URL</label>
                        <input type="text" id="agent_avatar" name="agent_avatar" placeholder="https://example.com/avatar.jpg">
                    </div>
                    
                    <div class="checkbox-group">
                        <input type="checkbox" id="is_global" name="is_global" checked>
                        <label for="is_global">Create as global agent for all users</label>
                    </div>
                    
                    <button type="submit" class="btn" style="margin-top: 20px;">
                        <span class="material-icons">save</span>
                        Create Agent
                    </button>
                </form>
            </div>
        </div>

        <!-- Analytics Section -->
        <div id="analytics-section" class="content-section" style="display: none;">
            <div class="section-header">
                <h2><span class="material-icons">analytics</span>System Analytics</h2>
            </div>

            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?php echo $stats['total_users']; ?></div>
                    <div class="stat-label">Registered Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?php echo $stats['users_with_key']; ?></div>
                    <div class="stat-label">Active API Keys</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">0</div>
                    <div class="stat-label">Daily Messages</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">100%</div>
                    <div class="stat-label">System Uptime</div>
                </div>
            </div>

            <div style="background: #f8fafc; padding: 20px; border-radius: 10px; margin-top: 20px;">
                <h3 style="color: var(--primary); margin-bottom: 15px;">Quick Actions</h3>
                <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                    <button class="btn" onclick="exportUserData()">
                        <span class="material-icons">download</span>
                        Export User Data
                    </button>
                    <button class="btn btn-warning" onclick="systemMaintenance()">
                        <span class="material-icons">build</span>
                        System Maintenance
                    </button>
                    <button class="btn btn-danger" onclick="clearAllData()">
                        <span class="material-icons">delete_sweep</span>
                        Clear All Data
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Assign Key Modal -->
    <div id="assignKeyModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><span class="material-icons">vpn_key</span>Assign Vine Access Key</h3>
                <button class="close-modal" onclick="closeAssignKeyModal()">&times;</button>
            </div>
            <form id="assignKeyForm" method="POST">
                <input type="hidden" name="action" value="assign_key">
                <input type="hidden" id="assign_user_id" name="user_id">
                
                <div class="form-group">
                    <label for="assign_user_name">User:</label>
                    <input type="text" id="assign_user_name" readonly style="background: #f8f9fa;">
                </div>
                
                <div class="form-group">
                    <label for="api_key">Vine Access Key:</label>
                    <div class="api-key-input-group">
                        <input type="text" id="api_key" name="api_key" placeholder="AIzaSy..." required style="font-family: monospace;">
                        <button type="submit" class="btn btn-success">
                            <span class="material-icons">save</span>
                            Save Key
                        </button>
                    </div>
                    <small style="color: var(--gray); margin-top: 5px; display: block;">
                        The key should start with "AIzaSy" and be 39 characters long.
                    </small>
                </div>
            </form>
        </div>
    </div>

    <script>
        // Section navigation
        function showSection(sectionName) {
            // Hide all sections
            document.getElementById('users-section').style.display = 'none';
            document.getElementById('agents-section').style.display = 'none';
            document.getElementById('analytics-section').style.display = 'none';
            
            // Show selected section
            document.getElementById(sectionName + '-section').style.display = 'block';
        }

        // Assign Key Modal
        function showAssignKeyModal(userId, userName) {
            document.getElementById('assign_user_id').value = userId;
            document.getElementById('assign_user_name').value = userName;
            document.getElementById('api_key').value = '';
            document.getElementById('assignKeyModal').classList.add('active');
        }

        function closeAssignKeyModal() {
            document.getElementById('assignKeyModal').classList.remove('active');
        }

        // Quick action functions
        function exportUserData() {
            alert('User data export feature would be implemented here.');
        }

        function systemMaintenance() {
            if (confirm('Run system maintenance? This may temporarily affect performance.')) {
                alert('Maintenance started...');
            }
        }

        function clearAllData() {
            if (confirm('WARNING: This will delete ALL user data. This action cannot be undone!')) {
                if (confirm('Are you absolutely sure? This will permanently delete all conversations, settings, and user data.')) {
                    alert('Data clearance initiated...');
                }
            }
        }

        // Show users section by default
        showSection('users');

        // Handle assign key form submission
        document.getElementById('assignKeyForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const apiKey = document.getElementById('api_key').value;
            
            if (!apiKey.startsWith('AIzaSy')) {
                alert('Invalid API key format. Key must start with "AIzaSy"');
                return;
            }
            
            if (apiKey.length !== 39) {
                alert('Invalid API key length. Key must be 39 characters long.');
                return;
            }
            
            this.submit();
        });
    </script>
</body>
</html>